package com.elitescloud.cloudt.system.provider.usersync.jde;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.model.vo.save.user.sync.JdeUserSaveVO;
import com.elitescloud.cloudt.system.provider.usersync.SyncUserResult;
import com.elitescloud.cloudt.system.provider.usersync.UserSyncProvider;
import com.elitescloud.cloudt.system.provider.usersync.jde.params.JdeUserSyncDTO;
import com.elitescloud.cloudt.system.provider.usersync.jde.params.JdeUserSyncResultDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import io.jsonwebtoken.lang.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientResponseException;
import org.springframework.web.client.RestTemplate;

import javax.servlet.http.HttpServletRequest;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;

/**
 * JDE同步账号.
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/30
 */
@Component
public class JdeUserSyncProvider implements UserSyncProvider<JdeUserSaveVO, JdeUserSyncDTO> {
    private static final Logger logger = LoggerFactory.getLogger(JdeUserSyncProvider.class);
    public static final String CODE = "JDE";

    private final RestTemplate restTemplate = RestTemplateFactory.instance();

    @Autowired
    private SystemProperties props;

    @Override
    public String sysCode() {
        return CODE;
    }

    @Override
    public String sysName() {
        return "ERP";
    }

    @Override
    public String getSyncUrl() {
        return props.getJdeConfig().getServerAddr();
    }

    @Override
    public JdeUserSyncDTO convertSyncData(HttpServletRequest request, SysUserBasicDTO userDTO, JdeUserSaveVO reqData) {
        JdeUserSyncDTO dto = new JdeUserSyncDTO();
        dto.setUluser(reqData.getJdeAccount());
        dto.setUllngp(reqData.getJdeLang());
        dto.setUlutctime(reqData.getJdeTimeZone());
        return dto;
    }

    @Override
    public SyncUserResult syncUser(Long userId, JdeUserSyncDTO syncData) {
        try {
            return this.executeSync(userId, syncData);
        } catch (RestClientResponseException e) {
            throw new BusinessException("同步账号失败，服务请求异常：" + e.getRawStatusCode() + "-" + e.getStatusText(), e);
        } catch (Exception e) {
            throw new BusinessException("同步账号失败，" + e.getMessage(), e);
        }
    }

    private SyncUserResult executeSync(Long userId, JdeUserSyncDTO syncData) throws Exception {
        var url = props.getJdeConfig().getServerAddr();
        Assert.hasText(url, sysName() + "服务端地址未配置");
        var clientId = props.getJdeConfig().getClientId();
        Assert.hasText(clientId, "客户端ID未配置");
        var clientSecret = props.getJdeConfig().getClientSecret();
        Assert.hasText(clientSecret, "客户端密码未配置");

        HttpHeaders headers = new HttpHeaders();
        headers.setBasicAuth(clientId, clientSecret, StandardCharsets.UTF_8);

        // 执行调用
        var respResult = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(syncData, headers), String.class);
        if (!respResult.getStatusCode().is2xxSuccessful()) {
            logger.info("请求服务异常：{}, {}", url, respResult);
            throw new BusinessException("请求服务异常：" + url + "，" + respResult.getStatusCode());
        }

        SyncUserResult syncUserResult = new SyncUserResult(userId);
        syncUserResult.setOuterUserCode(syncData.getUluser());
        syncUserResult.setReceipt(respResult.getBody());

        // 解析结果
        JdeUserSyncResultDTO jdeResult = this.convertJdeResult(respResult.getBody());
        syncUserResult.setSuccess(Boolean.TRUE.equals(jdeResult.getSuccess()));
        if (!syncUserResult.isSuccess()) {
            syncUserResult.setFailMsg(jdeResult.getMessage());
        }

        return syncUserResult;
    }

    private JdeUserSyncResultDTO convertJdeResult(String result) {
        JdeUserSyncResultDTO jdeResult = null;
        try {
            jdeResult = JSONUtil.json2Obj(result, JdeUserSyncResultDTO.class, true);
        } catch (Exception e) {
            logger.error("转换JDE结果异常：{}", result, e);

            jdeResult = new JdeUserSyncResultDTO();
            jdeResult.setSuccess(false);
            var mapResult = JSONUtil.json2Obj(result, new TypeReference<HashMap<String, Object>>() {
            });
            if (mapResult != null) {
                jdeResult.setSuccess(Boolean.FALSE.equals(mapResult.get("success")));
                jdeResult.setMessage(ObjUtil.defaultIfNull(mapResult.get("message"), result).toString());
            }
        }
        return jdeResult;
    }
}
