package com.elitescloud.cloudt.system.service.devops.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.core.support.common.ServiceInstanceClient;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.openfeign.common.DynamicClientHelper;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.convert.ServiceInstanceConvert;
import com.elitescloud.cloudt.system.model.vo.resp.devops.ServiceInstanceClassInfoRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.devops.ServiceInstanceInfoRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.devops.ServiceInstanceThreadPoolsRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.devops.ServiceInstanceThreadsRespVO;
import com.elitescloud.cloudt.system.service.devops.ServiceInstanceService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/8
 */
@Component
public class ServiceInstanceServiceImpl implements ServiceInstanceService {
    private static final Logger logger = LoggerFactory.getLogger(ServiceInstanceServiceImpl.class);

    @Override
    public ApiResult<ServiceInstanceInfoRespVO> getServiceInstanceInfo(String serviceId, String url) {
        var client = this.buildServiceInstanceClient(serviceId, url);

        try {
            return client.getServiceInstanceInfo(true).map(ServiceInstanceConvert.INSTANCE::dto2RepVO);
        } catch (Exception e) {
            logger.error("获取属性异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<Object> getProperty(String serviceId, String url, String propertyName) {
        if (CharSequenceUtil.isBlank(propertyName)) {
            return ApiResult.fail("属性名称为空");
        }
        var client = this.buildServiceInstanceClient(serviceId, url);

        try {
            return client.getProperty(propertyName);
        } catch (Exception e) {
            logger.error("获取属性异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<ServiceInstanceClassInfoRespVO> getClass(String serviceId, String url, String className) {
        if (CharSequenceUtil.isBlank(className)) {
            return ApiResult.fail("类路径为空");
        }
        var client = this.buildServiceInstanceClient(serviceId, url);

        try {
            return client.getClass(className).map(ServiceInstanceConvert.INSTANCE::dto2RepVO);
        } catch (Exception e) {
            logger.error("获取客户端版本异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<List<ServiceInstanceThreadsRespVO>> getThreads(String serviceId, String url) {
        var client = this.buildServiceInstanceClient(serviceId, url);

        try {
            return client.getThreadsInfo().map(ServiceInstanceConvert.INSTANCE::dto2RepVO);
        } catch (Exception e) {
            logger.error("获取线程信息异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<List<ServiceInstanceThreadPoolsRespVO>> getThreadPools(String serviceId, String url) {
        var client = this.buildServiceInstanceClient(serviceId, url);

        try {
            return client.getThreadPoolsInfo().map(ServiceInstanceConvert.INSTANCE::tpDto2RepVO);
        } catch (Exception e) {
            logger.error("获取线程池异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    private ServiceInstanceClient buildServiceInstanceClient(@NotBlank String serviceId, String url) {
        Assert.notBlank(serviceId, "服务ID为空");
        try {
            if (CharSequenceUtil.isBlank(url)) {
                return DynamicClientHelper.getClient(serviceId, ServiceInstanceClient.class, ServiceInstanceClient.URI);
            }
            return DynamicClientHelper.buildClient(serviceId, ServiceInstanceClient.class, ServiceInstanceClient.URI, url);
        } catch (Exception e) {
            throw new BusinessException("构建客户端失败", e);
        }
    }
}
