package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.system.service.AreaMngService;
import org.springframework.core.task.TaskExecutor;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 行政区域.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
public class PlatformAreaInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "行政区域";
    }

    @Override
    public String tableName() {
        return "sys_platform_area";
    }

    @Override
    public List<String> fields() {
        return List.of("id", "area_name", "area_code", "parent_area_code", "short_name", "area_type", "zip_code",
                "enabled", "sort_no", "pinyin", "std_version",  "create_time", "modify_time");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("id", "ID");
        titles.put("area_name", "名称");
        titles.put("area_code", "编码");
        titles.put("parent_area_code", "上级编码");
        titles.put("short_name", "简称");
        titles.put("area_type", "区域类型");
        titles.put("zip_code", "邮编");
        titles.put("sort_no", "顺序");
        titles.put("pinyin", "拼音");
        titles.put("enabled", "状态");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");
        return titles;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.asc("area_code"));
    }

    @Override
    public String fieldAppCode() {
        return "";
    }

    @Override
    public Map<String, Object> convertForExport(Map<String, Object> data, List<Map<String, Object>> preData) {
        // 过滤启用的数据
        var enabled = getBooleanValue(data.get("enabled"), null);
        if (enabled == null || !enabled) {
            return null;
        }

        return data;
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var area_code = this.getStringValue(data, "area_code", null);
        Assert.notBlank(area_code, "行政区域编码为空");

        this.normalizeBooleanValue(data, "enabled");

        if (currentData.isEmpty()) {
            return UpdateType.ADD ;
        }

        var existsData = currentData.stream()
                .filter(t -> area_code.equals(t.get("area_code")))
                .findFirst()
                .orElse(null);
        if (existsData == null) {
            this.normalizeForAdd(data);
            return UpdateType.ADD;
        }
        this.normalizeForUpdate(data, existsData);
        return UpdateType.UPDATE;
    }

    @Override
    public void afterExecute(boolean export, Throwable e) {
        if (export || e != null) {
            return;
        }

        // 重构行政组织树
        SpringContextHolder.getBean(TaskExecutor.class).execute(() -> {
            SpringContextHolder.getBean(AreaMngService.class).rebuildTree(null);
        });
    }

    private void normalizeForAdd(Map<String, Object> data) {
        data.put("enabled", this.getBooleanValue(data.get("enabled"), true));
    }

    private void normalizeForUpdate(Map<String, Object> data, Map<String, Object> dataExists) {
        data.put("id", dataExists.get("id"));
        data.put("enabled", ObjectUtil.defaultIfNull(dataExists.get("enabled"), this.getBooleanValue(data.get("enabled"), true)));
    }
}
