package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.pinyin.PinyinUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.common.param.TreeRespParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.AreaType;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.core.provider.IdGenerator;
import com.elitescloud.cloudt.platform.convert.SysPlatformAreaConvert;
import com.elitescloud.cloudt.platform.provider.area.GrabAreaCodeTool;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAreaDO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.AreaDetailRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.AreaMngTreeRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.save.PlatformAreaSaveVO;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAreaRepo;
import com.elitescloud.cloudt.system.convert.AreaConvert;
import com.elitescloud.cloudt.system.provider.imports.param.ImportAreaBO;
import com.elitescloud.cloudt.system.service.AreaMngService;
import com.elitescloud.cloudt.system.service.manager.AreaManager;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantAreaDO;
import com.elitescloud.cloudt.system.service.repo.AreaRepoProc;
import com.elitescloud.cloudt.system.service.repo.TenantAreaRepoProc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/30
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class AreaMngServiceImpl extends BaseServiceImpl implements AreaMngService {
    private static final Logger logger = LoggerFactory.getLogger(AreaMngServiceImpl.class);
    private static final SysPlatformAreaConvert CONVERT = SysPlatformAreaConvert.INSTANCE;
    @Autowired
    private TenantAreaRepoProc tenantAreaRepoProc;
    @Autowired
    private SysPlatformAreaRepo areaRepo;
    @Autowired
    private AreaRepoProc areaRepoProc;

    @Autowired
    private AreaManager areaManager;

    @Override
    public ApiResult<Boolean> getEnabledPlatform() {
        var enabled = this.enabledPlatform();
        return ApiResult.ok(enabled);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateEnabledPlatform() {
        areaManager.clearCache();

        // 获取当前状态
        var tenantId = super.currentTenantId();
        var enabled = tenantAreaRepoProc.getEnabled(tenantId);
        if (enabled == null) {
            // 尚未设置，则新增
            SysTenantAreaDO tenantAreaDO = new SysTenantAreaDO();
            tenantAreaDO.setSysTenantId(tenantId);
            tenantAreaDO.setEnabled(tenantId != TenantConstant.DEFAULT_TENANT_ID.longValue());
            tenantAreaRepoProc.save(tenantAreaDO);

            return ApiResult.ok(tenantAreaDO.getEnabled());
        }

        // 修改
        enabled = !enabled;
        tenantAreaRepoProc.updateEnabled(tenantId, enabled);
        return ApiResult.ok(enabled);
    }

    @Override
    public ApiResult<List<AreaMngTreeRespVO>> tree(Boolean tree) {
        var enabled = this.enabledPlatform();
        List<AreaMngTreeRespVO> areaList = null;
        if (enabled || TenantConstant.DEFAULT_TENANT_ID.longValue() == super.currentTenantId()) {
            areaList = tenantDataIsolateProvider.byDefaultDirectly(() -> areaRepoProc.getTree(null, null, null, this::do2TreeRespVO));
        } else {
            areaList = areaRepoProc.getTree(null, null, null, this::do2TreeRespVO);
        }

        // 是否转树
        if (!ObjectUtil.defaultIfNull(tree, true) || areaList.isEmpty()) {
            return ApiResult.ok(areaList);
        }
        TreeDataUtil<AreaMngTreeRespVO> treeDataUtil = new TreeDataUtil<AreaMngTreeRespVO>(areaList, AreaMngTreeRespVO::getId,
                AreaMngTreeRespVO::getParentId, AreaMngTreeRespVO::setChildren, Comparator.comparingInt(AreaMngTreeRespVO::getSortNo));
        var result = (List<AreaMngTreeRespVO>) treeDataUtil.getRoots();
        return ApiResult.ok(result);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(PlatformAreaSaveVO saveVO) {
        // 校验是否有操作权限
        this.validatePermission();

        SysPlatformAreaDO areaDO = null;
        try {
            areaDO = saveVO.getId() == null ? this.checkForInsert(saveVO) : this.checkForUpdate(saveVO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        areaManager.clearCache();

        // 默认值处理
        Long parentId = StringUtils.hasText(areaDO.getParentAreaCode()) ? areaRepoProc.getId(areaDO.getParentAreaCode()) : null;
        if (areaDO.getEnabled() == null) {
            areaDO.setEnabled(true);
        }
        if (areaDO.getSortNo() == null) {
            areaDO.setSortNo(areaRepoProc.findMaxSortNo(parentId) + 1);
        }

        String oldCodePath = saveVO.getId() == null ? null : areaRepoProc.getCodePath(saveVO.getId());
        areaRepo.save(areaDO);

        // 保存树节点信息
        areaRepoProc.saveTreeNode(areaDO, parentId, areaDO.getSortNo());
        // 如果codePath有调整，则修改下级
        if (oldCodePath != null && !StrUtil.equals(oldCodePath, areaDO.getCodePath())) {
            var idAndCodePathMap = areaRepoProc.getIdAndCodePathByCodePath(oldCodePath);
            var len = oldCodePath.length();
            for (Map.Entry<Long, String> entry : idAndCodePathMap.entrySet()) {
                String newPath = areaDO.getCodePath() + entry.getValue().substring(len);
                areaRepoProc.updateCodePath(entry.getKey(), newPath);
            }
        }

        return ApiResult.ok(areaDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(ImportAreaBO areaBO) {
        // 校验是否有操作权限
        this.validatePermission();

        SysPlatformAreaDO areaDO = null;
        try {
            areaDO = this.checkForInsert(areaBO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        areaManager.clearCache();

        // 默认值处理
        Long parentId = StringUtils.hasText(areaDO.getParentAreaCode()) ? areaRepoProc.getId(areaDO.getParentAreaCode()) : null;
        if (areaDO.getEnabled() == null) {
            areaDO.setEnabled(true);
        }
        if (areaDO.getSortNo() == null) {
            areaDO.setSortNo(areaRepoProc.findMaxSortNo(parentId) + 1);
        }

        areaRepo.save(areaDO);

        // 保存树节点信息
        areaRepoProc.saveTreeNode(areaDO, parentId, areaDO.getSortNo());

        return ApiResult.ok(areaDO.getId());
    }

    @Override
    public ApiResult<AreaDetailRespVO> get(Long id) {
        var enabled = this.enabledPlatform();

        var respVO = enabled || TenantConstant.DEFAULT_TENANT_ID.longValue() == super.currentTenantId() ?
                tenantDataIsolateProvider.byDefaultDirectly(() -> this.getDetailRespVO(id)) : this.getDetailRespVO(id);
        return ApiResult.ok(respVO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        // 校验是否有操作权限
        this.validatePermission();

        // 获取节点信息
        var data = areaRepoProc.get(id);
        if (data == null) {
            return ApiResult.ok(id);
        }

        areaManager.clearCache();

        // 移除树节点
        areaRepoProc.removeTreeNode(data);
        // 删除数据
        areaRepoProc.delete(id);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> rebuildTree(Long rootId) {
        rebuildTree(rootId, null);
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<String> grabUrl() {
        return ApiResult.ok(SpringContextHolder.getBean(SystemProperties.class).getArea().getGrabAreaUrl());
    }

    @Override
    public ApiResult<Object> grabPreview(String url, Boolean forceLevel, Boolean json) {
        if (CharSequenceUtil.isBlank(url)) {
            url = SpringContextHolder.getBean(SystemProperties.class).getArea().getGrabAreaUrl();
        }
        if (CharSequenceUtil.isBlank(url)) {
            return ApiResult.fail("采集地址为空");
        }

        GrabAreaCodeTool tool = new GrabAreaCodeTool(url);
        List<GrabAreaCodeTool.AreaNode> roots = tool.buildTree(forceLevel == null || forceLevel);

        if (json == null || !json) {
            StringBuilder preview = new StringBuilder();
            for (GrabAreaCodeTool.AreaNode areaNode : roots) {
                areaNode.preview("", preview);
            }
            return ApiResult.ok(preview.toString());
        }

        return ApiResult.ok(convertTreeForGrab(roots, null));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> grab(String url, Boolean forceLevel) {
        validatePermission();
        var props = SpringContextHolder.getBean(SystemProperties.class).getArea();
        if (CharSequenceUtil.isBlank(url)) {
            url = props.getGrabAreaUrl();
        }
        if (CharSequenceUtil.isBlank(url)) {
            return ApiResult.fail("采集地址为空");
        }

        // 抓取数据
        GrabAreaCodeTool test = new GrabAreaCodeTool(url);
        List<GrabAreaCodeTool.AreaNode> roots = test.buildTree(forceLevel == null || forceLevel);
        if (roots.isEmpty()) {
            return ApiResult.fail("采集的数据为空");
        }

        // 转换为DO
        Map<String, SysPlatformAreaDO> oldDataMap = areaRepoProc.all().stream().collect(Collectors.toMap(SysPlatformAreaDO::getAreaCode, t -> t, (t1, t2) -> t1));
        SysPlatformAreaDO china = oldDataMap.get(props.getChinaCode());
        if (china == null) {
            china = new SysPlatformAreaDO();
            oldDataMap.put(props.getChinaCode(), china);
            china.setId(IdGenerator.generateLong());
            china.setAreaName("中国");
            china.setAreaCode(props.getChinaCode());
            china.setAreaType(AreaType.COUNTRY.getValue());
            china.setEnabled(true);
            china.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR);
            china.setPinyin(convertPinyin(china.getAreaName()));
            china.setRootId(china.getId());
            china.setSortNo(0);
        }
        List<SysPlatformAreaDO> areaList = convertGrabAreaNodes(china, china, roots, oldDataMap);
        areaList.add(china);

        // 先删除老的
        areaRepoProc.deleteChildren(china.getAreaCode(), china.getId());
        // 保存新的
        areaRepoProc.save(areaList);

        // 重构行政组织树
        Map<String, Long> idCodeMap = areaList.stream().collect(Collectors.toMap(SysPlatformAreaDO::getAreaCode, SysPlatformAreaDO::getId, (t1, t2) -> t1));
        this.rebuildTree(china.getId(), idCodeMap);

        return ApiResult.ok(true);
    }

    private List<Area> convertTreeForGrab(List<GrabAreaCodeTool.AreaNode> nodes, String pCode) {
        if (CollUtil.isEmpty(nodes)) {
            return Collections.emptyList();
        }

        return nodes.stream().map(t -> {
            Area param = new Area();
            param.setCode(t.getCode());
            param.setName(t.getName());
            param.setParentCode(pCode);
            param.setChildren(convertTreeForGrab(t.getChildren(), t.getCode()));

            return param;
        }).collect(Collectors.toList());
    }

    private void rebuildTree(Long rootId, Map<String, Long> idCodeMap) {
        if (idCodeMap == null) {
            idCodeMap = areaRepoProc.getIdAndCode();
        }
        if (idCodeMap.isEmpty()) {
            logger.info("行政区域信息为空，忽略构建");
            return;
        }

        Map<String, Long> finalIdCodeMap = idCodeMap;
        CompletableFuture.runAsync(() -> {
            areaRepoProc.rebuildTree(rootId, t -> StringUtils.hasText(t.getParentAreaCode()) ? finalIdCodeMap.get(t.getParentAreaCode()) : null);
            // 清理缓存
            areaManager.clearCache();
        }, taskExecutor).whenComplete((r, e) -> {
            if (e == null) {
                logger.info("重构行政区域树成功");
                return;
            }
            logger.error("重构行政区域树失败", e);
        });
    }

    private List<SysPlatformAreaDO> convertGrabAreaNodes(SysPlatformAreaDO china, SysPlatformAreaDO parent, List<GrabAreaCodeTool.AreaNode> nodes,
                                                          Map<String, SysPlatformAreaDO> oldDataMap) {
        if (CollUtil.isEmpty(nodes)) {
            return Collections.emptyList();
        }

        List<SysPlatformAreaDO> areaList = new ArrayList<>(256);
        int i = 0;
        for (GrabAreaCodeTool.AreaNode node : nodes) {
            SysPlatformAreaDO areaDO = new SysPlatformAreaDO();
            SysPlatformAreaDO oldArea = oldDataMap.get(node.getCode());
            areaDO.setAreaName(node.getName());
            areaDO.setAreaCode(node.getCode());
            if (parent != null) {
                areaDO.setParentAreaCode(parent.getAreaCode());
            }
            areaDO.setShortName(oldArea != null ? oldArea.getShortName() : null);
            areaDO.setAreaType(convertLevel2AreaType(node.getLevel()));
            areaDO.setZipCode(oldArea != null ? oldArea.getZipCode() : null);
            areaDO.setEnabled(oldArea != null ? oldArea.getEnabled() : true);
            areaDO.setCodePath(parent != null ? parent.getCodePath() + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode() : SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
            areaDO.setPinyin(convertPinyin(areaDO.getAreaName()));
            areaDO.setRootId(china.getId());
            areaDO.setSortNo(i++);
            areaDO.setRemark(oldArea != null ? oldArea.getRemark() : null);

            areaList.add(areaDO);

            // 下级节点
            areaList.addAll(convertGrabAreaNodes(china, areaDO, node.getChildren(), oldDataMap));
        }
        return areaList;
    }

    private String convertLevel2AreaType(int level) {
        switch (level) {
            case 0:
                return AreaType.COUNTRY.getValue();
            case 1:
                return AreaType.PROVINCE.getValue();
            case 2:
                return AreaType.CITY.getValue();
            case 3:
                return AreaType.AREA.getValue();
            case 4:
                return AreaType.TOWN.getValue();
            case 5:
                return AreaType.VILLAGE.getValue();
            default:
                return null;
        }
    }

    private boolean enabledPlatform() {
        var tenantId = super.currentTenantId();
        var enabled = tenantAreaRepoProc.getEnabled(tenantId);
        return ObjectUtil.defaultIfNull(enabled, true);
    }

    private void validatePermission() {
        var tenantId = super.currentTenantId();
        if (TenantConstant.DEFAULT_TENANT_ID.longValue() != tenantId) {
            // 非默认租户，则使用平台树时无权修改
            var enabled = this.enabledPlatform();
            if (enabled) {
                throw new BusinessException("无权限修改平台默认配置");
            }
        }
    }

    private AreaMngTreeRespVO do2TreeRespVO(SysPlatformAreaDO t) {
        var respVO = new AreaMngTreeRespVO();
        respVO.setEnabled(t.getEnabled());
        respVO.setPinyin(t.getPinyin());
        respVO.setId(t.getId());
        respVO.setCode(t.getAreaCode());
        respVO.setName(t.getAreaName());
        respVO.setSortNo(t.getSortNo());
        respVO.setParentId(t.getPId());
        respVO.setParentCode(t.getParentAreaCode());

        return respVO;
    }

    private SysPlatformAreaDO checkForInsert(ImportAreaBO saveBO) {
        var areaDO = AreaConvert.INSTANCE.bo2Do(saveBO);

        // 判断编码是否已存在
        var exists = areaRepoProc.existsCode(areaDO.getAreaCode(), null);
        Assert.isTrue(!exists, "地区编码已存在");

        // 上级地区
        if (StringUtils.hasText(areaDO.getParentAreaCode())) {
            exists = areaRepoProc.existsCode(areaDO.getParentAreaCode(), null);
            Assert.isTrue(exists, "上级地区编码不存在");

            String parentCodePath = areaRepoProc.getCodePath(areaDO.getParentAreaCode());
            areaDO.setCodePath(parentCodePath + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        } else {
            areaDO.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        }

        // 拼音
        areaDO.setPinyin(convertPinyin(saveBO.getName()));

        return areaDO;
    }

    private String convertPinyin(String name) {
        if (CharSequenceUtil.isBlank(name)) {
            return null;
        }
        return PinyinUtil.getPinyin(name);
    }

    private SysPlatformAreaDO checkForInsert(PlatformAreaSaveVO saveVO) {
        var areaDO = CONVERT.saveVo2Do(saveVO);

        // 判断编码是否已存在
        var exists = areaRepoProc.existsCode(areaDO.getAreaCode(), null);
        Assert.isTrue(!exists, "地区编码已存在");

        // 上级地区
        if (StringUtils.hasText(areaDO.getParentAreaCode())) {
            exists = areaRepoProc.existsCode(areaDO.getParentAreaCode(), null);
            Assert.isTrue(exists, "上级地区编码不存在");

            String parentCodePath = areaRepoProc.getCodePath(areaDO.getParentAreaCode());
            areaDO.setCodePath(parentCodePath + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        } else {
            areaDO.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        }

        // 拼音
        areaDO.setPinyin(convertPinyin(saveVO.getAreaName()));

        return areaDO;
    }

    private SysPlatformAreaDO checkForUpdate(PlatformAreaSaveVO saveVO) {
        var areaDO = areaRepoProc.get(saveVO.getId());
        Assert.notNull(areaDO, "修改的数据不存在");

        // 判断编码是否已存在
        if (!StrUtil.equals(saveVO.getAreaCode(), areaDO.getAreaCode())) {
            var exists = areaRepoProc.existsCode(saveVO.getAreaCode(), null);
            Assert.isTrue(!exists, "地区编码已存在");
        }

        if (StrUtil.isBlank(saveVO.getParentAreaCode())) {
            // 上级为空
            areaDO.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        } else if (!StrUtil.equals(saveVO.getParentAreaCode(), areaDO.getParentAreaCode())) {
            // 上级有调整
            var exists = areaRepoProc.existsCode(saveVO.getParentAreaCode(), null);
            Assert.isTrue(exists, "上级地区编码不存在");

            String parentCodePath = areaRepoProc.getCodePath(areaDO.getParentAreaCode());
            areaDO.setCodePath(parentCodePath + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        }

        // 拼音
        if (!StrUtil.equals(saveVO.getAreaName(), areaDO.getAreaName()) || StrUtil.isBlank(areaDO.getAreaName())) {
            areaDO.setPinyin(PinyinUtil.getPinyin(saveVO.getAreaName()));
        }

        CONVERT.saveVo2Do(saveVO, areaDO);
        return areaDO;
    }

    private AreaDetailRespVO getDetailRespVO(long id) {
        return areaRepoProc.getOptional(id)
                .map(t -> {
                    var respVO = CONVERT.do2DetailRespVO(t);
                    // 父节点
                    if (StringUtils.hasText(t.getParentAreaCode())) {
                        respVO.setParentAreaName(areaRepoProc.getAreaName(t.getParentAreaCode()));
                    }
                    // 节点类型
                    if (StringUtils.hasText(t.getAreaType())) {
                        respVO.setAreaTypeName(super.udcValue(new AreaType(t.getAreaType())));
                    }
                    return respVO;
                }).orElse(null);
    }

    public static class Area extends TreeRespParam<Area> {}
}
