package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.datasecurity.config.DataSecurityProperties;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleConditionEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleRelationEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantOrgProvider;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.elitescloud.cloudt.system.constant.DataPermissionType;
import com.elitescloud.cloudt.system.constant.UserType;
import com.elitescloud.cloudt.system.convert.PermissionConverter;
import com.elitescloud.cloudt.system.dto.*;
import com.elitescloud.cloudt.system.model.bo.BusinessObjectBO;
import com.elitescloud.cloudt.system.model.bo.BusinessOperationParamBO;
import com.elitescloud.cloudt.system.model.bo.BusinessParamBO;
import com.elitescloud.cloudt.system.model.bo.PermissionParameterBO;
import com.elitescloud.cloudt.system.model.vo.resp.api.SysMenuApiRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.dpr.*;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeeUserInfoRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.permission.DataPermissionTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.*;
import com.elitescloud.cloudt.system.modules.dpr.SysDprValueType;
import com.elitescloud.cloudt.system.modules.dpr.strategy.DprValueResolverFactory;
import com.elitescloud.cloudt.system.modules.dpr.strategy.impl.DprSysInternallyDynamic;
import com.elitescloud.cloudt.system.provider.dto.SysApiPermissionMetadataDTO;
import com.elitescloud.cloudt.system.service.PermissionQueryService;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectParamInEnum;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.manager.BusinessObjectManager;
import com.elitescloud.cloudt.system.service.manager.PermissionMngManager;
import com.elitescloud.cloudt.system.service.manager.PermissionQueryManager;
import com.elitescloud.cloudt.system.service.manager.RoleMngManager;
import com.elitescloud.cloudt.system.service.model.entity.SysDprRoleApiRowRuleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAppDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleDataPermissionDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleFieldPermissionDO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.elitescloud.cloudt.system.vo.DprUdcVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Functions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/11/7
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class PermissionQueryServiceImpl extends BaseServiceImpl implements PermissionQueryService {

    @Autowired
    private ApiRepoProc apiRepoProc;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private TenantMenuTreeRepoProc tenantMenuTreeRepoProc;
    @Autowired
    private RolePermissionRepoProc rolePermissionRepoProc;
    @Autowired
    private RoleDataPermissionRepoProc dataPermissionRepoProc;
    @Autowired
    private BusinessOperationRepoProc businessOperationRepoProc;
    @Autowired
    private BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    private BusinessParamRepoProc businessParamRepoProc;
    @Autowired
    private BusinessOperationParamRepoProc businessOperationParamRepoProc;
    @Autowired
    private BusinessObjectManager businessObjectManager;
    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private MenuRepoProc menusRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;
    @Autowired
    private MenuApiRepoProc menuApiRepoProc;
    @Autowired
    private SysDprRoleApiRowRuleRepoProc roleApiRowRuleRepoProc;
    @Autowired
    private SysDpcrApiFieldsRepoProc apiFieldsRepoProc;
    @Autowired
    private ApiParameterRepoProc apiParameterRepoProc;
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private TenantOrgRepoProc tenantOrgRepoProc;
    @Autowired
    private TenantOrgProvider tenantOrgProvider;

    @Autowired
    private PermissionMngManager permissionMngManager;
    @Autowired
    private PermissionQueryManager permissionQueryManager;
    @Autowired
    private RoleMngManager roleMngManager;
    @Autowired
    private DataSecurityProperties dataSecurityProperties;
    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private RoleFieldPermissionRepoProc roleFieldPermissionRepoProc;
    @Autowired
    private UserQueryServiceImpl userQueryService;
    private static final Set<String> INNER_BUSINESS_OBJECT = new HashSet<>(8);

    static {
        INNER_BUSINESS_OBJECT.add(BusinessObjectConstant.SYS_USER.split(":")[0]);
        INNER_BUSINESS_OBJECT.add(BusinessObjectConstant.SYS_ORG.split(":")[0]);
        INNER_BUSINESS_OBJECT.add(BusinessObjectConstant.SYS_EMPLOYEE.split(":")[0]);
    }

    @Override
    public ApiResult<List<GroupRoleRespVO>> listGroupRole(Boolean showDisabled, Boolean showAll) {
        // 默认只显示启用的
        showDisabled = showDisabled != null && showDisabled;
        showAll = showAll != null && showAll;

        BelongType.Belonger belonger = showAll ? null : BelongType.getBelonger();
        var dataList = roleMngManager.listGroupRole(showDisabled, belonger);
        return ApiResult.ok(dataList);
    }

    @Override
    public ApiResult<List<GroupRoleRespVO>> listGroupRoleByRelated(String relatedType, String relationId) {
        var dataList = roleMngManager.listGroupRoleByRelated(relatedType, relationId, BelongType.getBelonger());
        return ApiResult.ok(dataList);
    }

    @Override
    public ApiResult<List<IdCodeNameCheckParam>> getRoleByUserId(Long userId, Boolean onlyHave) {
        Assert.notNull(userId, "账号ID为空");
        if (onlyHave == null) {
            onlyHave = true;
        }

        var roleList = permissionMngManager.getUserRole(userId, onlyHave);
        return ApiResult.ok(roleList);
    }

    @Override
    public ApiResult<List<RolePermissionRespVO>> getPermissionMenuByRole(Long roleId, String appCode, Boolean tree) {
        Assert.notNull(roleId, "角色ID为空");

        tree = tree == null || tree;
        var menuList = permissionMngManager.getPermissionMenuByRole(roleId, appCode, false, true, tree);
        return ApiResult.ok(menuList);
    }

    @Override
    public ApiResult<PermissionDetailRespVO> getPermissionDetailByCode(String code) {
        Assert.hasText(code, "权限编码为空");

        // 是否自定义菜单树
        var tenantId = super.currentTenantId();
        var enabledCustom = ObjectUtil.defaultIfNull(tenantMenuRepoProc.getEnabledByTenant(tenantId), false);

        // 获取信息
        var respVo = enabledCustom ? this.detailForCustom(code, tenantId) : this.detailForDefault(code, tenantId);
        if (respVo == null) {
            return ApiResult.ok(this.detailForApp(code));
        }

        // 详细信息
        if (StrUtil.isBlank(respVo.getNodeType())) {
            return ApiResult.fail("未知节点类型");
        }
        if (MenuTreeNodeType.APP.getValue().equals(respVo.getNodeType())) {
            // 节点是应用
            respVo = this.detailForApp(code);
            return ApiResult.ok(respVo);
        }
        if (MenuTreeNodeType.MENU.getValue().equals(respVo.getNodeType())) {
            // 节点是菜单
            this.fillApp(respVo);
            return ApiResult.ok(respVo);
        }
        if (MenuTreeNodeType.ACTION.getValue().equals(respVo.getNodeType())) {
            // 节点是按钮
            this.fillApp(respVo);
            this.fillMenu(respVo);
            return ApiResult.ok(respVo);
        }
        if (MenuTreeNodeType.API.getValue().equals(respVo.getNodeType())) {
            // 节点是API
            this.fillApp(respVo);
            this.fillMenu(respVo);
            return ApiResult.ok(respVo);
        }

        return ApiResult.fail("不支持的节点类型：" + respVo.getNodeType());
    }

    @Override
    public ApiResult<List<SysApiPermissionMetadataDTO>> queryApiMetadata(String appCode) {
        Assert.hasText(appCode, "应用编码为空");

        // 先查询应用下的权限
        var permissionMetadataList = tenantDataIsolateProvider.byDefaultDirectly(() -> apiRepoProc.queryPermissionMetadata(appCode));
        if (permissionMetadataList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 判断是否有自定义菜单树
        var tenantId = super.currentTenantId();
        var enabledCustom = tenantMenuRepoProc.getEnabledByTenant(tenantId);
        // 查询权限与角色关联
        var rolePermissionMap = rolePermissionRepoProc.queryRoleCodeForApi(ObjectUtil.defaultIfNull(enabledCustom, false));

        for (SysApiPermissionMetadataDTO dto : permissionMetadataList) {
            dto.setRoleCodes(rolePermissionMap.getOrDefault(dto.getPermissionCode(), Collections.emptySet()));
        }

        return ApiResult.ok(permissionMetadataList);
    }

    @Override
    public ApiResult<List<CodeNameParam>> dataPermissionRanges(String operationCode, String businessObjectCode) {
        Set<String> ignored = new HashSet<>(8);
        ignored.add(SysDprValueType.DPR_SYS_INTERNALLY_EMPLOYEE_SALESMAN.code());
        ignored.add(SysDprValueType.DPR_SYS_INTERNALLY_DYNAMIC.code());
        List<CodeNameParam> dataList = DprValueResolverFactory.getValueTypes()
                .stream()
                .filter(t -> !ignored.contains(t.getCode()))
                .collect(Collectors.toList());

        // 获取动态字段
        if (StringUtils.hasText(operationCode) || StringUtils.hasText(businessObjectCode)) {
            var userTypes = tenantDataIsolateProvider.byDefaultDirectly(() -> StringUtils.hasText(operationCode) ? businessParamRepoProc.listSimpleBoByBusinessOperationCode(operationCode, null)
                            : businessParamRepoProc.listSimpleBoByBusinessObjectCode(businessObjectCode))
                    .stream()
                    .filter(t -> Boolean.TRUE.equals(t.getDataPermissionEnabled()))
                    .map(BusinessParamBO::getUserType)
                    .filter(StringUtils::hasText)
                    .collect(Collectors.toSet());
            if (!userTypes.isEmpty()) {
                var nameMap = udcMap(new UserType());
                for (Map.Entry<String, String> entry : nameMap.entrySet()) {
                    if (userTypes.contains(entry.getKey())) {
                        dataList.add(new CodeNameParam(DprSysInternallyDynamic.PREFIX + entry.getKey(),
                                "本人-" + entry.getValue() + "(ID)"));
                    }
                }
            }
        }

        return ApiResult.ok(dataList);
    }

    @Override
    public ApiResult<List<CodeNameParam>> dataPermissionRuleCondition() {
        var result = Arrays.stream(DprRuleConditionEnum.values())
                .map(DprRuleConditionEnum::getUdcVO)
                .sorted(Comparator.comparing(DprUdcVO::getValSortNo, Comparator.nullsLast(Integer::compareTo)))
                .map(t -> new CodeNameParam(t.getUdcVal(), t.getValDesc()))
                .collect(Collectors.toList());
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<CodeNameParam>> dataPermissionRuleValueMethod() {
        var result = Stream.of(
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM,
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS,
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS
                )
                .map(DprRuleValueTypeEnum::getUdcVO)
                .sorted(Comparator.comparing(DprUdcVO::getValSortNo, Comparator.nullsLast(Integer::compareTo)))
                .map(t -> new CodeNameParam(t.getUdcVal(), t.getValDesc()))
                .collect(Collectors.toList());
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<SysMenuApiRespVO>> getApi(String menuCode) {
        Assert.hasText(menuCode, "菜单编码为空");

        Map<String, String> appNameMap = new HashMap<>(32);
        var apiList = tenantDataIsolateProvider.byDefaultDirectly(() -> {
                    for (IdCodeNameParam param : appRepoProc.allParams(null)) {
                        appNameMap.put(param.getCode(), param.getName());
                    }

                    return menuApiRepoProc.queryApiOfMenu(menuCode);
                }).stream()
                .map(t -> {
                    SysMenuApiRespVO respVO = new SysMenuApiRespVO();
                    respVO.setId(t.getId());
                    respVO.setAppCode(t.getAppCode());
                    respVO.setAppName(appNameMap.get(t.getAppCode()));
                    respVO.setName(t.getApiName());
                    respVO.setCode(t.getApiCode());
                    respVO.setRequestType(t.getRequestType());
                    respVO.setApiPath(t.getApiPath());
                    respVO.setApiDescribe(t.getApiDescribe());
                    respVO.setMenusCode(menuCode);

                    return respVO;
                }).collect(Collectors.toList());
        return ApiResult.ok(apiList);
    }

    @Override
    public ApiResult<List<Long>> getMenuIdOfApi(Long apiId, String appCode) {
        Assert.notNull(apiId, "接口ID为空");
        var menuIds = menuApiRepoProc.getMenuIdByApiId(apiId, appCode);
        return ApiResult.ok(menuIds);
    }

    @Override
    public ApiResult<List<PermissionMenuRespVO>> getMenuDetailOfApi(Long apiId, String appCode) {
        Assert.notNull(apiId, "接口ID为空");

        // 获取菜单列表
        var menuList = menuApiRepoProc.getMenuDetailByApiId(apiId, appCode).stream()
                .map(PermissionConverter.INSTANCE::convertMenuRespVO)
                .collect(Collectors.toList());
        if (menuList.isEmpty()) {
            return ApiResult.ok(menuList);
        }

        // 应用信息
        var appMap = appRepoProc.all().stream().collect(Collectors.toMap(SysPlatformAppDO::getAppCode, SysPlatformAppDO::getAppName, (t1, t2) -> t1));

        // 获取上级菜单
        var parentMenuCodes = menuList.stream().map(PermissionMenuRespVO::getMenusParentCode).filter(StringUtils::hasText).collect(Collectors.toSet());
        Map<String, String> parentMenuNameMap = parentMenuCodes.isEmpty() ? Collections.emptyMap() : menusRepoProc.getMenuNameByMenuCode(parentMenuCodes);

        // 翻译名称
        for (PermissionMenuRespVO respVO : menuList) {
            respVO.setMenusParentName(parentMenuNameMap.get(respVO.getMenusParentCode()));
            respVO.setMenusAppName(appMap.get(respVO.getMenusAppCode()));
        }

        return ApiResult.ok(menuList);
    }

    @Override
    public ApiResult<List<UserRoleMenuRespVO>> getMenusOfUser(long userId, Boolean includeAction, Boolean tree) {
        List<UserRoleMenuRespVO> menuList = permissionQueryManager.queryUserRoleMenus(userId, includeAction != null && includeAction, tree == null || tree);
        return ApiResult.ok(menuList);
    }

    @Override
    public ApiResult<List<UserRoleDataPermissionRespVO>> getDataPermissionOfUser(long userId) {
        // 查询用户的数据权限
        SysUserDTO user = userQueryService.getUserById(userId);
        if (user == null) {
            return ApiResult.fail("用户不存在");
        }
        if (CollUtil.isEmpty(user.getRoles())) {
            return ApiResult.ok(Collections.emptyList());
        }
        SysDprRoleApiRowColumnRuleDTO ruleDTO = permissionQueryManager.getDataPermissionByUser(user);
        if (CollUtil.isEmpty(ruleDTO.getSysDprRoleApiDataRuleListQueryDTO()) && CollUtil.isEmpty(ruleDTO.getSysDpcRoleApiFieldsDTOList())) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 根据角色分组
        Map<String, List<SysDprRoleApiDataRuleListQueryDTO>> rowRulesMap = CollUtil.isEmpty(ruleDTO.getSysDprRoleApiDataRuleListQueryDTO()) ? Collections.emptyMap()
                : ruleDTO.getSysDprRoleApiDataRuleListQueryDTO().stream()
                .filter(t -> !DataPermissionType.BUSINESS_RESOURCE_RULE.name().equals(t.getPermissionType())) // 忽略资源权限
                .collect(Collectors.groupingBy(BaseDataSecurityRuleDTO::getRoleCode));
        Map<String, List<SysDpcRoleApiFieldsDTO>> fieldRulesMap = CollUtil.isEmpty(ruleDTO.getSysDpcRoleApiFieldsDTOList()) ? Collections.emptyMap()
                : ruleDTO.getSysDpcRoleApiFieldsDTOList().stream().collect(Collectors.groupingBy(BaseDataSecurityRuleDTO::getRoleCode));

        List<UserRoleDataPermissionRespVO> respVoList = new ArrayList<>(64);
        List<CompletableFuture<Void>> futureList = new ArrayList<>(user.getRoles().size());
        for (IdCodeNameParam role : user.getRoles()) {
            if (!rowRulesMap.containsKey(role.getCode()) && !fieldRulesMap.containsKey(role.getCode())) {
                continue;
            }

            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
                UserRoleDataPermissionRespVO respVO = new UserRoleDataPermissionRespVO();
                respVO.setRoleCode(role.getCode());
                respVO.setRoleName(role.getName());
                respVO.setDataPermissions(this.convertDataPermission(rowRulesMap.get(role.getCode()), fieldRulesMap.get(role.getCode())));

                respVoList.add(respVO);
            });
            futureList.add(future);
        }
        CompletableFuture.allOf(futureList.toArray(new CompletableFuture[0])).join();
        return ApiResult.ok(respVoList);
    }

    private Set<String> collectFieldBusinessObjectCode(List<SysDpcRoleApiFieldsDTO> fieldRuleList) {
        return CollUtil.isEmpty(fieldRuleList) ? Collections.emptySet() : fieldRuleList.stream()
                .map(SysDpcRoleApiFieldsDTO::getBusinessObjectCode)
                .filter(StringUtils::hasText)
                .collect(Collectors.toSet());
    }

    private Set<String> collectRowBusinessObjectCode(List<SysDprRoleApiDataRuleListQueryDTO> rowRuleList) {
        if (CollUtil.isEmpty(rowRuleList)) {
            return Collections.emptySet();
        }

        Set<String> businessObjectCodes = new HashSet<>(rowRuleList.size());
        for (SysDprRoleApiDataRuleListQueryDTO rule : rowRuleList) {
            if (StringUtils.hasText(rule.getBusinessObjectCode())) {
                businessObjectCodes.add(rule.getBusinessObjectCode());
            }
            if (StringUtils.hasText(rule.getRefBusinessObject())) {
                businessObjectCodes.add(rule.getRefBusinessObject());
            }

            businessObjectCodes.addAll(collectRowBusinessObjectCode(rule.getGroupRules()));
        }
        return businessObjectCodes;
    }

    private String generateDataPermissionExpression(List<UserRoleDataPermissionRespVO.RowPermission> permissions, Boolean and, boolean value) {
        if (CollUtil.isEmpty(permissions)) {
            return null;
        }

        if (and == null) {
            for (UserRoleDataPermissionRespVO.RowPermission permission : permissions) {
                if (!Boolean.TRUE.equals(permission.getRuleGroup())) {
                    and = !DprRuleRelationEnum.DPR_RULE_RELATION_OR.name().equals(permission.getDprRuleRelation());
                    break;
                }
            }
            if (and == null) {
                and = true;
            }
        }

        List<String> expressions = new ArrayList<>(permissions.size());
        for (UserRoleDataPermissionRespVO.RowPermission permission : permissions) {
            if (CollUtil.isNotEmpty(permission.getGroupRuleList())) {
                String expression = generateDataPermissionExpression(permission.getGroupRuleList(), !DprRuleRelationEnum.DPR_RULE_RELATION_OR.name().equals(permission.getDprRuleRelation()), value);
                if (expression != null) {
                    expressions.add("(" + expression + ")");
                }
                continue;
            }

            boolean in = DprRuleConditionEnum.InList.name().equals(permission.getDprRuleCondition());
            if (value) {
                expressions.add(String.format("%s %s (%s)", permission.getDprRuleField(), in ? "in" : "not in", permission.getDprRuleValue()));
            } else {
                String ruleValue = CharSequenceUtil.blankToDefault(permission.getDprRuleValueName(), permission.getDprRuleValue());
                if (Boolean.TRUE.equals(permission.getRefResource())) {
                    ruleValue = "【公共资源】" + permission.getRefBusinessObjectName();
                }
                expressions.add(String.format("%s %s (%s)", permission.getDprRuleFieldName(), in ? "属于" : "不属于", ruleValue));
            }
        }

        return expressions.isEmpty() ? null : String.join(and ? (value ? " and " : " 且 ") : (value ? " or " : " 或 "), expressions);
    }

    private List<UserRoleDataPermissionRespVO.DataPermission> convertDataPermission(List<SysDprRoleApiDataRuleListQueryDTO> rowRuleList,
                                                                                    List<SysDpcRoleApiFieldsDTO> fieldRuleList) {
        // 根据权限类型分组
        Function<BaseDataSecurityRuleDTO, String> permissionGroupKey = t -> String.join("::", t.getPermissionType(), t.getAppCode(), t.getBusinessObjectCode(), t.getApiPermissionCode(), t.getMenusCode());
        Map<String, List<SysDprRoleApiDataRuleListQueryDTO>> rowRulesMap = CollUtil.isEmpty(rowRuleList) ? Collections.emptyMap()
                : rowRuleList.stream().collect(Collectors.groupingBy(permissionGroupKey));
        Map<String, List<SysDpcRoleApiFieldsDTO>> fieldRulesMap = CollUtil.isEmpty(fieldRuleList) ? Collections.emptyMap()
                : fieldRuleList.stream().collect(Collectors.groupingBy(permissionGroupKey));
        // 应用
        Map<String, String> appMap = appRepoProc.allParams(null).stream().collect(Collectors.toMap(IdCodeNameParam::getCode, IdCodeNameParam::getName, (t1, t2) -> t1));
        // 菜单
        Map<String, String> menuMap = menusRepoProc.getMenuNames(null, false, false, false);
        // 业务对象
        Set<String> businessObjectCodes = new HashSet<>(collectRowBusinessObjectCode(rowRuleList));
        businessObjectCodes.addAll(collectFieldBusinessObjectCode(fieldRuleList));
        Map<String, String> businessObjectMap = businessObjectCodes.isEmpty() ? Collections.emptyMap() : businessObjectRepoProc.getCodeAndNames(businessObjectCodes);
        // 业务操作
        Map<String, String> businessOperationMap = businessObjectCodes.isEmpty() ? Collections.emptyMap() : businessOperationRepoProc.getNamesByBusinessObject(businessObjectCodes);
        // 业务对象参数
        Map<String, Map<String, String>> businessParamsMap = businessObjectCodes.isEmpty() ? Collections.emptyMap() : businessParamRepoProc.getNamesByBusinessObject(businessObjectCodes);
        Function<BaseDataSecurityRuleDTO, UserRoleDataPermissionRespVO.DataPermission> convert = t -> {
            UserRoleDataPermissionRespVO.DataPermission dataPermission = new UserRoleDataPermissionRespVO.DataPermission();
            dataPermission.setPermissionType(t.getPermissionType());
            dataPermission.setPermissionTypeName(DataPermissionType.getDescription(t.getPermissionType()));
            dataPermission.setPermissionRef(t.getPermissionRef());
            dataPermission.setAppCode(t.getAppCode());
            dataPermission.setAppName(appMap.get(t.getAppCode()));
            dataPermission.setBusinessObjectCode(t.getBusinessObjectCode());
            dataPermission.setBusinessObjectName(businessObjectMap.get(t.getBusinessObjectCode()));
            dataPermission.setApiPermissionCode(t.getApiPermissionCode());
            dataPermission.setApiPermissionName(CharSequenceUtil.isBlank(t.getApiPermissionCode()) ? null : businessOperationMap.get(t.getApiPermissionCode()));
            dataPermission.setMenusCode(t.getMenusCode());
            dataPermission.setMenusName(CharSequenceUtil.isBlank(t.getMenusCode()) ? null : menuMap.get(t.getMenusCode()));

            return dataPermission;
        };

        List<UserRoleDataPermissionRespVO.DataPermission> dataPermissionList = new ArrayList<>(64);

        // 根据行权限
        for (Map.Entry<String, List<SysDprRoleApiDataRuleListQueryDTO>> entry : rowRulesMap.entrySet()) {
            UserRoleDataPermissionRespVO.DataPermission dataPermission = convert.apply(entry.getValue().get(0));
            dataPermission.setRowPermissions(convertRowPermissions(entry.getValue(), businessObjectMap, businessParamsMap));
            dataPermission.setFieldPermissions(convertFieldPermissions(fieldRulesMap.get(entry.getKey()), businessParamsMap.get(dataPermission.getBusinessObjectCode())));
            dataPermission.setRowPermissionExpression(generateDataPermissionExpression(dataPermission.getRowPermissions(), true, false));
            dataPermission.setRowPermissionValueExpression(generateDataPermissionExpression(dataPermission.getRowPermissions(), true, true));

            dataPermissionList.add(dataPermission);
        }
        // 根据字段权限
        for (Map.Entry<String, List<SysDpcRoleApiFieldsDTO>> entry : fieldRulesMap.entrySet()) {
            UserRoleDataPermissionRespVO.DataPermission dataPermission = convert.apply(entry.getValue().get(0));
            dataPermission.setRowPermissions(Collections.emptyList());
            dataPermission.setFieldPermissions(convertFieldPermissions(entry.getValue(), businessParamsMap.get(dataPermission.getBusinessObjectCode())));

            dataPermissionList.add(dataPermission);
        }
        return dataPermissionList;
    }

    private List<UserRoleDataPermissionRespVO.RowPermission> convertRowPermissions(List<SysDprRoleApiDataRuleListQueryDTO> ruleList,
                                                                                   Map<String, String> businessObjectMap,
                                                                                   Map<String, Map<String, String>> businessParamsMap) {
        if (CollUtil.isEmpty(ruleList)) {
            return Collections.emptyList();
        }

        return ruleList.stream().map(t -> {
            var permission = PermissionConverter.INSTANCE.convert2UserPermissionRow(t);

            // 规则组
            if (CollUtil.isNotEmpty(t.getGroupRules())) {
                permission.setRuleGroup(true);
                permission.setGroupRuleList(convertRowPermissions(t.getGroupRules(), businessObjectMap, businessParamsMap));
                return permission;
            }

            if (permission.getRefBusinessObject() != null) {
                permission.setRefBusinessObjectName(businessObjectMap.get(permission.getRefBusinessObject()));
            }

            // 字段
            var params = businessParamsMap.get(t.getBusinessObjectCode());
            if (params != null) {
                permission.setDprRuleFieldName(params.get(t.getDprRuleField()));
            }
            if (permission.getRefBusinessObject() != null && permission.getRefField() != null) {
                params = businessParamsMap.get(permission.getRefBusinessObject());
                permission.setRefFieldName(params.get(permission.getRefField()));
            }

            permission.setDprRuleConditionName(DprRuleConditionEnum.getValue(t.getDprRuleCondition()));
            permission.setDprRuleValueTypeName(DprRuleValueTypeEnum.getValue(t.getDprRuleValueType()));

            // 值名称
            if (CharSequenceUtil.isBlank(permission.getDprRuleValueName())) {
                if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name().equals(permission.getDprRuleValueType()) && StringUtils.hasText(permission.getDprRuleValue())) {
                    permission.setDprRuleValueName(DprValueResolverFactory.convertDprValueTypeName(permission.getDprRuleValue()));
                }
                if (permission.getDprRuleValueName() == null) {
                    permission.setDprRuleValueName(permission.getDprRuleValue());
                }
            }
            return permission;
        }).collect(Collectors.toList());
    }

    private List<UserRoleDataPermissionRespVO.FieldPermission> convertFieldPermissions(List<SysDpcRoleApiFieldsDTO> ruleList,
                                                                                       Map<String, String> businessParamsMap) {
        if (CollUtil.isEmpty(ruleList)) {
            return Collections.emptyList();
        }

        return ruleList.stream().map(t -> {
            var permission = PermissionConverter.INSTANCE.convert2UserPermissionField(t);
            if (businessParamsMap != null) {
                permission.setFieldDescription(businessParamsMap.get(t.getFieldName()));
            }
            return permission;
        }).collect(Collectors.toList());
    }

    @Override
    public ApiResult<SysDprRespVO> getApiDataPermission(Long roleId, String menuCode, String apiCode) {
        Assert.notNull(roleId, "角色ID为空");
        Assert.hasText(menuCode, "菜单编码为空");
        Assert.hasText(apiCode, "接口编码为空");

        // 查询自定义规则
        var respVO = queryRoleCustomRule(roleId, menuCode, apiCode);
        // 查询字段权限
        respVO.setFieldList(this.queryRoleApiFields(roleId, menuCode, apiCode));
        // 查询入参
        respVO.setArgList(this.queryReqArguments(apiCode));

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<DataPermissionRespVO> getDataPermission(String roleCode, String permissionType, String menuCode,
                                                             String operationCode, String businessObjectCode) {
        Assert.hasText(roleCode, "角色编码为空");
        Assert.hasText(permissionType, "数据权限类型为空");
        var dataPermissionType = DataPermissionType.valueOf(permissionType);
        Assert.notNull(dataPermissionType, "数据权限类型错误");

        DataPermissionRespVO respVO = new DataPermissionRespVO();

        // 查询规则列表
        var permissionDoList = this.queryDataPermissionRule(roleCode, dataPermissionType, menuCode,
                operationCode, businessObjectCode);
        respVO.setCustomRuleList(this.convertRule(permissionDoList));
        DprRuleRelationEnum relation = null;
        for (DataPermissionRuleRespVO rule : respVO.getCustomRuleList()) {
            if (!Boolean.TRUE.equals(rule.getRuleGroup())) {
                relation = rule.getGroupRelation();
                break;
            }
        }
        if (relation == null) {
            relation = DprRuleRelationEnum.DPR_RULE_RELATION_AND;
        }
        respVO.setCustomRuleExpression(convertPrettyExpression(respVO.getCustomRuleList(), relation));

        AtomicReference<String> refOperationCode = new AtomicReference<>();
        List<BusinessOperationParamBO> operationParamList = new ArrayList<>(64);
        // 参数列表
        var paramList = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            if (StringUtils.hasText(operationCode)) {
                refOperationCode.set(businessOperationRepoProc.getPermissionRefByOperationCode(operationCode));
                operationParamList.addAll(queryResponseParams(operationCode));
            }
            return this.queryPermissionParams(dataPermissionType, businessObjectCode, operationCode);
        });
        respVO.setArgList(this.convertRuleArgList(paramList));
        // 更新规则的参数
        if (CollUtil.isNotEmpty(respVO.getCustomRuleList())) {
            Map<String, DataPermissionParamRespVO> argMap = respVO.getArgList().stream().collect(Collectors.toMap(DataPermissionParamRespVO::getFieldName, Function.identity(), (t1, t2) -> t1));
            for (var vo : respVO.getCustomRuleList()) {
                var arg = argMap.get(vo.getDprRuleField());
                if (arg != null) {
                    vo.setRefBusinessObject(arg.getRelatedBusinessObject());
                    vo.setRefField(arg.getRelatedField());
                }
            }
        }
        var fieldPermissionList = this.queryFieldPermission(roleCode, dataPermissionType, menuCode,
                operationCode, businessObjectCode);
        respVO.setFieldList(this.convertFieldList(paramList, fieldPermissionList, operationParamList));

        // 是否引用了依赖的规则
        respVO.setUseRef(StringUtils.hasText(refOperationCode.get()) && CollUtil.isEmpty(permissionDoList) && CollUtil.isEmpty(fieldPermissionList));
        if (Boolean.TRUE.equals(respVO.getUseRef())) {
            var refPermission = getDataPermission(roleCode, permissionType, menuCode, refOperationCode.get(), businessObjectCode).getData();
            respVO.setCustomRuleList(refPermission.getCustomRuleList());
            respVO.setFieldList(refPermission.getFieldList());
        }

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<Set<Long>> getUserIdByRoleCode(String roleCode) {
        Assert.hasText(roleCode, "角色编码为空");

        var userIds = permissionQueryManager.queryUserIdOfRole(roleCode);
        return ApiResult.ok(userIds);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> listAllRoles(String tenantCode) {
        // 默认获取当前租户
        if (CharSequenceUtil.isBlank(tenantCode)) {
            var roleList = roleRepoProc.listQuery(BelongType.getBelonger(), null, null);
            return ApiResult.ok(roleList);
        }

        // 查询指定租户的
        var tenant = tenantClientProvider.getTenantByCode(tenantCode);
        if (tenant == null) {
            return ApiResult.fail("租户不存在");
        }
        var roleList = tenantDataIsolateProvider.byTenantDirectly(() -> roleRepoProc.listQuery(new BelongType.Belonger(BelongType.TENANT, tenant.getId().toString()),
                null, null), tenant);
        return ApiResult.ok(roleList);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> listAllRolesByTenantOrg(String tenantOrgCode) {
        Long orgId = null;
        if (StringUtils.hasText(tenantOrgCode)) {
            orgId = tenantOrgRepoProc.getOrgIdByCode(tenantOrgCode, false);
            if (orgId == null) {
                return ApiResult.fail("租户组织不存在或已禁用");
            }
        } else {
            // 获取当前用户的租户组织
            orgId = tenantOrgProvider.getSessionTenantOrgId();
            if (orgId == null) {
                return ApiResult.ok(Collections.emptyList());
            }
        }

        var roleList = roleRepoProc.listQuery(new BelongType.Belonger(BelongType.TENANT_ORG, orgId.toString()), null, null);
        return ApiResult.ok(roleList);
    }

    @Override
    public ApiResult<List<SysUserBasicDTO>> listUserByRole(String roleCode) {
        Assert.hasText(roleCode, "角色编码为空");
        var userIds = permissionQueryManager.queryUserIdOfRole(roleCode);
        if (CollUtil.isEmpty(userIds)) {
            return ApiResult.ok(Collections.emptyList());
        }

        var userList = tenantDataIsolateProvider.byDefaultDirectly(() -> userRepoProc.getBasicDto(userIds));
        return ApiResult.ok(userList);
    }

    @Override
    public ApiResult<List<EmployeeUserInfoRespVO>> listEmployeeUserByRole(Long roleId, Boolean onlyEmployee) {
        if (roleId == null) {
            return ApiResult.fail("角色ID为空");
        }
        if (onlyEmployee == null) {
            onlyEmployee = true;
        }

        var respVoLList = permissionQueryManager.queryUserOfRole(roleId, onlyEmployee);
        return ApiResult.ok(respVoLList);
    }

    @Override
    public ApiResult<List<DataPermissionTreeNodeRespVO>> dataPermissionTree(Long roleId, String appCode, Boolean tree) {
        tree = tree == null || tree;
        // 先查询菜单树
        var menuList = permissionMngManager.getDataPermissionByRole(roleId, appCode, true, tree);
        return ApiResult.ok(menuList);
    }

    @Override
    public ApiResult<List<DataPermissionTreeNodeRespVO>> dataPermissionTree(String roleCode, String dataPermissionTypeValue,
                                                                            Boolean withRef, String appCode, Boolean tree) {
        tree = tree == null || tree;
        withRef = withRef != null && withRef;

        var dataPermissionType = DataPermissionType.valueOf(dataPermissionTypeValue);

        var permissionList = permissionMngManager.getDataPermissionByRole(roleCode, dataPermissionType, withRef, appCode, true, tree);
        return ApiResult.ok(permissionList);
    }

    @Override
    public ApiResult<SysDprRoleApiRowColumnRuleDTO> getDataPermissionOfCurrentUser() {
        var roleRuleDTO = permissionQueryManager.getDataPermissionOfCurrentUser();
        return ApiResult.ok(roleRuleDTO);
    }

    private BusinessObjectBO getBusinessObjectInfo(String businessObjectCode, String operationCode, String operationRefCode) {
        if (StringUtils.hasText(operationRefCode)) {
            return businessObjectRepoProc.getSimpleByOperationCode(operationRefCode);
        }

        if (StringUtils.hasText(operationCode)) {
            return businessObjectRepoProc.getSimpleByOperationCode(operationCode);
        }

        if (StringUtils.hasText(businessObjectCode)) {
            return businessObjectRepoProc.getSimple(businessObjectCode);
        }

        return null;
    }

    private List<BusinessOperationParamBO> queryResponseParams(String operationCode) {
        var operationParams = businessOperationParamRepoProc.listBoByOperationCode(operationCode, BusinessObjectParamInEnum.RESPONSE_BODY);
        if (operationParams.isEmpty()) {
            return Collections.emptyList();
        }

        var paramVoList = businessObjectManager.convertParam(operationParams, null);
        return filterResponseData(paramVoList);
    }

    private List<BusinessOperationParamBO> filterResponseData(List<BusinessOperationParamBO> paramList) {
        if (CollUtil.isEmpty(paramList)) {
            return Collections.emptyList();
        }
        for (var operationParam : paramList) {
            if (operationParam.isApiResultData()) {
                // 判断是否有分页查询的
                var pageData = businessObjectManager.attemptObtainPagingData(operationParam.getChildrenParams());
                return CollUtil.defaultIfEmpty(pageData, operationParam.getChildrenParams());
            }
        }
        return paramList;
    }

    private List<DataPermissionFieldRespVO> convertFieldList(List<BusinessParamBO> paramList, List<SysRoleFieldPermissionDO> permissionList,
                                                             List<BusinessOperationParamBO> operationParamList) {
        if (CollectionUtils.isEmpty(paramList)) {
            return Collections.emptyList();
        }

        Map<String, SysRoleFieldPermissionDO> permissionMap = CollUtil.isEmpty(permissionList) ? Collections.emptyMap() :
                permissionList.stream().collect(Collectors.toMap(SysRoleFieldPermissionDO::getFieldName, Function.identity(), (t1, t2) -> t1));

        var fieldList = paramList.stream()
                .filter(t -> {
                    if (Boolean.FALSE.equals(t.getEnabled())) {
                        return false;
                    }
                    if (Boolean.FALSE.equals(t.getFieldPermissionEnabled())) {
                        if (Boolean.TRUE.equals(t.getBaseField())) {
                            // 基础字段
                            return dataSecurityProperties.getFieldPermission().getBaseFields().contains(t.getFieldName());
                        }
                        return false;
                    }

                    return true;
                })
                .map(t -> {
                    DataPermissionFieldRespVO respVO = new DataPermissionFieldRespVO();
                    respVO.setEntity(true);
                    respVO.setFieldName(t.getFieldName());
                    respVO.setFieldRemark(CharSequenceUtil.blankToDefault(t.getCustomDescription(), t.getFieldDescription()));

                    var permission = permissionMap.get(t.getFieldName());
                    if (permission == null) {
                        // 默认值
                        respVO.setReadable(ObjUtil.defaultIfNull(dataSecurityProperties.getDefaultAllow(), true));
                        respVO.setWriteable(ObjUtil.defaultIfNull(dataSecurityProperties.getDefaultAllow(), true));
                    } else {
                        respVO.setReadable(permission.getReadable());
                        respVO.setWriteable(permission.getWriteable());
                    }

                    return respVO;
                }).collect(Collectors.toList());

        // 添加操作的参数
        if (CollUtil.isNotEmpty(operationParamList)) {
            Set<String> excludeFieldNames = Set.of("extensionInfo");
            var fieldNames = fieldList.stream().map(DataPermissionFieldRespVO::getFieldName).collect(Collectors.toSet());
            for (var operationParam : operationParamList) {
                if (operationParam.getFieldName() == null || fieldNames.contains(operationParam.getFieldName())) {
                    continue;
                }
                if (excludeFieldNames.contains(operationParam.getFieldName())) {
                    continue;
                }
                DataPermissionFieldRespVO respVO = new DataPermissionFieldRespVO();
                respVO.setFieldName(operationParam.getFieldName());
                respVO.setFieldRemark(operationParam.getFieldDescription());
                respVO.setEntity(false);

                var permission = permissionMap.get(operationParam.getFieldName());
                if (permission == null) {
                    // 默认值
                    respVO.setReadable(true);
                    respVO.setWriteable(true);
                } else {
                    respVO.setReadable(permission.getReadable());
                    respVO.setWriteable(permission.getWriteable());
                }

                fieldNames.add(operationParam.getFieldName());
                fieldList.add(respVO);
            }
        }

        return fieldList;
    }

    private List<DataPermissionParamRespVO> convertRuleArgList(List<BusinessParamBO> paramList) {
        if (CollectionUtils.isEmpty(paramList)) {
            return Collections.emptyList();
        }

        var argList = paramList.stream()
                .filter(t -> {
                    if (Boolean.TRUE.equals(t.getDataPermissionEnabled())) {
                        return true;
                    }

                    if (Boolean.TRUE.equals(t.getBaseField())) {
                        // 基础字段
                        return dataSecurityProperties.getBaseFields().contains(t.getFieldName());
                    }

                    return false;
                })
                .map(t -> {
                    DataPermissionParamRespVO respVO = new DataPermissionParamRespVO();
                    respVO.setId(t.getId());
                    respVO.setFieldName(t.getFieldName());
                    respVO.setFieldRemark(CharSequenceUtil.blankToDefault(t.getCustomDescription(), t.getFieldDescription()));
                    respVO.setEnabled(t.getEnabled());
                    respVO.setRelatedBusinessObject(t.getRelatedBusinessObject());
                    respVO.setRelatedField(t.getRelatedField());
                    respVO.setRelatedResource(false);

                    // UDC时
                    if (StringUtils.hasText(t.getValuesJson()) && StringUtils.hasText(t.getRelatedBusinessObject())
                            && t.getRelatedBusinessObject().equals(BusinessObjectConstant.SYS_PLATFORM_UDC.split(":")[0])) {
                        String[] values = JSONUtil.json2Obj(t.getValuesJson(), new TypeReference<>() {
                        });
                        if (values.length >= 2) {
                            respVO.setRelatedAppCode(values[0]);
                            respVO.setRelatedUdcCode(values[1]);
                        }
                    }
                    if (Boolean.TRUE.equals(t.getBaseField())) {
                        this.fillBaseModelParam(respVO);
                    }
                    respVO.setInner(respVO.getRelatedBusinessObject() != null && isInnerBusinessObject(respVO.getRelatedBusinessObject()));

                    return respVO;
                }).collect(Collectors.toList());
        if (argList.isEmpty()) {
            return argList;
        }

        // 设置关联的业务对象信息
        var relatedBusinessObjectCodes = argList.stream().map(DataPermissionParamRespVO::getRelatedBusinessObject).filter(StringUtils::hasText).collect(Collectors.toSet());
        if (!relatedBusinessObjectCodes.isEmpty()) {
            var businessObjectMap = tenantDataIsolateProvider.byDefaultDirectly(() -> businessObjectRepoProc.listSimple(relatedBusinessObjectCodes)).stream().collect(Collectors.toMap(BusinessObjectBO::getCode, Functions.identity(), (t1, t2) -> t1));
            for (DataPermissionParamRespVO paramRespVO : argList) {
                if (!StringUtils.hasText(paramRespVO.getRelatedBusinessObject())) {
                    continue;
                }
                var ob = businessObjectMap.get(paramRespVO.getRelatedBusinessObject());
                if (ob == null) {
                    log.error("关联的业务对象不存在：{}", paramRespVO.getRelatedBusinessObject());
                    continue;
                }
                paramRespVO.setRelatedBusinessObjectName(CharSequenceUtil.blankToDefault(ob.getCustomName(), ob.getName()));
                paramRespVO.setRelatedResource(ob.getPublicResource());
            }
        }

        return argList;
    }

    private boolean isInnerBusinessObject(String businessObject) {
        if (INNER_BUSINESS_OBJECT.contains(businessObject)) {
            return true;
        }

        if (systemProperties.getBusinessObjectInnerCustom() != null && systemProperties.getBusinessObjectInnerCustom().contains(businessObject)) {
            return true;
        }

        return false;
    }

    private void fillBaseModelParam(DataPermissionParamRespVO respVO) {
        if (dataSecurityProperties.getOrgIdBaseFields().contains(respVO.getFieldName())) {
            respVO.setRelatedBusinessObject(BusinessObjectConstant.SYS_ORG.split(":")[0]);
            respVO.setRelatedField("id");
        } else if (dataSecurityProperties.getUserIdBaseFields().contains(respVO.getFieldName())) {
            respVO.setRelatedBusinessObject(BusinessObjectConstant.SYS_USER.split(":")[0]);
            respVO.setRelatedField("id");
        }
    }

    private List<BusinessParamBO> queryPermissionParams(DataPermissionType permissionType, String businessObjectCode,
                                                        String operationCode) {
        List<BusinessParamBO> paramBoList = null;
        switch (permissionType) {
            case MENU_OPERATION_RULE:
            case BUSINESS_OPERATION_RULE:
                paramBoList = businessParamRepoProc.listSimpleBoByBusinessOperationCode(operationCode, null);
                break;
            case BUSINESS_OBJECT_RULE:
            case BUSINESS_RESOURCE_RULE:
                paramBoList = businessParamRepoProc.listSimpleBoByBusinessObjectCode(businessObjectCode);
                break;
            default:
                throw new BusinessException("查询参数失败，暂不支持的权限类型");
        }

        return paramBoList;
    }

    private String convertPrettyExpression(List<DataPermissionRuleRespVO> ruleRespVoList, DprRuleRelationEnum relation) {
        if (CollUtil.isEmpty(ruleRespVoList)) {
            return null;
        }

        List<String> expressions = new ArrayList<>();
        for (DataPermissionRuleRespVO respVO : ruleRespVoList) {
            if (Boolean.TRUE.equals(respVO.getRuleGroup())) {
                String groupExpression = convertPrettyExpression(respVO.getGroupRules(), respVO.getGroupRelation());
                if (groupExpression != null) {
                    expressions.add("(" + groupExpression + ")");
                }
                continue;
            }
            expressions.add(respVO.getDprRuleName());
        }

        return String.join(" " + relation.getValueDescription() + " ", expressions);
    }

    private List<DataPermissionRuleRespVO> convertRule(List<SysRoleDataPermissionDO> permissionDoList) {
        if (CollUtil.isEmpty(permissionDoList)) {
            return Collections.emptyList();
        }

        var ruleList = permissionDoList.stream().map(t -> {
            DataPermissionRuleRespVO respVO = new DataPermissionRuleRespVO();
            respVO.setRuleGroup(Boolean.TRUE.equals(t.getRuleGroup()));
            respVO.setGroupRelation(DprRuleRelationEnum.valueOf(t.getRuleRelation()));
            if (respVO.getRuleGroup()) {
                // 规则组
                respVO.setRuleGroupCode(t.getRuleGroupCode());
                int index = t.getRuleGroupCode().lastIndexOf("_");
                String parentId = index > 0 ? t.getRuleGroupCode().substring(0, index) : SysRoleDataPermissionDO.DEFAULT_GROUP_CODE;
                respVO.setRuleGroupCodeParent(parentId);
                respVO.setRuleOrder(Integer.valueOf(t.getRuleGroupCode().substring(index + 1)));
                respVO.setGroupRules(new ArrayList<>());
            } else {
                respVO.setRuleGroupCode(t.getId().toString());
                respVO.setRuleGroupCodeParent(CharSequenceUtil.blankToDefault(t.getRuleGroupCode(), SysRoleDataPermissionDO.DEFAULT_GROUP_CODE));
                respVO.setRuleOrder(ObjUtil.defaultIfNull(t.getRuleOrder(), 0));
                respVO.setGroupRules(null);
            }

            respVO.setId(t.getId());
            respVO.setDprRuleName(t.getRuleName());
            respVO.setDprRuleField(t.getRuleField());
            respVO.setRefResource(Boolean.TRUE.equals(t.getRefResource()));
            respVO.setRefBusinessObject(t.getRefBusinessObject());
            respVO.setRefField(t.getRefField());
            if (StringUtils.hasText(t.getFieldValueCondition())) {
                var condition = DprRuleConditionEnum.valueOf(t.getFieldValueCondition());
                respVO.setDprRuleCondition(condition);
                respVO.setDprRuleConditionName(condition.getValueDescription());
            }
            if (StringUtils.hasText(t.getRuleValueType())) {
                var valueType = DprRuleValueTypeEnum.valueOf(t.getRuleValueType());
                respVO.setDprRuleValueType(valueType);
                respVO.setDprRuleValueTypeName(valueType.getValueDescription());
            }

            respVO.setDataSet(t.getDataSet());
            respVO.setDprRuleValue(t.getRuleValue());
            respVO.setDprRuleValueName(t.getRuleValueName());
            respVO.setDprRuleDeclare(t.getRuleDescription());
            respVO.setBs1(t.getBs1());
            respVO.setBs2(t.getBs2());
            respVO.setBs3(t.getBs3());

            return respVO;
        }).collect(Collectors.toList());

        var refBusinessObjectCodes = ruleList.stream().map(DataPermissionRuleRespVO::getRefBusinessObject).filter(CharSequenceUtil::isNotBlank).collect(Collectors.toSet());
        if (!refBusinessObjectCodes.isEmpty()) {
            var businessObjectMap = tenantDataIsolateProvider.byDefaultDirectly(() -> businessObjectRepoProc.listSimple(refBusinessObjectCodes)).stream().collect(Collectors.toMap(BusinessObjectBO::getCode, Function.identity(), (t1, t2) -> t1));
            for (DataPermissionRuleRespVO respVO : ruleList) {
                if (StringUtils.hasText(respVO.getRefBusinessObject())) {
                    ObjUtil.ifNotNull(businessObjectMap.get(respVO.getRefBusinessObject()), ob -> respVO.setRefBusinessObjectName(CharSequenceUtil.blankToDefault(ob.getCustomName(), ob.getName())));
                }
            }
        }
        return (List<DataPermissionRuleRespVO>) new TreeDataUtil<>(ruleList, DataPermissionRuleRespVO::getRuleGroupCode, DataPermissionRuleRespVO::getRuleGroupCodeParent,
                DataPermissionRuleRespVO::setGroupRules, Comparator.comparingInt(DataPermissionRuleRespVO::getRuleOrder))
                .getRoots();
    }

    private List<SysRoleDataPermissionDO> queryDataPermissionRule(String roleCode, DataPermissionType permissionType,
                                                                  String menuCode, String operationCode, String businessObjectCode) {
        switch (permissionType) {
            case MENU_OPERATION_RULE:
                Assert.hasText(menuCode, "菜单编码为空");
                Assert.hasText(operationCode, "操作编码为空");
                return dataPermissionRepoProc.listByRoleForMenuOperation(roleCode, menuCode, operationCode);
            case BUSINESS_OPERATION_RULE:
                Assert.hasText(operationCode, "操作编码为空");
                return dataPermissionRepoProc.listByRoleForBusinessOperation(roleCode, operationCode);
            case BUSINESS_OBJECT_RULE:
                Assert.hasText(businessObjectCode, "业务对象编码为空");
                return dataPermissionRepoProc.listByRoleForBusinessObject(roleCode, businessObjectCode);
            case BUSINESS_RESOURCE_RULE:
                Assert.hasText(businessObjectCode, "业务对象编码为空");
                return dataPermissionRepoProc.listByRoleForBusinessObjectResource(roleCode, businessObjectCode);
            default:
                throw new BusinessException("查询权限失败，暂不支持的数据权限类型");
        }
    }

    private List<SysRoleFieldPermissionDO> queryFieldPermission(String roleCode, DataPermissionType permissionType,
                                                                String menuCode, String operationCode, String businessObjectCode) {
        switch (permissionType) {
            case MENU_OPERATION_RULE:
                Assert.hasText(menuCode, "菜单编码为空");
                Assert.hasText(operationCode, "操作编码为空");
                return roleFieldPermissionRepoProc.listByRoleForMenuOperation(roleCode, menuCode, operationCode);
            case BUSINESS_OPERATION_RULE:
                Assert.hasText(operationCode, "操作编码为空");
                return roleFieldPermissionRepoProc.listByRoleForBusinessOperation(roleCode, operationCode);
            case BUSINESS_OBJECT_RULE:
                Assert.hasText(businessObjectCode, "业务对象编码为空");
                return roleFieldPermissionRepoProc.listByRoleForBusinessObject(roleCode, businessObjectCode);
            case BUSINESS_RESOURCE_RULE:
                Assert.hasText(businessObjectCode, "业务对象编码为空");
                return roleFieldPermissionRepoProc.listByRoleForBusinessObjectResource(roleCode, businessObjectCode);
            default:
                throw new BusinessException("查询权限失败，暂不支持的数据权限类型");
        }
    }

    private List<SysDprApiFieldRespVO> queryReqArguments(String apiCode) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> apiParameterRepoProc.queryInParamByApiCode(apiCode)).stream()
                .map(t -> {
                    SysDprApiFieldRespVO respVO = new SysDprApiFieldRespVO();
                    respVO.setFieldName(t.getCode());
                    respVO.setFieldRemark(t.getName());
                    respVO.setEnabled(false);

                    return respVO;
                }).collect(Collectors.toList());
    }

    private List<SysDprApiFieldRespVO> queryRoleApiFields(Long roleId, String menuCode, String apiCode) {
        var roleFieldMap = apiFieldsRepoProc.queryByApi(roleId, menuCode, apiCode).stream()
                .collect(Collectors.toMap(PermissionParameterBO::getFieldName, t -> t, (t1, t2) -> t1));

        // 查询出参
        return tenantDataIsolateProvider.byDefaultDirectly(() -> apiParameterRepoProc.queryOutParamByApiCode(apiCode)).stream()
                .map(t -> {
                    SysDprApiFieldRespVO respVO = new SysDprApiFieldRespVO();
                    respVO.setFieldName(t.getCode());
                    respVO.setFieldRemark(t.getName());

                    var roleField = roleFieldMap.get(t.getCode());
                    if (roleField == null) {
                        // 未设置
                        respVO.setFieldApiVisible(false);
                        respVO.setFieldFormVisible(false);
                        respVO.setFieldFormUpdate(false);
                        respVO.setEnabled(false);
                        return respVO;
                    }
                    respVO.setId(roleField.getId());
                    respVO.setFieldApiVisible(Boolean.TRUE.equals(roleField.getFieldApiVisible()));
                    respVO.setFieldFormVisible(Boolean.TRUE.equals(roleField.getFieldFormVisible()));
                    respVO.setFieldFormUpdate(Boolean.TRUE.equals(roleField.getFieldFormUpdate()));
                    respVO.setEnabled(true);

                    return respVO;
                }).collect(Collectors.toList());
    }

    private SysDprRespVO queryRoleCustomRule(Long roleId, String menuCode, String apiCode) {
        SysDprRespVO respVO = new SysDprRespVO();

        var ruleDoList = roleApiRowRuleRepoProc.queryByApi(roleId, menuCode, apiCode);
        if (ruleDoList.isEmpty()) {
            // 尚未配置
            respVO.setCustomRuleList(Collections.emptyList());
            return respVO;
        }

        List<SysDprApiCustomRuleRespVO> customRuleRespVoList = new ArrayList<>(ruleDoList.size());
        respVO.setCustomRuleList(customRuleRespVoList);
        for (SysDprRoleApiRowRuleDO ruleDO : ruleDoList) {
            if (Boolean.TRUE.equals(ruleDO.getDataRange())) {
                // 数据范围
                respVO.setRange(SysDprValueType.valueOf(ruleDO.getDprSysInternally()).code());
                continue;
            }
            SysDprApiCustomRuleRespVO vo = new SysDprApiCustomRuleRespVO();
            vo.setId(ruleDO.getId());
            vo.setDprRuleName(ruleDO.getDprRuleName());
            vo.setDprRuleField(ruleDO.getDprRuleField());
            if (StringUtils.hasText(ruleDO.getDprRuleCondition())) {
                vo.setDprRuleCondition(DprRuleConditionEnum.valueOf(ruleDO.getDprRuleCondition()));
            }
            if (StringUtils.hasText(ruleDO.getDprRuleValueType())) {
                vo.setDprRuleValueType(DprRuleValueTypeEnum.valueOf(ruleDO.getDprRuleValueType()));
            }
            vo.setDataSet(ruleDO.getDataSet());
            vo.setDprRuleValue(ruleDO.getDprRuleValue());
            if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS == vo.getDprRuleValueType() && StringUtils.hasText(ruleDO.getDprRuleValue())) {
                vo.setDprRuleValueName(DprValueResolverFactory.convertDprValueTypeName(ruleDO.getDprRuleValue()));
            }
            if (StringUtils.hasText(ruleDO.getDprRuleValueName())) {
                vo.setDprRuleValueName(ruleDO.getDprRuleValueName());
            }
            vo.setDprRuleDeclare(ruleDO.getDprRuleDeclare());
            vo.setBs1(ruleDO.getBs1());
            vo.setBs2(ruleDO.getBs2());
            vo.setBs3(ruleDO.getBs3());
            customRuleRespVoList.add(vo);
        }

        return respVO;
    }

    private PermissionDetailRespVO detailForCustom(String code, Long tenantId) {
        var menuTree = tenantMenuTreeRepoProc.getByMenuCode(code);
        if (menuTree == null) {
            return this.detailForDefault(code, tenantId);
        }

        PermissionDetailRespVO respVO = new PermissionDetailRespVO();
        respVO.setId(menuTree.getId());
        respVO.setCode(menuTree.getMenuCode());
        respVO.setName(menuTree.getMenuName());
        respVO.setNodeType(menuTree.getNodeType());
        respVO.setParentCode(menuTree.getParentMenuCode());

        if (StringUtils.hasText(menuTree.getParentMenuCode())) {
            // 有父节点
            var menuTreeParent = tenantMenuTreeRepoProc.getByMenuCode(menuTree.getParentMenuCode());
            if (menuTreeParent != null) {
                respVO.setParentId(menuTreeParent.getId());
                respVO.setParentName(menuTreeParent.getMenuName());
            }
        }

        if (StrUtil.isBlank(menuTree.getMenuName())) {
            // 获取名称
            var menuName = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuNameByMenuCode(code));
            respVO.setName(menuName);
            respVO.setApiName(menuName);
        }
        if (StrUtil.isBlank(respVO.getParentName()) && StrUtil.isNotBlank(respVO.getParentCode())) {
            var menuName = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuNameByMenuCode(respVO.getParentCode()));
            respVO.setParentName(menuName);
        }

        return respVO;
    }

    private PermissionDetailRespVO detailForDefault(String code, Long tenantId) {
        var menuBO = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getByMenuCode(code));
        if (menuBO == null) {
            // 没有找到菜单，则尝试获取API
            return this.detailForApi(code);
        }

        PermissionDetailRespVO respVO = new PermissionDetailRespVO();
        respVO.setId(menuBO.getId());
        respVO.setCode(menuBO.getMenusCode());
        respVO.setName(menuBO.getMenusName());

        // 节点类型
        var nodeType = MenuTreeNodeType.valueOfPlatformMenu(menuBO.getNodeType());
        Assert.notNull(nodeType, "不支持的节点类型");
        respVO.setNodeType(nodeType.getValue());

        respVO.setParentCode(menuBO.getMenusParentCode());
        respVO.setAppCode(menuBO.getMenusAppCode());
        respVO.setAction(menuBO.getMenusRoute());

        if (StrUtil.isNotBlank(menuBO.getMenusParentCode())) {
            // 父菜单
            var menuParentBO = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getByMenuCode(menuBO.getMenusParentCode()));
            if (menuParentBO != null) {
                respVO.setParentId(menuParentBO.getId());
                respVO.setParentCode(menuParentBO.getMenusCode());
                respVO.setParentName(menuParentBO.getMenusName());

                var parentNodeType = MenuTreeNodeType.valueOfPlatformMenu(menuParentBO.getNodeType());
                if (parentNodeType == MenuTreeNodeType.MENU || parentNodeType == MenuTreeNodeType.MENU_GROUP) {
                    respVO.setMenuCode(menuParentBO.getMenusCode());
                    respVO.setMenuName(menuParentBO.getMenusName());
                }
            }
        }

        return respVO;
    }

    private PermissionDetailRespVO detailForApi(String code) {
        var apiDo = tenantDataIsolateProvider.byDefaultDirectly(() -> apiRepoProc.getByCode(code));
        if (apiDo == null) {
            return null;
        }

        PermissionDetailRespVO respVO = new PermissionDetailRespVO();
        respVO.setId(apiDo.getId());
        respVO.setCode(apiDo.getPermissonCode());
        respVO.setName(apiDo.getPermissonName());
        respVO.setNodeType(MenuTreeNodeType.API.getValue());

        // 获取菜单
        respVO.setParentCode(apiDo.getMenusCode());
        if (StringUtils.hasText(apiDo.getMenusCode())) {
            var menuBO = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getByMenuCode(apiDo.getMenusCode()));
            if (menuBO != null) {
                respVO.setParentId(menuBO.getId());
                respVO.setParentName(menuBO.getMenusName());

                respVO.setMenuCode(menuBO.getMenusCode());
                respVO.setMenuName(menuBO.getMenusName());
            }
        }

        respVO.setAppCode(apiDo.getAppCode());
        respVO.setAction(apiDo.getPermissonPath());

        return respVO;
    }

    private PermissionDetailRespVO detailForApp(String code) {
        var appDO = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getByCode(code));
        if (appDO == null) {
            return null;
        }
        PermissionDetailRespVO respVO = new PermissionDetailRespVO();
        respVO.setId(appDO.getId());
        respVO.setCode(appDO.getAppCode());
        respVO.setName(appDO.getAppName());
        respVO.setNodeType(MenuTreeNodeType.APP.getValue());

        return respVO;
    }

    private void fillApp(PermissionDetailRespVO respVO) {
        String appCode = respVO.getAppCode();
        // 查询应用编码
        if (StrUtil.isBlank(appCode)) {
            appCode = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuNameByMenuCode(respVO.getCode()));
            if (StrUtil.isBlank(appCode)) {
                return;
            }
            respVO.setAppCode(appCode);
        }

        // 查询应用
        var app = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getByCode(respVO.getAppCode()));
        if (app == null) {
            return;
        }
        respVO.setAppId(app.getId());
        respVO.setAppName(app.getAppName());
    }

    private void fillMenu(PermissionDetailRespVO respVO) {
        if (StrUtil.isNotBlank(respVO.getMenuCode())) {
            return;
        }

        String menuCode = respVO.getParentCode();
        if (StrUtil.isBlank(menuCode)) {
            return;
        }

        // 查询菜单
        var menu = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            var temp = menusRepoProc.getByMenuCode(menuCode);
            while (temp != null) {
                var tempType = MenuTreeNodeType.valueOfPlatformMenu(temp.getNodeType());
                if (tempType == MenuTreeNodeType.MENU || tempType == MenuTreeNodeType.MENU_GROUP) {
                    // 已是菜单
                    break;
                }
                if (StrUtil.isBlank(temp.getMenusParentCode())) {
                    // 已无父节点
                    temp = null;
                    break;
                }

                temp = menusRepoProc.getByMenuCode(temp.getMenusParentCode());
            }
            return temp;
        });
        if (menu == null) {
            return;
        }
        respVO.setMenuCode(menu.getMenusCode());
        respVO.setMenuName(menu.getMenusName());
    }
}
