package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.CloudtContextProperties;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.core.yst.common.YstConstant;
import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.common.TraceIdProvider;
import com.elitescloud.boot.support.app.CloudtAppHolder;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.SysThirdApiSystem;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.convert.ThirdApiLogConvert;
import com.elitescloud.cloudt.system.dto.ThirdApiLogDTO;
import com.elitescloud.cloudt.system.dto.ThirdApiRetryParamDTO;
import com.elitescloud.cloudt.system.model.entity.SysThirdApiBusinessDO;
import com.elitescloud.cloudt.system.model.entity.SysThirdApiLogDO;
import com.elitescloud.cloudt.system.model.entity.SysThirdApiRetryDO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.ThirdApiLogPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.ThirdApiLogSaveVO;
import com.elitescloud.cloudt.system.service.SysAlertService;
import com.elitescloud.cloudt.system.service.ThirdApiLogService;
import com.elitescloud.cloudt.system.service.ThirdApiRetrySupportService;
import com.elitescloud.cloudt.system.service.repo.ThirdApiBusinessRepoProc;
import com.elitescloud.cloudt.system.service.repo.ThirdApiLogRepoProc;
import com.elitescloud.cloudt.system.service.repo.ThirdApiRetryRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * 三方接口重试服务.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/11
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class ThirdApiLogRetrySupportServiceImpl extends BaseServiceImpl implements ThirdApiRetrySupportService {

    @Autowired
    private ThirdApiLogRepoProc repoProc;
    @Autowired
    private ThirdApiRetryRepoProc retryRepoProc;
    @Autowired
    private ThirdApiBusinessRepoProc businessRepoProc;
    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private CloudtContextProperties contextProperties;
    @Autowired(required = false)
    private ThirdApiRetry thirdApiRetry;
    @Autowired
    private LogProperties logProperties;
    @Autowired
    private List<TraceIdProvider> providers;
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private UdcProvider userProvider;
    @Autowired
    private SysAlertService alertService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveLog(ThirdApiLogDTO logDTO) {
        log.info("保存推送日志：{}", JSONUtil.toJsonString(logDTO));
        if (logDTO.getOriginalRetryId() != null) {
            // 更新重试结果
            updateRetryRespResult(logDTO.getOriginalRetryId(), false, Boolean.TRUE.equals(logDTO.getRespSuccess()),
                    CharSequenceUtil.blankToDefault(logDTO.getRespFailMsg(), logDTO.getReqFailMsg()), logDTO.getRespBody());
            return ApiResult.ok(logDTO.getOriginalRetryId());
        }

        var logDO = this.dto2Do(logDTO);
        repoProc.save(logDO);

        // 添加业务信息
        this.saveBusinessInfo(logDO);

        // 添加重试
        if (!logDO.getRespSuccess() && logDO.getNeedRetry() && Boolean.TRUE.equals(logDO.getRestful())) {
            if (Boolean.TRUE.equals(systemProperties.getThirdApiLog().getEnabledRetry())) {
                this.addRetry(logDO);
            } else {
                repoProc.updateRetryFailResult(logDO.getId(), "未启用自动重试");
            }
        }

        // 添加预警
        if (!logDO.getRespSuccess()) {
            this.sendAlert(logDO, CharSequenceUtil.blankToDefault(logDO.getRespFailMsg(), logDO.getReqFailMsg()));
        }

        return ApiResult.ok(logDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveLog(ThirdApiLogSaveVO saveVO, boolean retrySend) {
        Assert.notNull(saveVO.getId(), "ID为空");
        var original = repoProc.get(saveVO.getId());
        if (original.getOriginalId() != null) {
            original = repoProc.get(original.getOriginalId());
        }
        Assert.notNull(original, "修改的数据不存在");

        // 保存原始记录
        ThirdApiRetryParamDTO retryParamDTO = new ThirdApiRetryParamDTO();
        retryParamDTO.setServerAddr(saveVO.getServerAddr());
        retryParamDTO.setUri(saveVO.getUri());
        retryParamDTO.setReqMethod(saveVO.getReqMethod());
        retryParamDTO.setReqQueryParams(JSONUtil.json2Obj(saveVO.getReqQueryParamsJson(), true, () -> "查询参数格式不正确，请使用JSON格式"));
        retryParamDTO.setReqBody(saveVO.getReqBody());
        retryParamDTO.setReqHeaders(JSONUtil.json2Obj(saveVO.getReqHeadersJson(), true, () -> "请求头格式不正确，请使用JSON格式"));

        original.setRetryParamJson(super.obj2Json(retryParamDTO));
        repoProc.save(original);

        // 重试发送
        if (retrySend) {
            this.manualSaveAndRetry(saveVO, original);
        }

        return ApiResult.ok(saveVO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> trySend(Long id, Integer version) {
        var originalId = repoProc.getOriginalLogId(id);
        if (originalId == null) {
            log.info("未查询到原始记录：{}", id);
            return ApiResult.ok(false);
        }
        var retryVersion = retryRepoProc.getVersionByLastLogId(originalId);

        var canSend = retryVersion != null && retryVersion.intValue() == ObjectUtil.defaultIfNull(version, 0);
        if (canSend) {
            // 更新消息的发送时间
            repoProc.updateReqTime(id, LocalDateTime.now());
        }
        return ApiResult.ok(canSend);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteRetry(Long id, String reason) {
        var originalId = repoProc.getOriginalLogId(id);
        if (originalId == null) {
            log.error("未查询到原始记录：{}", id);
            return ApiResult.fail("未查询到原始记录");
        }

        // 删除重试消息
        retryRepoProc.deleteByLogId(originalId);
        repoProc.delete(id);

        repoProc.updateRetryFailResult(originalId, reason);

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateRetryResult(Long id, boolean success, String reason) {
        if (id == null) {
            log.error("更新重试结果失败，ID为空");
            return ApiResult.fail("ID为空");
        }
        if (success) {
            // 成功的在调用后就已更新
            return ApiResult.ok(true);
        }

        // 更新发送结果
        repoProc.updateReqResult(id, success, reason);
        var originalLogId = repoProc.getOriginalLogId(id);

        if (originalLogId == null) {
            originalLogId = id;
        }
        repoProc.updateRetryTimes(originalLogId);
        this.addRetry(repoProc.get(originalLogId));

        // 添加预警
        this.sendAlert(repoProc.get(id), reason);

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateRetryRespResult(Long id, boolean manual, boolean success, String reason, String respBody) {
        var apiLog = repoProc.get(id);
        if (apiLog == null) {
            return ApiResult.fail("记录不存在");
        }
        var originalId = apiLog.getOriginalId();
        if (success) {
            if (originalId != null) {
                // 删除重试记录
                retryRepoProc.deleteByLogId(originalId);
            }
            if (CharSequenceUtil.isAllNotBlank(apiLog.getBusinessType(), apiLog.getBusinessKey())) {
                businessRepoProc.updateSuccess(apiLog.getBusinessType(), apiLog.getBusinessKey(), true);
            }
        }

        repoProc.updateRespResult(id, success, reason, respBody);
        if (originalId != null) {
            repoProc.updateRespResult(originalId, success, reason, respBody);
        }

        if (!success && !manual && originalId != null) {
            // 失败后，重试
            var original = ObjUtil.defaultIfNull(repoProc.get(originalId), apiLog);
            this.addRetry(original);
        }

        // 添加预警
        if (!success) {
            this.sendAlert(apiLog, reason);
        }
        return ApiResult.ok(true);
    }

    private String convertBusinessTypeName(String businessType) {
        if (CharSequenceUtil.isBlank(businessType)) {
            return null;
        }
        var types = SpringContextHolder.getBean(ThirdApiLogService.class).listBusinessTypes().computeData();
        for (CodeNameParam type : types) {
            if (businessType.equals(type.getCode()) || businessType.equals(type.getName())) {
                return type.getName();
            }
        }
        return businessType;
    }

    private String convertSystemName(String thirdApp, boolean isSource) {
        if (CharSequenceUtil.isBlank(thirdApp)) {
            return null;
        }

        var types = isSource ? SpringContextHolder.getBean(ThirdApiLogService.class).listSourceSystems().computeData() :
                SpringContextHolder.getBean(ThirdApiLogService.class).listTargetSystems().computeData();
        for (CodeNameParam type : types) {
            if (thirdApp.equals(type.getCode()) || thirdApp.equals(type.getName())) {
                return type.getName();
            }
        }
        return thirdApp;
    }

    private ThirdApiLogPageRespVO do2PageRespVO(SysThirdApiLogDO logDO) {
        ThirdApiLogPageRespVO respVO = new ThirdApiLogPageRespVO();
        respVO.setId(logDO.getId());
        respVO.setAppCode(logDO.getAppCode());

        if (Boolean.TRUE.equals(logDO.getServer())) {
            // 被第三方调用时
            respVO.setSourceSystem(this.convertSystemName(logDO.getThirdApp(), true));
            respVO.setTargetSystem(SysThirdApiSystem.YST.getValue());
        } else {
            // 调用第三方时
            respVO.setSourceSystem(SysThirdApiSystem.YST.getValue());
            respVO.setTargetSystem(this.convertSystemName(logDO.getThirdApp(), false));
        }

        respVO.setServerAddr(logDO.getServerAddr());
        respVO.setBusinessType(this.convertBusinessTypeName(logDO.getBusinessType()));
        respVO.setBusinessKey(logDO.getBusinessKey());
        respVO.setUsername(logDO.getUsername());
        respVO.setUri(logDO.getUri());
        respVO.setReqMethod(logDO.getReqMethod());
        respVO.setReqTime(logDO.getReqTime());
        respVO.setRespTime(logDO.getRespTime());
        respVO.setRetryTimes(logDO.getRetryTimes());
        respVO.setRespSuccess(logDO.getRespSuccess());
        respVO.setFailReason(CharSequenceUtil.blankToDefault(logDO.getRetryFailReason(),
                CharSequenceUtil.blankToDefault(logDO.getRespFailMsg(), logDO.getReqFailMsg())));

        return respVO;
    }

    private void saveBusinessInfo(SysThirdApiLogDO logDO) {
        if (CharSequenceUtil.hasBlank(logDO.getBusinessType(), logDO.getBusinessKey())) {
            return;
        }

        var success = Boolean.TRUE.equals(logDO.getRespSuccess());
        var existsId = businessRepoProc.getId(logDO.getBusinessType(), logDO.getBusinessKey());
        if (existsId == null) {
            SysThirdApiBusinessDO businessDO = new SysThirdApiBusinessDO();
            businessDO.setSysTenantId(logDO.getSysTenantId());
            businessDO.setBusinessType(logDO.getBusinessType());
            businessDO.setBusinessKey(logDO.getBusinessKey());
            businessDO.setSuccess(success);

            businessRepoProc.save(businessDO);
            return;
        }

        businessRepoProc.updateSuccess(existsId, success);
    }

    private void addRetry(SysThirdApiLogDO originalLog) {
        // 添加重试记录
        var retryLog = this.addRetryLog(originalLog);
        // 添加重试
        var retryInfo = this.upsertRetryRecord(retryLog);

        // 添加重试
        this.executeRetry(retryInfo, retryLog);
    }

    private void executeRetry(SysThirdApiRetryDO retryDO, SysThirdApiLogDO retryLog) {
        ThirdApiRetry.RetryParam retryParam = new ThirdApiRetry.RetryParam();
        retryParam.setTaskId(retryLog.getId().toString());
        retryParam.setVersion(retryDO.getAuditDataVersion());
        retryParam.setRetryTimes(retryDO.getRetryTimes());
        retryParam.setRetryTime(retryDO.getSendTimeNext());

        taskExecutor.execute(() -> thirdApiRetry.addRetryTask(retryParam));
    }

    private SysThirdApiRetryDO upsertRetryRecord(SysThirdApiLogDO logDO) {
        var logId = ObjectUtil.defaultIfNull(logDO.getOriginalId(), logDO.getId());

        var retryDO = retryRepoProc.getByRecordLogId(logId);
        if (retryDO != null) {
            // 查询最后一次请求时间
            var lastReqTime = ObjectUtil.defaultIfNull(repoProc.getLastReqTime(logId), retryDO.getSendTimeNext());

            retryDO.setRetryTimes(retryDO.getRetryTimes() + 1);
            retryDO.setSendTime(ObjectUtil.defaultIfNull(lastReqTime, () -> LocalDateTime.now()));
            retryDO.setSendTimeNext(thirdApiRetry.generateNextRetryTime(retryDO.getSendTime(), retryDO.getRetryTimes()));
            retryDO.setAuditDataVersion(retryDO.getAuditDataVersion() + 1);
            retryRepoProc.save(retryDO);
            return retryDO;
        }

        retryDO = new SysThirdApiRetryDO();
        retryDO.setSysTenantId(logDO.getSysTenantId());
        retryDO.setRecordLogId(logId);
        retryDO.setRetryTimes(0);
        retryDO.setSendTime(ObjectUtil.defaultIfNull(logDO.getReqTime(), () -> LocalDateTime.now()));
        retryDO.setSendTimeNext(thirdApiRetry.generateNextRetryTime(retryDO.getSendTime(), retryDO.getRetryTimes()));
        retryDO.setAuditDataVersion(0);

        retryRepoProc.save(retryDO);
        return retryDO;
    }

    private SysThirdApiLogDO addRetryLog(SysThirdApiLogDO original) {
        SysThirdApiLogDO logDO = new SysThirdApiLogDO();
        logDO.setSysTenantId(original.getSysTenantId());
        logDO.setAppCode(original.getAppCode());
        logDO.setRestful(original.getRestful());
        logDO.setServer(original.getServer());
        logDO.setServerAddr(original.getServerAddr());
        logDO.setThirdApp(original.getThirdApp());
        logDO.setBusinessType(original.getBusinessType());
        logDO.setBusinessKey(original.getBusinessKey());
        logDO.setClientId(original.getClientId());
        logDO.setUserId(original.getUserId());
        logDO.setUsername(original.getUsername());
        logDO.setUri(original.getUri());
        logDO.setReqMethod(original.getReqMethod());
        logDO.setReqQueryParamsJson(original.getReqQueryParamsJson());
        logDO.setReqBody(original.getReqBody());
        logDO.setReqHeadersJson(original.getReqHeadersJson());
        logDO.setReqSuccess(original.getReqSuccess());
//        logDO.setReqFailMsg(original.getReqFailMsg());
//        logDO.setReqTime();
        logDO.setReqIp(CloudtAppHolder.getServerIp());
//        logDO.setRespBody();
//        logDO.setRespSuccess();
//        logDO.setRespFailMsg();
//        logDO.setRespTime();
        logDO.setNeedRetry(true);
        logDO.setDetectedOperatorAuth(original.getDetectedOperatorAuth());
        logDO.setRetryTimes(0);
//        logDO.setRetryFailReason();
        logDO.setRetried(true);
        logDO.setManualRetry(false);
        logDO.setOriginalId(original.getId());
        logDO.setRetryParamJson(original.getRetryParamJson());

        repoProc.save(logDO);
        repoProc.updateLastRetryLogId(original.getId(), logDO.getId());

        return logDO;
    }

    private SysThirdApiLogDO manualSaveAndRetry(ThirdApiLogSaveVO saveVO, SysThirdApiLogDO original) {
        Assert.isTrue(Boolean.TRUE.equals(original.getRestful()), "仅Restful格式接口支持重试");
        Assert.isTrue(Boolean.FALSE.equals(original.getRetried()), "请编辑原纪录");

        SysThirdApiLogDO logDO = new SysThirdApiLogDO();
        logDO.setSysTenantId(original.getSysTenantId());
        logDO.setAppCode(original.getAppCode());
        logDO.setRestful(original.getRestful());
        logDO.setServer(original.getServer());
        logDO.setServerAddr(saveVO.getServerAddr());
        logDO.setThirdApp(original.getThirdApp());
        logDO.setBusinessType(original.getBusinessType());
        logDO.setBusinessKey(original.getBusinessKey());
        logDO.setClientId(original.getClientId());
        var user = super.currentUser(true);
        logDO.setUserId(user.getUserId());
        logDO.setUsername(CharSequenceUtil.blankToDefault(user.getUser().getPrettyName(), user.getUsername()));
        logDO.setUri(cn.hutool.core.lang.Assert.notBlank(saveVO.getUri(), "请求的接口地址为空"));
        logDO.setReqMethod(cn.hutool.core.lang.Assert.notNull(saveVO.getReqMethod(), "请求方式为空"));
        logDO.setReqQueryParamsJson(saveVO.getReqQueryParamsJson());
        logDO.setReqBody(saveVO.getReqBody());
        logDO.setReqHeadersJson(saveVO.getReqHeadersJson());
        logDO.setReqSuccess(false);
//        logDO.setReqFailMsg();
        logDO.setReqTime(LocalDateTime.now());
        logDO.setReqIp(CloudtAppHolder.getServerIp());
//        logDO.setRespBody();
//        logDO.setRespSuccess();
//        logDO.setRespFailMsg();
//        logDO.setRespTime();
        logDO.setNeedRetry(false);
        logDO.setDetectedOperatorAuth(original.getDetectedOperatorAuth());
//        logDO.setRetryTimes();
//        logDO.setRetryFailReason();
        logDO.setRetried(true);
        logDO.setManualRetry(true);
        logDO.setOriginalId(original.getId());
        logDO.setRetryParamJson(original.getRetryParamJson());
//        logDO.setLastRetryId();
        repoProc.save(logDO);
        repoProc.updateLastRetryLogId(original.getId(), logDO.getId());

        // 执行重试
        CompletableFuture.runAsync(() -> thirdApiRetry.executeByManual(logDO), taskExecutor)
                .whenComplete((r, e) -> {
                    if (e == null) {
                        log.warn("{}手动重试成功：", logDO.getId());
                        return;
                    }
                    log.warn("{}手动重试失败：", logDO.getId(), e);

                    try {
                        SpringContextHolder.getBean(ThirdApiRetryTaskProvider.class).updateRetryResult(logDO.getId().toString(), false, "重试失败，请检查请求参数");
                    } catch (Throwable ex) {
                        log.error("更新手动重试的结果异常：", ex);
                    }
                });

        return logDO;
    }

    private SysThirdApiLogDO dto2Do(ThirdApiLogDTO dto) {
        SysThirdApiLogDO logDO = ThirdApiLogConvert.INSTANCE.dto2Do(dto);
        ObjUtil.ifNull(logDO.getBusinessKey(), "", logDO::setBusinessKey);
        ObjUtil.ifNull(logDO.getBusinessType(), "", logDO::setBusinessType);
        ObjUtil.ifNull(logDO.getThirdApp(), "", logDO::setThirdApp);

        logDO.setSysTenantId(super.currentTenantId());
        logDO.setServer(ObjectUtil.defaultIfNull(dto.getServer(), false));
        logDO.setReqTime(ObjectUtil.defaultIfNull(dto.getReqTime(), () -> LocalDateTime.now()));
        logDO.setReqSuccess(ObjectUtil.defaultIfNull(dto.getReqSuccess(), false));
        logDO.setRespSuccess(ObjectUtil.defaultIfNull(dto.getRespSuccess(), false));

        if (!logDO.getServer()) {
            Assert.hasText(logDO.getServerAddr(), "服务端地址为空");
        }

        logDO.setReqHeadersJson(super.obj2Json(dto.getReqHeaders()));
        logDO.setReqQueryParamsJson(super.obj2Json(dto.getReqQueryParams()));

        logDO.setNeedRetry(ObjectUtil.defaultIfNull(dto.getNeedRetry(), false));
        logDO.setDetectedOperatorAuth(ObjectUtil.defaultIfNull(dto.getDetectedOperatorAuth(), false));
        if (dto.getRetryParam() != null) {
            logDO.setRetryParamJson(super.obj2Json(dto.getRetryParam()));
        }
        logDO.setRetryTimes(0);
        logDO.setRetried(false);
        logDO.setManualRetry(false);

        return logDO;
    }

    private void sendAlert(SysThirdApiLogDO logDO, String failMsg) {
        if (logDO == null) {
            log.info("发送预警失败，日志信息为空");
            return;
        }
        String traceId = getTraceId();
        CompletableFuture.supplyAsync(() -> {
                    var respVO = do2PageRespVO(logDO);

                    Map<String, Object> tmplParams = new HashMap<>(32);
                    tmplParams.put("businessType", convertBusinessTypeName(respVO.getBusinessType()));
                    tmplParams.put("direction", Boolean.TRUE.equals(logDO.getServer()) ? "接收" : "推送");
                    tmplParams.put("businessKey", respVO.getBusinessKey());
                    tmplParams.put("reqTime", formatTime(respVO.getReqTime()));
                    tmplParams.put("respTime", formatTime(respVO.getRespTime()));

                    tmplParams.put("sourceApp", convertSystemName(respVO.getSourceSystem(), true));
                    tmplParams.put("targetApp", convertSystemName(respVO.getTargetSystem(), false));
                    tmplParams.put("traceId", traceId);
                    tmplParams.put("url", respVO.getUri());
                    tmplParams.put("failMsg", failMsg);

                    return tenantDataIsolateProvider.byTenantDirectly(() -> alertService.sendAlertByTmpl(SysAlertService.BUSINESS_TYPE_THIRD_API,
                                    logDO.getThirdApp(), tmplParams),
                            logDO.getSysTenantId());
                }, taskExecutor)
                .whenComplete((r, e) -> {
                    if (e == null) {
                        log.info("发送预警结果：{}", JSONUtil.toJsonString(r));
                        return;
                    }
                    log.info("发送预警异常：", e);
                });
    }

    private String getTraceId() {
        if (CharSequenceUtil.isBlank(logProperties.getTrace().getTraceCode())) {
            return null;
        }
        for (TraceIdProvider provider : providers) {
            if (logProperties.getTrace().getTraceCode().equals(provider.code())) {
                return provider.getTraceId();
            }
        }
        return null;
    }

    private String formatTime(LocalDateTime dateTime) {
        if (dateTime == null) {
            return "";
        }

        return DatetimeUtil.FORMATTER_DATETIME.format(dateTime);
    }
}
