package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjUtil;
import com.elitescloud.boot.CloudtContextProperties;
import com.elitescloud.boot.common.BaseUdc;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.core.yst.common.YstConstant;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.SysThirdApiBusinessType;
import com.elitescloud.cloudt.constant.SysThirdApiSystem;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.convert.ThirdApiLogConvert;
import com.elitescloud.cloudt.system.dto.ThirdApiLogDTO;
import com.elitescloud.cloudt.system.dto.ThirdApiRetryParamDTO;
import com.elitescloud.cloudt.system.dto.req.ThirdApiLogQueryDTO;
import com.elitescloud.cloudt.system.model.entity.SysThirdApiLogDO;
import com.elitescloud.cloudt.system.model.vo.query.extend.ThirdApiLogQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.ThirdApiRetryLogQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.ThirdApiLogAggPageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.ThirdApiLogDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.ThirdApiLogPageRespVO;
import com.elitescloud.cloudt.system.service.ThirdApiLogService;
import com.elitescloud.cloudt.system.service.repo.ThirdApiLogRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/4
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class ThirdApiLogServiceImpl extends BaseServiceImpl implements ThirdApiLogService {

    @Autowired
    private ThirdApiLogRepoProc repoProc;
    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private CloudtContextProperties contextProperties;

    @Override
    public ApiResult<PagingVO<ThirdApiLogDTO>> queryByPage(ThirdApiLogQueryDTO requestDTO) {
        var tenantId = super.currentTenantId();
        var pageData = repoProc.queryByPage(tenantId, requestDTO).map(t -> {
            var dto = ThirdApiLogConvert.INSTANCE.do2DTO(t);
            if (StringUtils.hasText(t.getReqHeadersJson())) {
                dto.setReqHeaders(super.json2Obj(t.getReqHeadersJson(), new TypeReference<>() {
                }));
            }
            if (StringUtils.hasText(t.getReqQueryParamsJson())) {
                dto.setReqQueryParams(super.json2Obj(t.getReqQueryParamsJson(), new TypeReference<>() {
                }));
            }

            return dto;
        });
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<PagingVO<ThirdApiLogPageRespVO>> queryByPage(ThirdApiLogQueryVO queryVO) {
        var queryParam = this.convertPagedQueryParam(queryVO);
        if (queryParam == null) {
            return ApiResult.ok(PagingVO.empty());
        }
        var pageData = repoProc.queryByPage(queryParam.getQueryVO(), queryParam.getTenantId(), queryParam.getServer(), queryParam.getThirdApp())
                .map(this::do2PageRespVO);
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<List<ThirdApiLogPageRespVO>> listRetryRecord(Long originalId) {
        if (originalId == null) {
            return ApiResult.fail("记录ID为空");
        }

        var listData = repoProc.listRetryRecord(originalId).stream()
                .filter(t -> t.getReqTime() != null)
                .sorted(Comparator.comparing(SysThirdApiLogDO::getReqTime, Comparator.nullsLast(LocalDateTime::compareTo)).reversed())
                .map(this::do2PageRespVO)
                .collect(Collectors.toList());
        return ApiResult.ok(listData);
    }

    @Override
    public ApiResult<PagingVO<ThirdApiLogPageRespVO>> pageRetryRecord(ThirdApiRetryLogQueryVO queryVO) {
        if (queryVO.getId() == null) {
            return ApiResult.fail("记录ID为空");
        }

        var listData = repoProc.pageRetryRecord(queryVO.getId(), false, queryVO.getPageRequest())
                .map(this::do2PageRespVO);
        return ApiResult.ok(listData);
    }

    @Override
    public ApiResult<PagingVO<ThirdApiLogAggPageRespVO>> queryAggByPage(ThirdApiLogQueryVO queryVO) {
        var queryParam = this.convertPagedQueryParam(queryVO);
        if (queryParam == null) {
            return ApiResult.ok(PagingVO.empty());
        }

        // 默认查询天数
        var aggDays = systemProperties.getThirdApiLog().getAggDefaultDays();
        if (aggDays != null && aggDays > 0) {
            if (queryVO.getReqTimeStart() == null && queryVO.getReqTimeEnd() == null) {
                queryVO.setReqTimeEnd(LocalDateTime.now());
                queryVO.setReqTimeStart(queryVO.getReqTimeEnd().minusDays(aggDays));
            }
        }

        var pageData = repoProc.queryAggByPage(queryParam.getQueryVO(), queryParam.getTenantId(), queryParam.getServer(), queryParam.getThirdApp());
        pageData.each(this::convertPageRespVO);
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<List<ThirdApiLogPageRespVO>> listAggDetail(ThirdApiLogQueryVO queryVO, Long id) {
        var queryParam = this.convertPagedQueryParam(queryVO);
        if (queryParam == null) {
            return ApiResult.ok(Collections.emptyList());
        }

        SysThirdApiLogDO record = this.queryOriginalRecord(queryVO, id);
        if (record == null) {
            return ApiResult.fail("选择的记录不存在");
        }

        var detailList = repoProc.listDetailForAgg(queryVO, record).stream()
                .map(this::do2PageRespVO)
                .collect(Collectors.toList());
        return ApiResult.ok(detailList);
    }

    @Override
    public ApiResult<PagingVO<ThirdApiLogPageRespVO>> pageAggDetail(ThirdApiLogQueryVO queryVO, Long id) {
        var queryParam = this.convertPagedQueryParam(queryVO);
        if (queryParam == null) {
            return ApiResult.ok(PagingVO.empty());
        }

        SysThirdApiLogDO record = this.queryOriginalRecord(queryVO, id);
        if (record == null) {
            return ApiResult.fail("选择的记录不存在");
        }

        PagingVO<ThirdApiLogPageRespVO> detailList = repoProc.pageDetailForAgg(queryVO, record)
                .map(this::do2PageRespVO);
        return ApiResult.ok(detailList);
    }

    @Override
    public ApiResult<ThirdApiLogDetailRespVO> getDetail(Long id) {
        if (id == null) {
            return ApiResult.fail("记录ID为空");
        }

        return repoProc.getOptional(id)
                .map(t -> {
                    var respVO = ThirdApiLogConvert.INSTANCE.do2DetailRespVO(t);
                    respVO.setBusinessType(this.convertBusinessTypeName(t.getBusinessType()));
                    if (Boolean.TRUE.equals(t.getServer())) {
                        // 被第三方调用时
                        respVO.setSourceSystem(this.convertSystemName(t.getThirdApp(), true));
                        respVO.setTargetSystem(this.currentServerName());
                    } else {
                        // 调用第三方时
                        respVO.setSourceSystem(this.currentServerName());
                        respVO.setTargetSystem(this.convertSystemName(t.getThirdApp(), false));
                    }

                    ThirdApiLogDetailRespVO.RetryParamVO retryParamVO = new ThirdApiLogDetailRespVO.RetryParamVO();
                    respVO.setRetryParam(retryParamVO);
                    if (CharSequenceUtil.isBlank(t.getRetryParamJson())) {
                        retryParamVO.setServerAddr(t.getServerAddr());
                        retryParamVO.setUri(t.getUri());
                        retryParamVO.setReqMethod(t.getReqMethod());
                        retryParamVO.setReqQueryParamsJson(t.getReqQueryParamsJson());
                        retryParamVO.setReqBody(t.getReqBody());
                        retryParamVO.setReqHeadersJson(t.getReqHeadersJson());
                    } else {
                        var retryParams = JSONUtil.json2Obj(t.getRetryParamJson(), ThirdApiRetryParamDTO.class, true);
                        retryParamVO.setServerAddr(CharSequenceUtil.blankToDefault(retryParams.getServerAddr(), contextProperties.getServerAddr()));
                        retryParamVO.setUri(retryParams.getUri());
                        retryParamVO.setReqMethod(retryParams.getReqMethod());
                        retryParamVO.setReqQueryParamsJson(super.obj2Json(retryParams.getReqQueryParams()));
                        retryParamVO.setReqBody(retryParams.getReqBody());
                        retryParamVO.setReqHeadersJson(super.obj2Json(retryParams.getReqHeaders()));
                    }
                    return respVO;
                })
                .map(ApiResult::ok)
                .orElse(ApiResult.noData());
    }

    @Override
    public ApiResult<List<CodeNameParam>> listBusinessTypes() {
        var businessTypes = querySystemsByUdc(new SysThirdApiBusinessType());
        if (!businessTypes.isEmpty()) {
            return ApiResult.ok(businessTypes);
        }

        businessTypes = systemProperties.getThirdApiLog().getBusinessTypes();
        return ApiResult.ok(ObjUtil.defaultIfNull(businessTypes, Collections.emptyList()));
    }

    @Override
    public ApiResult<List<CodeNameParam>> listTargetSystems() {
        var targetSystems = querySystemsByUdc(SysThirdApiSystem.YST);
        if (!targetSystems.isEmpty()) {
            return ApiResult.ok(targetSystems);
        }

        targetSystems = systemProperties.getThirdApiLog().getTargetSystems();
        return ApiResult.ok(ObjUtil.defaultIfNull(targetSystems, Collections.emptyList()));
    }

    @Override
    public ApiResult<List<CodeNameParam>> listSourceSystems() {
        var sourceSystems = querySystemsByUdc(SysThirdApiSystem.YST);
        if (!sourceSystems.isEmpty()) {
            return ApiResult.ok(sourceSystems);
        }

        sourceSystems = systemProperties.getThirdApiLog().getSourceSystems();
        return ApiResult.ok(ObjUtil.defaultIfNull(sourceSystems, Collections.emptyList()));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> clearExpiredLogs(LocalDateTime expiredTime) {
        if (expiredTime == null) {
            return ApiResult.fail("过期时间为空");
        }

        repoProc.clearExpiredLogs(expiredTime);
        return ApiResult.ok(true);
    }

    private <T extends BaseUdc<T>> List<CodeNameParam> querySystemsByUdc(T udc) {
        var udcValueMap = super.udcMap(udc);
        if (CollUtil.isEmpty(udcValueMap)) {
            return Collections.emptyList();
        }

        List<CodeNameParam> udcValueList = new ArrayList<>(udcValueMap.size());
        for (Map.Entry<String, String> entry : udcValueMap.entrySet()) {
            udcValueList.add(new CodeNameParam(entry.getKey(), entry.getValue()));
        }

        return udcValueList;
    }

    private SysThirdApiLogDO queryOriginalRecord(ThirdApiLogQueryVO queryVO, Long id) {
        // 获取选择的记录
        var record = repoProc.get(id);
        if (record == null) {
            return null;
        }

        // 默认查询天数
        var aggDays = systemProperties.getThirdApiLog().getAggDefaultDays();
        if (aggDays != null && aggDays > 0) {
            if (queryVO.getReqTimeStart() == null && queryVO.getReqTimeEnd() == null) {
                queryVO.setReqTimeEnd(LocalDateTime.now());
                queryVO.setReqTimeStart(queryVO.getReqTimeEnd().minusDays(aggDays));
            }
        }

        return record;
    }

    private String convertBusinessTypeName(String businessType) {
        if (CharSequenceUtil.isBlank(businessType)) {
            return null;
        }
        var types = this.listBusinessTypes().computeData();
        for (CodeNameParam type : types) {
            if (businessType.equals(type.getCode()) || businessType.equals(type.getName())) {
                return type.getName();
            }
        }
        return businessType;
    }

    private String convertSystemName(String thirdApp, boolean isSource) {
        if (CharSequenceUtil.isBlank(thirdApp)) {
            return null;
        }
        var types = this.listSourceSystems().computeData();
        for (CodeNameParam type : types) {
            if (thirdApp.equals(type.getCode()) || thirdApp.equals(type.getName())) {
                return type.getName();
            }
        }
        return thirdApp;
    }

    private String convertSystem(String thirdAppName, boolean isSource) {
        if (CharSequenceUtil.isBlank(thirdAppName)) {
            return null;
        }
        var types = this.listSourceSystems().computeData();
        for (CodeNameParam type : types) {
            if (thirdAppName.equals(type.getName()) || thirdAppName.equals(type.getCode())) {
                return type.getCode();
            }
        }
        return thirdAppName;
    }

    private void convertPageRespVO(ThirdApiLogAggPageRespVO respVO) {
        if (Boolean.TRUE.equals(respVO.getServer())) {
            // 被第三方调用时
            respVO.setSourceSystem(this.convertSystemName(respVO.getThirdApp(), true));
            respVO.setTargetSystem(this.currentServerName());
        } else {
            // 调用第三方时
            respVO.setSourceSystem(this.currentServerName());
            respVO.setTargetSystem(this.convertSystemName(respVO.getThirdApp(), false));
        }

        respVO.setBusinessType(this.convertBusinessTypeName(respVO.getBusinessType()));
    }

    private ThirdApiLogPageRespVO do2PageRespVO(SysThirdApiLogDO logDO) {
        ThirdApiLogPageRespVO respVO = new ThirdApiLogPageRespVO();
        respVO.setId(logDO.getId());
        respVO.setAppCode(logDO.getAppCode());

        if (Boolean.TRUE.equals(logDO.getServer())) {
            // 被第三方调用时
            respVO.setSourceSystem(this.convertSystemName(logDO.getThirdApp(), true));
            respVO.setTargetSystem(this.currentServerName());
        } else {
            // 调用第三方时
            respVO.setSourceSystem(this.currentServerName());
            respVO.setTargetSystem(this.convertSystemName(logDO.getThirdApp(), false));
        }

        respVO.setServerAddr(logDO.getServerAddr());
        respVO.setBusinessType(this.convertBusinessTypeName(logDO.getBusinessType()));
        respVO.setBusinessKey(logDO.getBusinessKey());
        respVO.setUsername(logDO.getUsername());
        respVO.setUri(logDO.getUri());
        respVO.setReqMethod(logDO.getReqMethod());
        respVO.setReqTime(logDO.getReqTime());
        respVO.setRespTime(logDO.getRespTime());
        respVO.setRetryTimes(logDO.getRetryTimes());
        respVO.setRespSuccess(logDO.getRespSuccess());
        respVO.setFailReason(CharSequenceUtil.blankToDefault(logDO.getRetryFailReason(),
                CharSequenceUtil.blankToDefault(logDO.getRespFailMsg(), logDO.getReqFailMsg())));

        return respVO;
    }

    private String currentServerName() {
        String projectName = super.udcMap(SysThirdApiSystem.YST).get(YstConstant.projectCode);
        if (CharSequenceUtil.isNotBlank(projectName)) {
            return projectName;
        }
        return CharSequenceUtil.blankToDefault(contextProperties.getProjectName(), YstConstant.projectName);
    }

    private PagedQueryParam convertPagedQueryParam(ThirdApiLogQueryVO queryVO) {
        var tenant = super.currentTenant();
        var tenantId = tenant == null ? null : tenant.getId();

        Boolean server = null;
        String thirdApp = null;

        String currentProject = SysThirdApiSystem.YST.getValue();
        String currentProjectName = this.currentServerName();
        if (CharSequenceUtil.equals(queryVO.getSourceSystem(), currentProject) || CharSequenceUtil.equals(queryVO.getSourceSystem(), currentProjectName)) {
            server = false;
            thirdApp = this.convertSystem(queryVO.getTargetSystem(), false);
        } else if (CharSequenceUtil.equals(queryVO.getTargetSystem(), currentProject) || CharSequenceUtil.equals(queryVO.getTargetSystem(), currentProjectName)) {
            server = true;
            thirdApp = this.convertSystem(queryVO.getSourceSystem(), true);
        } else {
            if (CharSequenceUtil.isAllNotBlank(queryVO.getSourceSystem(), queryVO.getTargetSystem())) {
                return null;
            } else if (CharSequenceUtil.isNotBlank(queryVO.getSourceSystem())) {
                server = true;
            } else if (CharSequenceUtil.isNotBlank(queryVO.getTargetSystem())) {
                server = false;
            }
            thirdApp = CharSequenceUtil.blankToDefault(queryVO.getTargetSystem(), queryVO.getSourceSystem());
        }

        PagedQueryParam queryParam = new PagedQueryParam();
        queryParam.setQueryVO(queryVO);
        queryParam.setTenantId(tenantId);
        queryParam.setServer(server);
        queryParam.setThirdApp(thirdApp);

        return queryParam;
    }

    private static class PagedQueryParam {
        private ThirdApiLogQueryVO queryVO;
        private Long tenantId;
        private Boolean server;
        private String thirdApp;

        public ThirdApiLogQueryVO getQueryVO() {
            return queryVO;
        }

        public void setQueryVO(ThirdApiLogQueryVO queryVO) {
            this.queryVO = queryVO;
        }

        public Long getTenantId() {
            return tenantId;
        }

        public void setTenantId(Long tenantId) {
            this.tenantId = tenantId;
        }

        public Boolean getServer() {
            return server;
        }

        public void setServer(Boolean server) {
            this.server = server;
        }

        public String getThirdApp() {
            return thirdApp;
        }

        public void setThirdApp(String thirdApp) {
            this.thirdApp = thirdApp;
        }
    }
}
