package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.system.service.common.constant.SyncDataType;
import com.elitescloud.cloudt.system.service.convert.SysSyncConvert;
import com.elitescloud.cloudt.system.service.model.bo.SysSyncSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysSyncDO;
import com.elitescloud.cloudt.system.service.repo.SyncRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.util.function.Supplier;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/26
 */
@Component
public class SysSyncManager {
    @Autowired
    private SyncRepoProc syncRepoProc;

    /**
     * 保存同步信息
     *
     * @param saveBO 待同步信息
     * @return 同步记录ID
     */
    @Transactional(rollbackFor = Exception.class)
    public Long save(SysSyncSaveBO saveBO) {
        Assert.notNull(saveBO, "保存信息为空");

        var syncDO = this.saveBo2Do(saveBO);
        if (Boolean.FALSE.equals(syncDO.getSyncSuccess())) {
            syncDO.setOuterDataKey(null);
        }
        syncRepoProc.save(syncDO);

        return syncDO.getId();
    }

    /**
     * 更新同步结果
     *
     * @param id
     * @param outerDataKey
     * @param failReason
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateSyncResult(long id, boolean success, String outerDataKey, String failReason) {
        syncRepoProc.updateSyncResult(id, success, outerDataKey, failReason);
    }

    /**
     * 删除同步记录
     *
     * @param dataType 数据类型
     * @param dataKey  数据标识
     * @param outerApp 外部系统标识
     */
    @Transactional(rollbackFor = Exception.class)
    public void delete(@NotNull SyncDataType dataType, @NotBlank String dataKey, String outerApp) {
        Assert.notNull(dataKey, "数据类型为空");
        Assert.notBlank(dataKey, "数据标识为空");
        Assert.notBlank(outerApp, "外部系统标识为空");

        syncRepoProc.delete(dataType.name(), dataKey, outerApp);
    }

    /**
     * 根据ID删除
     *
     * @param id
     */
    @Transactional(rollbackFor = Exception.class)
    public void delete(long id) {
        syncRepoProc.delete(id);
    }

    private SysSyncDO saveBo2Do(SysSyncSaveBO saveBO) {
        SysSyncDO syncDO = SysSyncConvert.INSTANCE.bo2DO(saveBO);

        Assert.notBlank(syncDO.getDataType(), "同步的数据类型为空");
        Assert.notBlank(syncDO.getDataKey(), "同步的数据唯一标识为空");

        ObjUtil.ifNull(syncDO.getSyncOuter(), true, syncDO::setSyncOuter);
        Assert.notBlank(syncDO.getOuterApp(), "外部系统标识为空");
        ObjUtil.ifNull(syncDO.getSyncDataTime(), (Supplier<LocalDateTime>) LocalDateTime::now, syncDO::setSyncDataTime);
        ObjUtil.ifNull(syncDO.getManual(), true, syncDO::setManual);
        ObjUtil.ifNull(syncDO.getSyncSuccess(), false, syncDO::setSyncSuccess);

        return syncDO;
    }
}
