package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.model.bo.BusinessObjectBO;
import com.elitescloud.cloudt.system.model.vo.query.businessobject.BusinessObjectPageQueryVO;
import com.elitescloud.cloudt.system.service.common.constant.OpenApiSourceEnum;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessObjectDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessOperationDO;
import com.elitescloud.cloudt.system.service.model.entity.SysBusinessObjectDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/6
 */
@Repository
public class BusinessObjectRepoProc extends BaseRepoProc<SysBusinessObjectDO> {
    private static final QSysBusinessObjectDO QDO = QSysBusinessObjectDO.sysBusinessObjectDO;

    @Autowired
    private SystemProperties systemProperties;

    public BusinessObjectRepoProc() {
        super(QDO);
    }

    /**
     * 根据OpenApiCode删除
     *
     * @param openApiCode
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteForGatherByOpenApiCode(@NotBlank String openApiCode) {
        var predicate = QDO.openApiCode.eq(openApiCode)
                .and(QDO.dataSource.in(OpenApiSourceEnum.GATHER_PULL.name(), OpenApiSourceEnum.GATHER_PUSH.name()));
        super.delete(predicate);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByCodes(@NotBlank String openApiId, Collection<String> codes) {
        super.delete(QDO.code.in(codes).and(QDO.openApiCode.eq(openApiId)));
    }

    /**
     * 统计OpenApi对应的业务对象数量
     *
     * @param openApiCodes
     * @return
     */
    public Map<String, Long> countByOpenApi(Collection<String> openApiCodes) {
        return super.jpaQueryFactory.select(QDO.openApiCode, QDO.openApiCode.count())
                .from(QDO)
                .where(QDO.openApiCode.in(openApiCodes))
                .groupBy(QDO.openApiCode)
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.openApiCode), t -> t.get(QDO.openApiCode.count()), (t1, t2) -> t1));
    }

    /**
     * 根据OpenApi的Code获取对应的业务对象的ID和Code
     *
     * @param openApiCode
     * @return
     */
    public Map<String, Long> getIdAndCodeByOpenApiId(@NotBlank String openApiCode) {
        return super.jpaQueryFactory.select(QDO.id, QDO.code)
                .from(QDO)
                .where(QDO.openApiCode.eq(openApiCode))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.code), t -> t.get(QDO.id), (t1, t2) -> t1));
    }

    public List<SysBusinessObjectDO> listByOpenApiCode(@NotBlank String openApiCode) {
        return super.getList(QDO.openApiCode.eq(openApiCode));
    }

    /**
     * 分页查询管理
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysBusinessObjectDO> pageMng(BusinessObjectPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.openApiCode, queryVO.getOpenApiCode())
                .andEq(QDO.appCode, queryVO.getAppCode())
                .andEq(QDO.code, queryVO.getCode())
                .andLike(QDO.name, queryVO.getName())
                .andLike(QDO.description, queryVO.getDescription())
                .andLike(QDO.domain, queryVO.getDomain())
                .andEq(QDO.enabled, queryVO.getEnabled())
                .build();

        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.modifyTime.desc());
    }

    public List<BusinessObjectBO> listSimpleByAppCodes(@NotEmpty Collection<String> appCodes, Boolean enabled, Boolean publicResource) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.appCode, appCodes)
                .andEq(QDO.enabled, enabled)
                .and(Boolean.TRUE.equals(publicResource), () -> {
                    if (CollUtil.isEmpty(systemProperties.getBusinessObjectPublicResources())) {
                        return QDO.publicResource.eq(true);
                    }
                    return QDO.publicResource.eq(true).or(QDO.code.in(systemProperties.getBusinessObjectPublicResources()));
                })
                .build();

        return super.jpaQueryFactory.select(qBeanSimpleBO())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.createTime.asc())
                .fetch();
    }

    public BusinessObjectBO getSimple(@NotBlank String code) {
        return super.getOneByValue(qBeanSimpleBO(), QDO.code, code);
    }

    public BusinessObjectBO getSimpleByOperationCode(@NotBlank String operationCode) {
        var QDO_OPERATION = QSysBusinessOperationDO.sysBusinessOperationDO;
        var predicate = QDO.code.eq(JPAExpressions.select(QDO_OPERATION.businessObjectCode).from(QDO_OPERATION).where(QDO_OPERATION.operationCode.eq(operationCode)));
        return super.getOne(qBeanSimpleBO(), predicate);
    }

    public List<BusinessObjectBO> listSimple(@NotEmpty Collection<String> codes) {
        return super.getList(qBeanSimpleBO(), QDO.code.in(codes));
    }

    public List<BusinessObjectBO> listSimple(Boolean enabled) {
        Predicate predicate = enabled == null ? null : QDO.enabled.eq(enabled);
        return super.getList(qBeanSimpleBO(), predicate);
    }

    public List<String> filterCodes(@NotEmpty Collection<String> codes) {
        return super.getValueListByValue(QDO.code, QDO.code, codes);
    }

    public Map<String, String> getCodeAndNames(@NotEmpty Collection<String> codes) {
        return super.jpaQueryFactory.select(QDO.code, QDO.customName, QDO.name)
                .from(QDO)
                .where(QDO.code.in(codes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.code), t -> CharSequenceUtil.blankToDefault(t.get(QDO.customName), t.get(QDO.name)), (t1, t2) -> t1));
    }

    public String getName(@NotBlank String code) {
        return super.getValueByValue(QDO.name, QDO.code, code);
    }

    public String getAppCode(@NotBlank String code) {
        return super.getValueByValue(QDO.appCode, QDO.code, code);
    }

    public Boolean getSupportTenant(@NotBlank String code) {
        return super.getValueByValue(QDO.supportTenant, QDO.code, code);
    }

    private QBean<BusinessObjectBO> qBeanSimpleBO() {
        return Projections.bean(BusinessObjectBO.class, QDO.id, QDO.appCode, QDO.code, QDO.name, QDO.customName,
                QDO.description, QDO.enabled, QDO.publicResource, QDO.entityClassName);
    }
}
