package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.sys.OpenApiOperationPageQueryVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysOpenApiOperationDO;
import com.elitescloud.cloudt.system.service.model.entity.SysOpenApiOperationDO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/4
 */
@Slf4j
@Repository
public class OpenApiOperationRepoProc extends BaseRepoProc<SysOpenApiOperationDO> {
    private static final QSysOpenApiOperationDO QDO = QSysOpenApiOperationDO.sysOpenApiOperationDO;

    public OpenApiOperationRepoProc() {
        super(QDO);
    }

    /**
     * 删除数据
     *
     * @param openId
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteByOpenApiId(long openId) {
        super.delete(QDO.openApiInfoId.eq(openId));
    }

    /**
     * 统计OpenApi的数量
     *
     * @param openApiIds
     * @return
     */
    public Map<Long, Long> countByOpenApiId(Collection<Long> openApiIds) {
        return super.jpaQueryFactory
                .select(QDO.openApiInfoId, QDO.openApiInfoId.count())
                .from(QDO)
                .where(QDO.openApiInfoId.in(openApiIds))
                .groupBy(QDO.openApiInfoId)
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.openApiInfoId), t -> t.get(QDO.openApiInfoId.count()), (t1, t2) -> t1));
    }

    public Map<Long, Long> countByFeignOpenApiId(Collection<Long> openApiIds) {
        return super.jpaQueryFactory
                .select(QDO.openApiInfoId, QDO.openApiInfoId.count())
                .from(QDO)
                .where(QDO.openApiInfoId.in(openApiIds)
                        .and(QDO.apiType.eq(2)))
                .groupBy(QDO.openApiInfoId)
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.openApiInfoId), t -> t.get(QDO.openApiInfoId.count()), (t1, t2) -> t1));
    }

    /**
     * 分页查询操作
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysOpenApiOperationDO> pageMng(OpenApiOperationPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.openApiInfoId, queryVO.getOpenApiId())
                .andLike(QDO.summary, queryVO.getSummary())
                .andLike(QDO.description, queryVO.getDescription())
                .andLike(QDO.url, queryVO.getUrl())
                .andLike(QDO.operationGroup, queryVO.getOperationGroup())
                .andEq(QDO.apiType, queryVO.getApiType())
                .andEq(QDO.businessOperation, queryVO.getBusinessOperation())
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest());
    }
    /**
     * 去重获取全部的接口分组数据，用于内部接口的下拉查询
     *
     */
    public List<String> selectDistinctOperationGroupByApiType() {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.apiType, 2)
                .build();
        return super.jpaQueryFactory.selectDistinct(QDO.operationGroup).from(QDO)
                .where(predicate)
                .fetchAll().fetch();
    }
    /**
     * 根据OpenApiID获取
     *
     * @param openApiId
     * @return
     */
    public List<SysOpenApiOperationDO> listByOpenApiId(long openApiId) {
        return super.getList(QDO.openApiInfoId.eq(openApiId));
    }
}
