package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.model.bo.RoleApiRuleBO;
import com.elitescloud.cloudt.system.service.model.entity.QSysDprRoleApiRowRuleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysDprRoleApiRowRuleDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 3/29/2023
 */
@Repository
public class SysDprRoleApiRowRuleRepoProc extends BaseRepoProc<SysDprRoleApiRowRuleDO> {
    private static final QSysDprRoleApiRowRuleDO QDO = QSysDprRoleApiRowRuleDO.sysDprRoleApiRowRuleDO;

    public SysDprRoleApiRowRuleRepoProc() {
        super(QDO);
    }

    /**
     * 获取自定义规则
     *
     * @param roleId   角色ID
     * @param menuCode 菜单编码
     * @param apiCode  api编码
     * @return 规则列表
     */
    public List<SysDprRoleApiRowRuleDO> queryByApi(long roleId, @NotBlank String menuCode, @NotBlank String apiCode) {
        return super.jpaQueryFactory.selectFrom(QDO)
                .where(QDO.roleId.eq(roleId).and(QDO.menuCode.eq(menuCode)).and(QDO.apiCode.eq(apiCode)))
                .fetch()
                .stream()
                .sorted(Comparator.comparing(SysDprRoleApiRowRuleDO::getRuleOrder, Comparator.nullsLast(Integer::compareTo)))
                .collect(Collectors.toList());
    }

    /**
     * 删除自定义规则
     *
     * @param roleId   角色ID
     * @param menuCode 菜单编码
     * @param apiCode  api编码
     */
    public void deleteByApi(long roleId, @NotBlank String menuCode, @NotBlank String apiCode) {
        super.jpaQueryFactory.delete(QDO)
                .where(QDO.roleId.eq(roleId).and(QDO.menuCode.eq(menuCode)).and(QDO.apiCode.eq(apiCode)))
                .execute();
    }

    /**
     * 删除菜单下已不存在的api
     *
     * @param menuCode 菜单编码
     * @param apiCodes api编码
     */
    public void deleteForNotExistsApi(@NotBlank String menuCode, @NotEmpty Collection<String> apiCodes) {
        super.jpaQueryFactory.delete(QDO)
                .where(QDO.menuCode.eq(menuCode).and(QDO.apiCode.notIn(apiCodes)))
                .execute();
    }

    /**
     * 根据角色删除
     *
     * @param roleId
     */
    public void deleteByRole(long roleId) {
        super.deleteByValue(QDO.roleId, roleId);
    }

    /**
     * 根据角色查询
     *
     * @param roleId
     * @return
     */
    public List<SysDprRoleApiRowRuleDO> listByRole(long roleId) {
        return super.getListByValue(QDO.roleId, roleId);
    }

    /**
     * 根据角色查询数据权限规则
     *
     * @param roleId
     * @return
     */
    public List<RoleApiRuleBO> queryByRole(long roleId) {
        return super.jpaQueryFactory.select(qBeanRoleApiRule())
                .from(QDO)
                .where(QDO.roleId.eq(roleId))
                .fetch();
    }

    private QBean<RoleApiRuleBO> qBeanRoleApiRule() {
        return Projections.bean(RoleApiRuleBO.class, QDO.roleId, QDO.menuCode, QDO.apiCode);
    }
}
