package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantUserDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/23
 */
@Repository
public class TenantUserRepoProc extends BaseRepoProc<SysTenantUserDO> {
    private static final QSysTenantUserDO QDO = QSysTenantUserDO.sysTenantUserDO;
    private static final QSysTenantDO QDO_TENANT = QSysTenantDO.sysTenantDO;
    private static final QSysUserDO QDO_USER = QSysUserDO.sysUserDO;

    public TenantUserRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用禁用
     *
     * @param tenantId 租户ID
     * @param userId   用户ID
     * @param enabled  是否启用
     */
    public void updateEnabled(long tenantId, long userId, Boolean enabled) {
        jpaQueryFactory.update(QDO)
                .set(QDO.enabled, enabled)
                .where(QDO.sysUserId.eq(userId).and(QDO.sysTenantId.eq(tenantId)));
    }

    public void updateLoginTime(long id, long tenantId, LocalDateTime loginTime) {
        super.updateValue(QDO.lastLoginTime, loginTime, QDO.sysUserId.eq(id).and(QDO.sysTenantId.eq(tenantId)));
    }

    /**
     * 删除用户与租户关联
     *
     * @param tenantId 租户ID
     * @param userId   用户ID
     */
    public void delete(long tenantId, long userId) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.sysUserId.eq(userId).and(QDO.sysTenantId.eq(tenantId)))
                .execute();
    }

    /**
     * 根据用户ID删除
     *
     * @param userId 用户ID
     */
    public void deleteByUserId(long userId) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.sysUserId.eq(userId))
                .execute();
    }

    /**
     * 获取是否启用
     *
     * @param tenantId 租户ID
     * @param userId   用户ID
     * @return 是否启用
     */
    public Boolean getEnabled(long tenantId, long userId) {
        return jpaQueryFactory.select(QDO.enabled)
                .from(QDO)
                .where(QDO.sysUserId.eq(userId).and(QDO.sysTenantId.eq(tenantId)))
                .limit(1)
                .fetchOne();
    }


    /**
     * 获取是否启用
     *
     * @param tenantId 租户ID
     * @param userIds  用户ID
     * @return 是否启用
     */
    public Map<Long, Boolean> getEnabled(long tenantId, @NotEmpty Collection<Long> userIds) {
        return jpaQueryFactory.select(QDO.sysUserId, QDO.enabled)
                .from(QDO)
                .where(QDO.sysUserId.in(userIds).and(QDO.sysTenantId.eq(tenantId)))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.sysUserId), t -> ObjectUtil.defaultIfNull(t.get(QDO.enabled), false), (t1, t2) -> t1))
                ;
    }

    /**
     * 统计用户的租户数
     *
     * @param userId
     * @return
     */
    public long countByUserId(long userId) {
        return super.count(QDO.sysUserId, userId);
    }

    /**
     * 获取用户绑定的租户ID
     *
     * @param sysUserId
     * @param enabled
     * @return
     */
    public List<Long> listTenantIdByUserID(long sysUserId, Boolean enabled) {
        var predicate = QDO.sysUserId.eq(sysUserId);
        if (enabled != null) {
            predicate = predicate.and(QDO.enabled.eq(enabled));
        }
        return super.getValueList(QDO.sysTenantId, predicate);
    }

    /**
     * 用户用户
     *
     * @param tenantId
     * @param userIds
     * @return
     */
    public List<SysTenantUserDO> listByUserIds(long tenantId, @NotEmpty Collection<Long> userIds) {
        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(QDO.sysUserId.in(userIds).and(QDO.sysTenantId.eq(tenantId)))
                .fetch();
    }

    /**
     * 获取用户的租户
     *
     * @param userIds
     * @return
     */
    public Map<Long, List<IdCodeNameParam>> getTenantsOfUser(@NotEmpty Collection<Long> userIds) {
        return jpaQueryFactory.select(QDO.sysUserId, QDO_TENANT.id, QDO_TENANT.tenantName, QDO_TENANT.tenantCode)
                .from(QDO)
                .leftJoin(QDO_TENANT).on(QDO_TENANT.id.eq(QDO.sysTenantId))
                .where(QDO.sysUserId.in(userIds).and(QDO_TENANT.id.isNotNull()))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.sysUserId),
                        Collectors.mapping(t -> new IdCodeNameParam(t.get(QDO_TENANT.id), t.get(QDO_TENANT.tenantCode), t.get(QDO_TENANT.tenantName)), Collectors.toList())));
    }

    /**
     * 根据用户名称查询ID
     *
     * @param tenantId
     * @param lastname
     * @return
     */
    public List<Long> queryUserIdByLastname(long tenantId, String lastname, Boolean enabled) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.sysTenantId, tenantId)
                .andIn(StrUtil.isNotBlank(lastname), QDO.sysUserId, () -> {
                    Predicate predicateEnabled = null;
                    if (enabled != null) {
                        predicateEnabled = QDO_USER.enabled.eq(enabled);
                    }
                    return JPAExpressions.select(QDO_USER.id).from(QDO_USER).where(
                            QDO_USER.lastName.like("%" + lastname + "%").and(predicateEnabled)
                    );
                })
                .andEq(QDO.enabled, enabled)
                .build();
        return super.getValueList(QDO.sysUserId, predicate);
    }
}
