package com.elitescloud.cloudt.ucenter.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.model.dto.FileObjDTO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.el.coordinator.file.business.vo.FileObjRespVO;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;

import com.elitescloud.cloudt.ucenter.api.dto.AdInfoDTO;
import com.elitescloud.cloudt.ucenter.api.dto.AdSpaceDTO;
import com.elitescloud.cloudt.ucenter.api.dto.ArticleInfoDTO;
import com.elitescloud.cloudt.ucenter.api.vo.param.*;
import com.elitescloud.cloudt.ucenter.api.vo.resp.AdInfoDetailRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.resp.AdInfoRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.resp.AdSpaceRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.resp.ArticleLinkRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.save.AdInfoSaveVO;
import com.elitescloud.cloudt.ucenter.api.vo.save.AdSpaceSaveVO;
import com.elitescloud.cloudt.ucenter.common.constant.ConstantsUcenter;
import com.elitescloud.cloudt.ucenter.common.constant.UdcEnum;
import com.elitescloud.cloudt.ucenter.convert.AdManageConvert;
import com.elitescloud.cloudt.ucenter.convert.ArticleManageConvert;
import com.elitescloud.cloudt.ucenter.entity.AdInfoDO;
import com.elitescloud.cloudt.ucenter.entity.AdSpaceDO;
import com.elitescloud.cloudt.ucenter.repo.*;
import com.elitescloud.cloudt.ucenter.service.AdManageService;
import com.elitescloud.cloudt.ucenter.utils.SysGenerator;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
@Slf4j
public class AdManageServiceImpl extends BaseServiceImpl implements AdManageService {
    private static final String UDC_LINK_TYPE = "LINK_TYPE";
    private static final String UDC_SHOW_FLAG = "SHOW_FLAG";
    private static final String UDC_TARGET_TYPE = "TARGET_TYPE";
    private static final String UDC_SOURCE_TYPE = "SOURCE_TYPE";

    @Autowired
    private  AdSpaceRepo adSpaceRepo;
    @Autowired
    private  AdInfoRepo adInfoRepo;
    @Autowired
    private  AdSpaceRepoProc adSpaceRepoProc;
    @Autowired
    private  AdInfoRepoProc adInfoRepoProc;
    @Autowired
    private  ArticleInfoRepoProc articleInfoRepoProc;
    @Autowired
    private  SysGenerator sysGenerator;
    @Autowired
    private FileService<String> fileService;

    @Override
    @SysCodeProc
    public PagingVO<AdSpaceRespVO> adSpacePagingSearch(AdSpacePagingParam param) {
        PagingVO<AdSpaceDTO> result = adSpaceRepoProc.findAdSpacePagingResults(param);
        List<AdSpaceDTO> records = result.getRecords();
        return PagingVO.<AdSpaceRespVO>builder()
                .records(AdManageConvert.INSTANCE.adSpaceDtoSToVoS(records))
                .total(result.getTotal())
                .build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdateAdSpace(AdSpaceSaveVO saveVO) {
        // 新增
        if (ObjectUtil.isNull(saveVO.getId())) {
            if (adSpaceRepo.existsByAdSpaceName(saveVO.getAdSpaceName())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "广告位名称已存在");
            }
            AdSpaceDO spaceDO = AdManageConvert.INSTANCE.copyAdSpaceSaveToDo(saveVO);
            AdSpaceDO save = adSpaceRepo.save(spaceDO);
            return save.getId();
        } else {
            // 编辑
            Optional<AdSpaceDO> optional = adSpaceRepo.findById(saveVO.getId());
            if (optional.isEmpty()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未找到广告位信息");
            }
            AdSpaceDO spaceDO = optional.get();
            // 编辑只能编辑更改
            spaceDO.setAdSpaceDesc(saveVO.getAdSpaceDesc());
            if (StrUtil.isNotEmpty(saveVO.getShowStyle())) {
                spaceDO.setShowStyle(saveVO.getShowStyle());
            }
            adSpaceRepo.save(spaceDO);
            return spaceDO.getId();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<Long> deleteAdInfo(List<Long> ids) {
        List<AdInfoDO> byId = adInfoRepo.findAllByIdIn(ids);
        if (CollectionUtil.isNotEmpty(byId)) {
            List<Long> deleteIds = byId.stream().filter(info ->
                    ObjectUtil.equals(info.getShowFlag(), UdcEnum.AD_SHOW_FLAG_HIDE.getValueCode())
            ).map(info -> info.getId()).collect(Collectors.toList());
            adInfoRepo.deleteAdInfoByIds(deleteIds, "1");
            return deleteIds;
        } else {
            throw new BusinessException("未查询到广告信息");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdateAdInfo(AdInfoSaveVO saveVO) {
        AdSpaceDO spaceDO = adSpaceRepo.findByAdSpaceCode(saveVO.getAdSpaceCode());
        // 新增
        if (ObjectUtil.isNull(saveVO.getId())) {
            String adCode = sysGenerator.generateCode(ConstantsUcenter.AD_NO);
            if (adInfoRepo.existsByAdCode(adCode)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "发号器生成的广告编号：" + adCode + " 已存在，请检查发号器配置是否正确");
            }
            // 使用发号器生成广告编号
            saveVO.setAdCode(adCode);
        } else {
            // 编辑
            Optional<AdInfoDO> optional = adInfoRepo.findById(saveVO.getId());
            if (optional.isEmpty()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未找到指定广告信息");
            }
            AdInfoDO adInfoDO = optional.get();

            if (StrUtil.equals(adInfoDO.getShowFlag(), UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能编辑隐藏状态的广告信息");
            }

        }
        AdInfoDO adInfoDO = AdManageConvert.INSTANCE.copyAdInfoSaveToDo(saveVO);
        if (ObjectUtil.isNotNull(spaceDO)) {
            adInfoDO.setAdSpaceCode(spaceDO.getAdSpaceCode()).setAdSpaceId(spaceDO.getId()).setAdSpaceName(spaceDO.getAdSpaceName());
        }
        AdInfoDO saveDo = adInfoRepo.save(adInfoDO);
        return saveDo.getId();
    }

    @Override
    @SysCodeProc
    public PagingVO<AdInfoRespVO> search(AdInfoPagingParam searchParam) {

        List<AdInfoDO> infoDOS = adInfoRepo.findAllByShowFlag(UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode());
        // 广告结束时间在当前时间之前自动将是否显示设置为隐藏
        List<Long> list = infoDOS.stream().
                filter(item -> item.getShowEndTime().isBefore(LocalDateTime.now()))
                .filter(item -> StrUtil.equals(UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode(), item.getShowFlag()))
                .map(item -> item.getId()).collect(Collectors.toList());

        if (CollectionUtil.isNotEmpty(list)) {
            adInfoRepo.updateShowFlagByIds(list, UdcEnum.AD_SHOW_FLAG_HIDE.getValueCode());
        }

        // 设置是否显示完成之后进行分页查询
        PagingVO<AdInfoDTO> result = adInfoRepoProc.findAdInfoPagingResults(searchParam);
        List<AdInfoDTO> records = result.getRecords();
        return PagingVO.<AdInfoRespVO>builder()
                .records(AdManageConvert.INSTANCE.adInfoDtoSToVoS(records))
                .total(result.getTotal())
                .build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object showUpdate(AdInfoShowParam param) {

        /**
         * 单条：
         * 针对是否显示为‘是’的数据，支持点击隐藏按钮
         * 校验：除此广告外，是否同位置的还有其余显示的广告数据：有，则隐藏此数据；
         * 无，则弹窗提示：
         * 此条数据为此广告位置的最后一个显示的广告，如果隐藏后，
         * 此广告位置在前端展示空白信息，是否继续？确认后执行隐藏此数据
         */
        Optional<AdInfoDO> byId = adInfoRepo.findById(param.getId());
        if (byId.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "没有查询到广告信息");
        }
        LocalDateTime showEndTime = byId.get().getShowEndTime();
        // 广告结束时间 早于 当前时间 不允许修改为显示
        if (ObjectUtil.isNotNull(showEndTime) && LocalDateTime.now().isAfter(showEndTime)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "显示时间维护有问题，请检查");
        }
        return adInfoRepo.updateShowFlagById(param.getId(), param.getShowFlag());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer sortUpdate(AdInfoSortParam param) {
        return adInfoRepo.updateSortNoById(param.getId(), param.getSortNo());
    }

    @Override
    @SysCodeProc
    public AdInfoDetailRespVO queryInfoDetail(Long id) {
        Optional<AdInfoDO> byId = adInfoRepo.findById(id);
        if (byId.isPresent()) {
            return AdManageConvert.INSTANCE.infoDoToDetailVo(byId.get());
        } else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "没有查询到广告信息");
        }
    }

    @Override
    @SysCodeProc
    public Object queryLinkAddress(LinkAddressPagingParam param) {
        // 文章类型--目前只有文章
        if (StrUtil.equals(param.getTargetType(), UdcEnum.WS_TARGET_TYPE_ARTICLE.getValueCode())) {
            ArticleInfoPagingParam pagingParam = ArticleManageConvert.INSTANCE.paramToParam(param);
            PagingVO<ArticleInfoDTO> results = articleInfoRepoProc.findArticleInfoPagingResults(pagingParam);
            List<ArticleLinkRespVO> respVoList = ArticleManageConvert.INSTANCE.infoDTOsToVos(results.getRecords());
            return PagingVO.<ArticleLinkRespVO>builder()
                    .total(results.getTotal())
                    .records(respVoList).build();
        }
        // 广告类型
        else if (StrUtil.equals(param.getTargetType(), UdcEnum.WS_TARGET_TYPE_AD.getValueCode())) {
            return Collections.emptyList();
        }
        return Collections.emptyList();
    }

    @Override
    public List<AdInfoDetailRespVO> queryDetailList(AdInfoQueryParam queryParam) {
        if (queryParam == null) {
            queryParam = new AdInfoQueryParam();
        }

        var adList = adInfoRepoProc.queryList(queryParam);
        if (adList.isEmpty()) {
            return Collections.emptyList();
        }

        // 图片地址
        var fileCodes = adList.stream().flatMap(t -> Stream.of(t.getImgFileCode(), t.getVideoFileCode()))
                .filter(StrUtil::isNotBlank)
                .collect(Collectors.toSet());
        Map<String, String> fileUrlMap;
        if (!fileCodes.isEmpty()) {
            var fileObjs = fileService.queryDto(new ArrayList<>(fileCodes)).getData();
            if (CollUtil.isNotEmpty(fileObjs)) {
                fileUrlMap = fileObjs.stream().collect(Collectors.toMap(FileObjDTO::getFileCode, FileObjDTO::getUrl, (v1, v2) -> v1));
            } else {
                fileUrlMap = Collections.emptyMap();
            }
        } else {
            fileUrlMap = Collections.emptyMap();
        }

        // UDC翻译
        var udcMap = super.udcMap(Application.NAME, Set.of(UDC_LINK_TYPE, UDC_SHOW_FLAG, UDC_TARGET_TYPE, UDC_SOURCE_TYPE));
        return adList.stream()
                .map(t -> {
                    var adVO = AdManageConvert.INSTANCE.infoDoToDetailVo(t);
                    if (adVO.getAdLinkType() != null) {
                        adVO.setAdLinkTypeName(udcMap.getOrDefault(UDC_LINK_TYPE, Collections.emptyMap()).get(adVO.getAdLinkType()));
                    }
                    if (adVO.getShowFlag() != null) {
                        adVO.setShowFlagName(udcMap.getOrDefault(UDC_SHOW_FLAG, Collections.emptyMap()).get(adVO.getShowFlag()));
                    }
                    if (adVO.getTargetType() != null) {
                        adVO.setTargetTypeName(udcMap.getOrDefault(UDC_TARGET_TYPE, Collections.emptyMap()).get(adVO.getTargetType()));
                    }
                    if (adVO.getSourceType() != null) {
                        adVO.setSourceTypeName(udcMap.getOrDefault(UDC_SOURCE_TYPE, Collections.emptyMap()).get(adVO.getSourceType()));
                    }
                    if (adVO.getImgFileCode() != null) {
                        adVO.setImgUrl(fileUrlMap.get(adVO.getImgFileCode()));
                    }
                    if (adVO.getVideoFileCode() != null) {
                        adVO.setVideoUrl(fileUrlMap.get(adVO.getVideoFileCode()));
                    }
                    return adVO;
                }).collect(Collectors.toList());
    }


}
