package com.elitescloud.cloudt.system.common;

import cn.hutool.core.util.StrUtil;
import com.elitescloud.cloudt.system.util.BeanUtils;
import com.elitescloud.cloudt.system.util.ExceptionUtils;
import com.elitescloud.cloudt.system.util.RedisUtil;
import com.elitescloud.cloudt.system.util.RestWrapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.elitescloud.cloudt.system.util.StringUtils.tenantKey;

/**
 * @Auther: Mark
 * @Date: 2024/12/17 18:46
 * @Description:
 */
@Slf4j
@RequiredArgsConstructor
public class OAuth2Client {
    //    private final WebClient webClient;
    private final RestWrapper restWrapper;
    private final RedisUtil redisUtil;


    /**
     * 获取可用的token
     *
     * @param platformCode 平台代码
     * @param url          认证地址
     * @param clientId     账号
     * @param clientSecret 密码
     * @return token
     */
    public String getAccessToken(String platformCode, String url, String clientId, String clientSecret) {
        String tokenKey = tenantKey(platformCode, clientId);
        String cacheToken = redisUtil.getString(tokenKey);
        if (StrUtil.isNotBlank(cacheToken)) {
            log.info("[BPMN][INFINITY] CacheToken(" + tokenKey + "): ", cacheToken);
            return cacheToken;
        }
        return createAccessToken(platformCode, url, clientId, clientSecret);
    }

    /**
     * 生成新的token
     *
     * @param platformCode 平台代码
     * @param url          认证地址
     * @param clientId     账号
     * @param clientSecret 密码
     * @return token
     */
    private String createAccessToken(String platformCode, String url, String clientId, String clientSecret) {
        AccessToken token = authentication(url, clientId, clientSecret, AccessToken.class);
        String accessToken = token.getAccessToken();
        Long expiresIn = token.getExpiresIn();
        log.info("[BPMN][INFINITY] AccessToken: ", accessToken);
        log.info("[BPMN][INFINITY] ExpiresIn: ", expiresIn);
        if (expiresIn > 10) {
            redisUtil.set(tenantKey(platformCode, clientId), accessToken, expiresIn - 5, TimeUnit.SECONDS);
        }
        return accessToken;
    }

    public String authentication(String url, String clientId, String clientSecret) {
        String accessToken = authentication(url, clientId, clientSecret, String.class);
        log.info("[BPMN][INFINITY] AccessToken: ", accessToken);
        return accessToken;
    }

//    public <T> T authentication(String url, String clientId, String clientSecret, Class<T> clazz) {
//        log.info("[BPMN][INFINITY] OAuth2 Url: " + url);
//        log.info("[BPMN][INFINITY] OAuth2 clientId: " + clientId);
//        var uriSpec = webClient.method(HttpMethod.POST);
//        var bodySpec = uriSpec.uri(url);
//        bodySpec.acceptCharset(StandardCharsets.UTF_8);
//        bodySpec.contentType(MediaType.APPLICATION_FORM_URLENCODED);
//        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
//        multiValueMap.add("grant_type", "client_credentials");
//        multiValueMap.add("client_id", clientId);
//        multiValueMap.add("client_secret", clientSecret);
//        bodySpec.body(BodyInserters.fromMultipartData(multiValueMap));
//        return bodySpec.retrieve().bodyToMono(clazz).block();
//    }

    /**
     * TODO 暂时的解决方案 待完善，使用相对简单的对接方式
     */
    public <T> T authentication(String url, String clientId, String clientSecret, Class<T> clazz) {
        log.info("[BPMN][INFINITY] Url: ", url);
        log.info("[BPMN][INFINITY] ClientId: ", clientId);
        try {
            Map<String, Object> param = new HashMap<>();
            param.put("grant_type", "client_credentials");
            param.put("client_id", clientId);
            param.put("client_secret", clientSecret);
            String result = restWrapper.postFormUrlencoded(url, param);
            return BeanUtils.toBean(result, clazz);
        } catch (Exception e) {
            log.error("[BPMN][INFINITY] 认证失败：", ExceptionUtils.formatException(e));
            throw new RuntimeException(e);
        }
    }
}
