package com.elitescloud.cloudt.system.controller.mng.common;

import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.elitescloud.cloudt.system.dto.resp.OrgTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonCompanyPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonEmployeePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonOrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmpGroupPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmployeeListQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.*;
import com.elitescloud.cloudt.system.service.*;
import com.elitescloud.cloudt.system.service.common.constant.BusinessOperationConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * 组织相关通用接口.
 *
 * @author Kaiser（wang shao）
 * 2022/9/20
 */
@Api(tags = "通用组织接口")
@RestController
@RequestMapping(value = "/mng/common/org", produces = MediaType.APPLICATION_JSON_VALUE)
@BusinessObject(businessType = BusinessObjectConstant.SYS_COMMON)
@Validated
public class CommonOrgController {

    private OrgQueryService orgQueryService;
    private EmployeeQueryService employeeQueryService;
    private OuQueryService ouQueryService;
    private EmpGroupQueryService empGroupQueryService;
    private SysPositionQueryService portalQueryService;

    /**
     * 获取行政组织树
     *
     * @param tree     是否返回树形结构数据
     * @param employee 是否返回员工数据
     * @param all      是否返回租户下所有组织数据，默认仅返回所在租户组织
     * @return 行政组织树
     */
    @BusinessObjectOperation(businessObjectType = BusinessObjectConstant.SYS_ORG, operationCode = BusinessOperationConstant.COMMON_ORG_TREE, operationDescription = "[通用组件]查询行政组织树",
            operationType = OperationTypeEnum.SELECT, dataPermissionEnabled = true)
    @ApiOperation(value = "[通用组件]查询行政组织树")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pCode", value = "上级节点编码"),
            @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true"),
            @ApiImplicitParam(name = "employee", value = "是否返回员工数据", dataType = "boolean", defaultValue = "false"),
            @ApiImplicitParam(name = "all", value = "包含所有组织数据", dataType = "boolean", defaultValue = "false"),
    })
    @GetMapping(value = "/tree/get")
    public ApiResult<List<OrgTreeNodeRespVO>> getTree(@RequestParam(name = "pCode", required = false) String pCode,
                                                      @RequestParam(name = "tree", defaultValue = "true") Boolean tree,
                                                      @RequestParam(name = "employee", defaultValue = "false") Boolean employee,
                                                      @RequestParam(name = "all", defaultValue = "false") Boolean all) {
        return orgQueryService.getOrgTree(pCode, tree, employee, all);
    }

    /**
     * 异步获取行政组织树
     *
     * @param pId      上级ID
     * @param employee 是否返回员工
     * @return 组织树
     */
    @BusinessObjectOperation(businessObjectType = BusinessObjectConstant.SYS_ORG, operationCode = BusinessOperationConstant.COMMON_ORG_TREE_ASYNC, operationDescription = "[通用组件]异步查询行政组织树",
            operationType = OperationTypeEnum.SELECT, dataPermissionEnabled = true)
    @ApiOperation(value = "[通用组件]异步查询行政组织树")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pId", value = "上级节点ID", dataType = "long"),
            @ApiImplicitParam(name = "pCode", value = "上级节点编码"),
            @ApiImplicitParam(name = "employee", value = "是否返回员工数据", dataType = "boolean", defaultValue = "false"),
            @ApiImplicitParam(name = "all", value = "包含所有组织数据", dataType = "boolean", defaultValue = "false"),
    })
    @GetMapping(value = "/tree/getAsync")
    public ApiResult<List<OrgTreeNodeRespVO>> getTreeAsync(@RequestParam(name = "pId", required = false) Long pId,
                                                           @RequestParam(name = "pCode", required = false) String pCode,
                                                           @RequestParam(name = "employee", defaultValue = "false") Boolean employee,
                                                           @RequestParam(name = "all", defaultValue = "false") Boolean all) {
        return orgQueryService.getOrgTreeAsync(pCode, pId, employee, all);
    }

    @ApiOperation(value = "[通用组件]查询多组织树")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true"),
            @ApiImplicitParam(name = "employee", value = "是否返回员工数据", dataType = "boolean", defaultValue = "false"),
            @ApiImplicitParam(name = "all", value = "包含所有组织数据", dataType = "boolean", defaultValue = "false"),
            @ApiImplicitParam(name = "treeCode", value = "树编码", allowMultiple = true),
    })
    @GetMapping(value = "/multi/tree/get")
    public ApiResult<Map<IdCodeNameParam, List<OrgTreeNodeRespVO>>> getMultiTree(@RequestParam(name = "tree", defaultValue = "true") Boolean tree,
                                                                                 @RequestParam(name = "employee", defaultValue = "false") Boolean employee,
                                                                                 @RequestParam(name = "all", defaultValue = "false") Boolean all,
                                                                                 @RequestParam(name = "treeCode", required = false) String[] treeCode) {
        return orgQueryService.getMultiOrgTree(tree, employee, all, treeCode);
    }

    /**
     * 分页查询组织
     *
     * @param queryVO 查询参数
     * @return 组织列表
     */
    @ApiOperation(value = "[通用组件]分页查询组织")
    @ApiOperationSupport(order = 5)
    @PostMapping(value = "/org/page")
    public ApiResult<PagingVO<OrgPagedRespVO>> page(@RequestBody CommonOrgPageQueryVO queryVO) {
        return orgQueryService.page(queryVO);
    }

    /**
     * 获取特定类型的上级组织
     *
     * @param id         组织ID
     * @param code       组织编码
     * @param parentType 特定上级类型
     * @return 公司列表
     */
    @ApiOperation(value = "获取特定类型的上级组织")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "组织ID", dataType = "long"),
            @ApiImplicitParam(name = "code", value = "组织编码"),
            @ApiImplicitParam(name = "parentType", value = "特定上级类型", required = true),
    })
    @PostMapping(value = "/org/getParentOrgByType")
    public ApiResult<IdCodeNameParam> getParentOrgByType(@RequestParam(name = "id", required = false) Long id,
                                                         @RequestParam(name = "code", required = false) String code,
                                                         @RequestParam(name = "parentType") String parentType) {
        return orgQueryService.getParentOrgByType(id, code, parentType);
    }

    /**
     * 分页查询员工
     *
     * @param queryVO 查询参数
     * @return 员工列表
     */
    @BusinessObjectOperation(businessObjectType = BusinessObjectConstant.SYS_EMPLOYEE, operationCode = BusinessOperationConstant.COMMON_EMP_PAGE, operationDescription = "[通用组件]分页查询员工",
            operationType = OperationTypeEnum.SELECT, dataPermissionEnabled = true)
    @ApiOperation(value = "[通用组件]分页查询员工")
    @ApiOperationSupport(order = 11)
    @PostMapping(value = "/employee/page")
    public ApiResult<PagingVO<EmployeePagedRespVO>> pageQueryEmployee(@RequestBody CommonEmployeePageQueryVO queryVO) {
        return employeeQueryService.pageQuery(queryVO);
    }

    /**
     * 查询员工列表
     * <p>
     * 查询条件为空时返回空集合
     *
     * @param queryVO 查询参数
     * @return 员工列表
     */
    @ApiOperation(value = "[通用组件]查询员工列表")
    @ApiOperationSupport(order = 12)
    @GetMapping(value = "/employee/list")
    public ApiResult<List<EmployeeListRespVO>> listQuery(EmployeeListQueryVO queryVO) {
        return employeeQueryService.listQuery(queryVO);
    }

    @ApiOperation(value = "[通用组件]分页查询员工组")
    @ApiOperationSupport(order = 13)
    @PostMapping(value = "/empGroup/page")
    public ApiResult<PagingVO<EmpGroupPageRespVO>> pageQuery(@RequestBody EmpGroupPageQueryVO queryVO) {
        return empGroupQueryService.pageQuery(queryVO);
    }

    @ApiOperation(value = "[通用组件]查询员工组列表")
    @ApiOperationSupport(order = 14)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "type", value = "员工组类型"),
            @ApiImplicitParam(name = "enabled", value = "启用状态", dataType = "boolean"),
    })
    @GetMapping(value = "/empGroup/list")
    public ApiResult<List<EmpGroupListRespVO>> listQuery(@RequestParam(name = "type", required = false) String type,
                                                         @RequestParam(name = "enabled", required = false) Boolean enabled) {
        return empGroupQueryService.listQuery(type, enabled);
    }

    /**
     * 分页查询公司
     *
     * @param queryVO 查询参数
     * @return 公司列表
     */
    @ApiOperation(value = "[通用组件]分页查询公司")
    @ApiOperationSupport(order = 21)
    @PostMapping(value = "/company/page")
    public ApiResult<PagingVO<OuPageRespVO>> page(@RequestBody CommonCompanyPageQueryVO queryVO) {
        return ouQueryService.pageQuery(queryVO);
    }

    @ApiOperation(value = "[通用组件]查询岗位列表")
    @ApiOperationSupport(order = 31)
    @PostMapping(value = "/post/list")
    public ApiResult<List<IdCodeNameParam>> queryPosition(@RequestParam(name = "orgId", required = false) Long orgId,
                                                          @RequestParam(name = "orgCode", required = false) String orgCode,
                                                          @RequestParam(name = "size", required = false) Integer size,
                                                          @RequestParam(name = "postName", required = false) String postName) {
        return portalQueryService.queryList(orgId, orgCode, size, postName);
    }

    @Autowired
    public void setOrgQueryService(OrgQueryService orgQueryService) {
        this.orgQueryService = orgQueryService;
    }

    @Autowired
    public void setEmployeeQueryService(EmployeeQueryService employeeQueryService) {
        this.employeeQueryService = employeeQueryService;
    }

    @Autowired
    public void setOuQueryService(OuQueryService ouQueryService) {
        this.ouQueryService = ouQueryService;
    }

    @Autowired
    public void setEmpGroupQueryService(EmpGroupQueryService empGroupQueryService) {
        this.empGroupQueryService = empGroupQueryService;
    }

    @Autowired
    public void setPortalQueryService(SysPositionQueryService portalQueryService) {
        this.portalQueryService = portalQueryService;
    }
}
