package com.elitescloud.cloudt.system.modules.message.service.impl;

import cn.hutool.core.lang.UUID;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.constant.MsgSendStateEnum;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.constant.MsgTypeEnum;
import com.elitescloud.cloudt.system.constant.SysMsgReceiverTypeEnum;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.dto.req.SysSiteMsgSaveDTO;
import com.elitescloud.cloudt.system.modules.message.entity.SysMsgSendRecordDO;
import com.elitescloud.cloudt.system.modules.message.entity.SysMsgSendRecordDtlDO;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordDtlRepoProc;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordRepoProc;
import com.elitescloud.cloudt.system.modules.message.service.SiteMsgMngService;
import com.elitescloud.cloudt.system.service.manager.UserQueryManager;
import com.elitescloud.cloudt.system.service.repo.SysTenantUserRepoProc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/11/29
 */
@Service
public class SiteMsgMngServiceImpl implements SiteMsgMngService {
    private static final Logger logger = LoggerFactory.getLogger(SiteMsgMngServiceImpl.class);

    @Autowired
    private UserQueryManager userQueryManager;
    @Autowired
    private SysMsgSendRecordRepoProc msgSendRecordRepoProc;
    @Autowired
    private SysMsgSendRecordDtlRepoProc msgSendRecordDtlRepoProc;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> send(SysSiteMsgSaveDTO saveDTO) {
        Assert.notNull(saveDTO, "保存参数为空");
        SysTenantDTO tenant = tenantClientProvider.getCurrentTenant();

        // 过滤账户
        Assert.notEmpty(saveDTO.getReceiverUserIds(), "接收人账户ID为空");
        Assert.hasText(saveDTO.getTitle(), "标题为空");
        List<Long> userIds = new ArrayList<>(saveDTO.getReceiverUserIds());
        if (saveDTO.getSenderUserId() != null) {
            userIds.add(saveDTO.getSenderUserId());
        }
        Map<Long, IdCodeNameParam> userMap = userQueryManager.queryUsers(userIds, tenant == null ? null : tenant.getId())
                .stream().collect(Collectors.toMap(IdCodeNameParam::getId, param -> param, (t1, t2) -> t1));

        String batchId = UUID.fastUUID().toString();
        LocalDateTime now = LocalDateTime.now();
        String customParamStr = JSONUtil.toJsonString(saveDTO.getCustomParams());
        // 发送人
        IdCodeNameParam sender = null;
        if (saveDTO.getSenderUserId() != null) {
            sender = userMap.get(saveDTO.getSenderUserId());
            Assert.notNull(sender, "发送人不存在");
        }
        // 消息记录
        List<SysMsgSendRecordDtlDO> recordDtlList = new ArrayList<>(userIds.size());
        for (Long receiverUserId : saveDTO.getReceiverUserIds()) {
            SysMsgSendRecordDO record = new SysMsgSendRecordDO();
            record.setMsgType(MsgTypeEnum.CUSTOM.getCode());
            if (sender != null) {
                record.setSendUserId(sender.getId().toString());
                record.setSendUserName(sender.getName());
                record.setSendUserCode(sender.getCode());
            }
            IdCodeNameParam receiver = userMap.get(receiverUserId);
            if (receiver == null) {
                if (Boolean.TRUE.equals(saveDTO.getIgnoreUserNotExist())) {
                    continue;
                }
                throw new BusinessException("账户" + receiverUserId + "不存在");
            }
            record.setRecipientUserId(receiver.getId().toString());
            record.setRecipientUserName(receiver.getName());
            record.setRecipientUserCode(receiver.getCode());
            record.setSendTime(now);
            record.setBatchUuid(batchId);
            record.setCustomParamJson(customParamStr);
            msgSendRecordRepoProc.save(record);

            // 保存明细
            SysMsgSendRecordDtlDO recordDtlDO = saveRecordDtl(saveDTO, record);
            recordDtlList.add(recordDtlDO);
        }
        if (recordDtlList.isEmpty()) {
            throw new BusinessException("没有有效的接收人账号");
        }
        msgSendRecordDtlRepoProc.save(recordDtlList);

        return ApiResult.ok(batchId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> delete(String batchId) {
        Assert.hasText(batchId, "批次号不能为空");

        msgSendRecordRepoProc.deleteByBatchId(batchId);
        msgSendRecordDtlRepoProc.deleteByBatchId(batchId);
        return ApiResult.ok(batchId);
    }

    private SysMsgSendRecordDtlDO saveRecordDtl(SysSiteMsgSaveDTO saveDTO, SysMsgSendRecordDO record) {
        SysMsgSendRecordDtlDO recordDtlDO = new SysMsgSendRecordDtlDO();
        recordDtlDO.setMsgSendRecordId(record.getId());
        recordDtlDO.setMsgType(record.getMsgType());
        recordDtlDO.setSendState(MsgSendStateEnum.OK.name());
        recordDtlDO.setSentTypeCode(MsgSendTypeEnum.SYS_INTERIOR.name());
        recordDtlDO.setSendTypeName(MsgSendTypeEnum.SYS_INTERIOR.getDescription());
        recordDtlDO.setSentTimeStart(record.getSendTime());
        recordDtlDO.setSentTimeEnd(record.getSendTime());
        recordDtlDO.setSendUserId(record.getSendUserId());
        recordDtlDO.setSendUserName(record.getSendUserName());
        recordDtlDO.setSendUserCode(record.getSendUserCode());
        recordDtlDO.setRecipientUserId(record.getRecipientUserId());
        recordDtlDO.setRecipientUserCode(record.getRecipientUserCode());
        recordDtlDO.setRecipientUserName(record.getRecipientUserName());
        recordDtlDO.setRecipientAccountType(SysMsgReceiverTypeEnum.RECEIVER.name());
        recordDtlDO.setRecipientAccount(record.getRecipientUserCode());
        recordDtlDO.setSubject(saveDTO.getTitle());
        recordDtlDO.setTitleContent(saveDTO.getTitle());
        recordDtlDO.setMessageContent(saveDTO.getContent());
        recordDtlDO.setCustomParamJson(record.getCustomParamJson());
        recordDtlDO.setReadFlg(false);
        recordDtlDO.setBatchUuid(record.getBatchUuid());

        return recordDtlDO;
    }
}
