package com.elitescloud.cloudt.system.modules.orgtree.service.impl;

import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.system.modules.orgtree.common.OrgUdcEnum;
import com.elitescloud.cloudt.system.modules.orgtree.common.UserUtil;
import com.elitescloud.cloudt.system.modules.orgtree.common.UtilEnum;
import com.elitescloud.cloudt.system.modules.orgtree.convert.OrgBuTreeConvert;
import com.elitescloud.cloudt.system.modules.orgtree.convert.OrgBuTreeDConvert;
import com.elitescloud.cloudt.system.modules.orgtree.model.*;
import com.elitescloud.cloudt.system.modules.orgtree.model.entity.OrgBuTreeDDO;
import com.elitescloud.cloudt.system.modules.orgtree.model.entity.OrgBuTreeDO;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.*;
import com.elitescloud.cloudt.system.modules.orgtree.service.OrgBuTreeDDomainService;
import com.elitescloud.cloudt.system.modules.orgtree.service.OrgBuTreeVersionDomainService;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeDRepo;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeDRepoProc;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeRepo;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeRepoProc;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author Tristan
 * @date 2020/7/2
 */
@Service
@RequiredArgsConstructor
public class OrgBuTreeDDomainServiceImpl implements OrgBuTreeDDomainService {
    private static final String ONE = "1";

    private final OrgBuTreeDRepo orgBuTreeDRepo;
    private final OrgBuTreeDRepoProc orgBuTreeDRepoProc;
    private final OrgBuTreeRepo orgBuTreeRepo;
    private final OrgBuTreeRepoProc orgBuTreeRepoProc;
    private final OrgBuTreeVersionDomainService orgBuTreeVersionService;

    private final UdcProvider sysUdcService;

    /**
     * 组织树明细新增
     *
     * @param paramList
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<OrgBuTreeDDetailDTO> orgBuTreeDCreate(List<OrgBuTreeDSaveParam> paramList) {

        if (CollectionUtils.isEmpty(paramList)) {
            return Collections.emptyList();
        }

        List<OrgBuTreeDDO> buTreeDDos = new ArrayList<OrgBuTreeDDO>();

        // 校验并转换为do
        paramList.forEach(s -> {
            if (!this.buIdIsExist(s)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "组织树节点新增失败，当前组织树上已存在该组织，请勿重复添加");
            }
            buTreeDDos.add(OrgBuTreeDConvert.INSTANCE.saveParamToDo(s));
        });

        // 设置树形结构信息
        boolean isRoot = (buTreeDDos.get(0).getPid() == null || buTreeDDos.get(0).getPid().equals(0L));
        if (isRoot) {
            var pid = 0L;
            var level = 1;
            buTreeDDos.forEach(ddo -> {
                ddo.setPid(pid);
                ddo.setLevel(level);
                ddo.setCodePath(ddo.getBuCode());
            });
        } else {
            var pBuNodeOptional = orgBuTreeDRepo.findById(buTreeDDos.get(0).getPid());
            if (pBuNodeOptional.isEmpty()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据异常，未找到指定的父级组织");
            }
            var pBuNode = pBuNodeOptional.get();
            var level = pBuNode.getLevel() == null ? null : pBuNode.getLevel() + 1;
            var codePath = pBuNode.getCodePath() == null ? null : pBuNode.getCodePath();
            buTreeDDos.forEach(ddo -> {
                ddo.setLevel(level);
                ddo.setCodePath(codePath == null ? null : codePath + "/" + ddo.getBuCode());
            });
        }
        // 保存数据
        var resultList = orgBuTreeDRepo.saveAll(buTreeDDos);

        //组织树节点添加父级节点
        if (StringUtils.isNotBlank(paramList.get(0).getBuTreeDId())) {
            var orgBuTreeDetailList = this.orgBuTreeDUpdate(paramList, resultList);
            buTreeDDos.addAll(orgBuTreeDetailList);
        }

        // 结果转换为 dto
        var orgBuTreeDDetailDTOList = OrgBuTreeDConvert.INSTANCE.doToDto(buTreeDDos);

        return orgBuTreeDDetailDTOList;
    }

    /**
     * 组织树节点添加父级节点
     *
     * @param paramList
     * @param resultList
     */
    private List<OrgBuTreeDDO> orgBuTreeDUpdate(List<OrgBuTreeDSaveParam> paramList, List<OrgBuTreeDDO> resultList) {
        List<OrgBuTreeDDO> orgBuTreeDetailList;
        if (paramList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能选一个父级");
        }
        var orgBuTreeDSaveParam = paramList.get(0);
        var orgBuTreeDOptional = orgBuTreeDRepo.findById(Long.valueOf(orgBuTreeDSaveParam.getBuTreeDId()));
        if (orgBuTreeDOptional.isPresent()) {
            var orgBuTreeDDO = orgBuTreeDOptional.get();
            var orgBuTreeDDOList = orgBuTreeDRepo.findAllByBuTreeIdAndCodePathLike(orgBuTreeDDO.getBuTreeId(), orgBuTreeDDO.getCodePath() + "%");
            var pid = resultList.get(0).getId();
            var oldCodePath = orgBuTreeDDO.getCodePath();
            var newCodePath = resultList.get(0).getCodePath() + "/" + orgBuTreeDDO.getBuCode();
            var saveDos = orgBuTreeDDOList.stream().map(s -> {
                if (String.valueOf(s.getId()).equals(orgBuTreeDSaveParam.getBuTreeDId())) {
                    s.setPid(pid);
                }
                s.setCodePath(s.getCodePath().replace(oldCodePath, newCodePath));
                s.setLevel(s.getLevel() + 1);
                return s;
            }).collect(Collectors.toList());
            orgBuTreeDetailList = orgBuTreeDRepo.saveAll(saveDos);
        } else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据异常,组织树节点:" + orgBuTreeDSaveParam.getBuTreeDId() + "，不存在");
        }
        return orgBuTreeDetailList;
    }


    /**
     * 组织树节点删除
     *
     * @param id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteTreeDetail(Long id, Long buTreeId) {
        //下级所有节点同步删除
        var result = this.orgBuTreedAllSearch(id, UtilEnum.CODE_TYPE.getValue(), buTreeId);
        result.stream().map(s -> s.getId()).forEach(orgBuTreeDRepo::deleteById);
        orgBuTreeDRepo.deleteById(id);
    }

    /**
     * 通过id查询所有下级
     * 通过id查询下一级
     *
     * @param id
     */
//    @SysCodeProc
//    @Override
//    public List<OrgBuTreedAllDTO> orgBuTreedAllSearch(Long id, String searchType, Long buTreeId) {
//        if (UtilEnum.CODE_TYPE.getValue().equals(searchType)) {
//            //通过id查询所有下级
//            List<OrgBuTreedAllDTO> result = new ArrayList<>();
//            var orgBuTreedVOList = orgBuTreeDRepoProc.selectAllBuByBuTreeId(buTreeId);
//            if (CollectionUtils.isNotEmpty(orgBuTreedVOList)) {
//                var udcMap = sysUdcService.getValueMapByUdcCode("yst-supp", "BU_TYPE");
//                orgBuTreedVOList.forEach(s -> s.setBuTypeName(orgBuService.buTypeConvert(s.getBuType(), udcMap)));
//                return build(orgBuTreedVOList, id, result);
//            } else {
//                return Collections.emptyList();
//            }
//        } else {
//            //通过id查询下一级
//            return orgBuTreeDRepoProc.selectByPid(id);
//        }
//    }

    /**
     * 通过id查询所有下级
     * 通过id查询下一级
     *
     * @param id
     */
    @SysCodeProc
    @Override
    public List<OrgBuTreedAllDTO> orgBuTreedAllSearch(Long id, String searchType, Long buTreeId) {
        if (UtilEnum.CODE_TYPE.getValue().equals(searchType)) {
            //通过id查询所有下级
            List<OrgBuTreedAllDTO> result = new ArrayList<>();
            var orgBuTreedVOList = orgBuTreeDRepoProc.selectAllBuByBuTreeId(buTreeId);
            if (CollectionUtils.isNotEmpty(orgBuTreedVOList)) {
                var udcMap = sysUdcService.getValueMapByUdcCode("yst-supp", "BU_TYPE");
                orgBuTreedVOList.forEach(s -> s.setBuTypeName(udcMap.get(s.getBuType())));
                return build(orgBuTreedVOList, id, result);
            } else {
                return Collections.emptyList();
            }
        } else {
            // 通过id查询下一级
            return orgBuTreeDRepoProc.selectByPid(id);
        }
    }
    /**
     * 获取组织树形结构
     *
     * @param buTreeId
     * @return
     */
//    @Override
//    public List<OrgBuTreeDDTO> orgBuTreeDSearchTree(Long buTreeId, String searchType, Long buTreeDId) {
//        var treeVos = orgBuTreeDRepoProc.searchByBuTreeId(buTreeId, searchType);
//        if (CollectionUtils.isNotEmpty(treeVos)) {
//            var udcMap = sysUdcService.getValueMapByUdcCode("yst-supp", "BU_TYPE");
//            treeVos.forEach(s -> s.setBuTypeName(orgBuService.buTypeConvert(s.getBuType(), udcMap)));
//            List<OrgBuTreeDDTO> orgBuTreeDVOS = TreeUtils.buildTree(buTreeDId, true, treeVos);
//            return orgBuTreeDVOS;
//        } else {
//            return Collections.emptyList();
//        }
//    }

    /**
     * 获取组织树形结构
     *
     * @param buTreeId
     * @return
     */
    @Override
    public List<OrgBuTreeDDTO> orgBuTreeDSearchTree(Long buTreeId, String searchType, Long buTreeDId) {

        return orgBuTreeVersionService.getOrgBuTreeDFromSystem(buTreeId, searchType, buTreeDId);

//        var treeVos = orgBuTreeDRepoProc.searchByBuTreeId(buTreeId, searchType);
//        if (CollectionUtils.isNotEmpty(treeVos)) {
//            var udcMap = sysUdcService.getValueMapByUdcCode("yst-supp", "BU_TYPE");
//            treeVos.forEach(s -> s.setBuTypeName(orgBuService.buTypeConvert(s.getBuType(), udcMap)));
//            List<OrgBuTreeDDTO> orgBuTreeDVOS = TreeUtils.buildTree(buTreeDId, true, treeVos);
//            return orgBuTreeDVOS;
//        } else {
//            return Collections.emptyList();
//        }
    }

    /**
     * 递归查询组织树明细，由上往下
     *
     * @param orgBuTreedVOList
     * @param id
     * @param result
     * @return 树形结构
     */
    private List<OrgBuTreedAllDTO> build(List<OrgBuTreedAllDTO> orgBuTreedVOList, Long id, List<OrgBuTreedAllDTO> result) {
        var treedList = orgBuTreedVOList.stream().filter(s -> id.equals(s.getPid())).collect(Collectors.toList());
        for (OrgBuTreedAllDTO vo : treedList) {
            result.add(vo);
            build(orgBuTreedVOList, vo.getId(), result);
        }
        return result;
    }

    /**
     * 组织树上，一个组织节点下不能出现重复的组织
     *
     * @param param
     * @return
     */
    public Boolean buIdIsExist(OrgBuTreeDSaveParam param) {
        var optional = orgBuTreeDRepo.findByBuTreeId(Long.valueOf(param.getBuTreeId()));
        if (optional.isPresent() && StringUtils.isNotBlank(param.getBuId())) {
            var buIds = optional.get().stream().map(s -> s.getBuId()).collect(Collectors.toList());
            if (buIds.contains(Long.valueOf(param.getBuId()))) {
                return false;
            }
        }
        return true;
    }

    /**
     * 递归查询组织树明细，由下往上
     */
    public List<OrgBuTreedAllDTO> buTreedDown(List<OrgBuTreedAllDTO> orgBuTreeDDOs, Long pid, List<OrgBuTreedAllDTO> result) {
        if (pid == null || pid == 0L) {
            return result;
        }
        var buOptional = orgBuTreeDDOs.stream().filter(s -> s.getId().equals(pid)).findFirst();
        buOptional.ifPresent(t -> {
            result.add(t);
            buTreedDown(orgBuTreeDDOs, t.getPid(), result);
        });
        return result;
    }

    /**
     * 新增/编辑 组织树版本数据
     *
     * @param param 数据参数
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(cacheNames = "ORG_BU_TREE", key = "'BU_TREE_ID_' + #param.id", condition = "#param.id != null")
    public void saveBuTreeDV(OrgBuTreeVSaveParam param) {
        // 首先获取组织树信息
        OrgBuTreeDO orgBuTreeDO = OrgBuTreeConvert.INSTANCE.saveVDParaToDo(param);
        boolean updateFlag = false;
        // 如果id为空 表示新增 需要校验是否可以新增组织树版本数据
        if (null == param.getId()) {
            // 如果是新增组织树 需要判断 组织编码是否重复
            if (ONE.equals(param.getBuTreeVersion())) {
                // 设置版本为1
                orgBuTreeDO.setNowVersion(ONE);
                if (orgBuTreeRepo.existsByBuTreeCode(param.getBuTreeCode())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "组织树编号:" + param.getBuTreeCode() + "，已存在，请勿重复添加");
                }
            } else {
                // 获取该组织树其他版本信息 确定版本号 确定组织树是否可以进行版本的新增
                List<OrgBuTreeDTO> buTreeVersion = orgBuTreeRepoProc.getBuTreeListByBuTreeCode(
                        OrgBuTreeVListParam.builder().buTreeCode(param.getBuTreeCode()).build());
                // 最大版本的数据 判断是否可以新增数据
                Map<Integer, String> collect = buTreeVersion.stream()
                        .collect(HashMap::new, (m, v) -> m.put(Integer.valueOf(v.getNowVersion()), v.getBuTreeStatus()), HashMap::putAll);
                Optional<Integer> maxNowVersion = collect.keySet().stream().max(Integer::compareTo);
                if (maxNowVersion.isPresent()) {
                    String status = collect.get(maxNowVersion.get());
                    // 最新版本 如果是草稿状态 不允许发布新版本
                    if (OrgUdcEnum.ORG_BUTREE_STATUS_DRAFT.getUdcVal().equals(status)) {
                        throw new BusinessException("当前存在尚未发布的组织树，无法创建新版本组织树");
                    }
                }

                // 设置是否当前版本正确
                Integer maxVersion = null;
                if (maxNowVersion.isPresent()) {
                    maxVersion = maxNowVersion.get();
                    String nowVersion = (maxVersion + 1) + "";
                    if (param.getBuTreeVersion().equals(nowVersion)) {
                        orgBuTreeDO.setNowVersion(param.getBuTreeVersion());
                    } else {
                        orgBuTreeDO.setNowVersion(nowVersion);
                    }
                } else {
                    orgBuTreeDO.setNowVersion(param.getBuTreeVersion());
                }
            }
        } else {
            orgBuTreeDO.setNowVersion(param.getBuTreeVersion());
            updateFlag = true;
        }
        orgBuTreeDO.setBuTreeVersion(null);

        // ----- -----------------  更新操作  单独业务处理

        Long id = null;
        if (null != param.getId()) {
            // 更新
            Optional<OrgBuTreeDO> byId = orgBuTreeRepo.findById(param.getId());
            if (byId.isPresent()) {
                OrgBuTreeDO buTreeDO = byId.get();
                OrgBuTreeConvert.INSTANCE.updateDoFromParamV(buTreeDO, param);
                orgBuTreeRepo.save(buTreeDO);
                id = buTreeDO.getId();
            } else {
                throw new BusinessException("组织树不存在");
            }
        } else {
            // 新增
            orgBuTreeDO.setIsNowVersion(false);
            orgBuTreeDO.setBuTreeStatus(OrgUdcEnum.ORG_BUTREE_STATUS_DRAFT.getUdcVal());
            orgBuTreeDO.setDeleteFlag(0);
            // 保存组织树信息 并获取id
            OrgBuTreeDO save = orgBuTreeRepo.save(orgBuTreeDO);
            id = save.getId();
        }
        // 获取树节点列表
        List<OrgBuTreeVDSaveParam> buTreeDList = param.getTreeVDSaveParams();

        // 保存节点信息
        orgBuTreeVersionService.saveBuTreeDV(buTreeDList, id, updateFlag);
    }

    /**
     * 根据组织号获取组织版本信息
     *
     * @param buTreeCode 组织树编码
     * @return list
     */
    @Override
    public List<String> getBuTreeVersionByBuTreeCode(String buTreeCode) {
        return orgBuTreeRepoProc.getBuTreeVersionByBuTreeCode(buTreeCode);
    }

    /**
     * 根据组织号获取所有组织树信息
     *
     * @param param 组织树编码
     * @return list
     */
    @Override
    public List<OrgBuTreeDTO> getBuTreeListByParam(OrgBuTreeVListParam param) {
        return orgBuTreeRepoProc.getBuTreeListByBuTreeCode(param);
    }

    /**
     * 根据param获取 组织树节点版本信息
     *
     * @param param param
     * @return list
     */
    @Override
    @SysCodeProc
//    @CacheEvict(cacheNames = "ORG_BU_TREE", key = "'BU_TREE_ID_' + #param.id")
    public OrgBuTreeAndTreeDDTO getBuTreeDListByBuTreeCode(OrgBuTreeVListParam param) {
        List<OrgBuTreeDTO> buTreeListByBuTreeCode = orgBuTreeRepoProc.getBuTreeListByBuTreeCode(param);
        if (buTreeListByBuTreeCode.isEmpty()) {
            return null;
        }
        if (buTreeListByBuTreeCode.size() == 1) {
            OrgBuTreeDTO orgBuTreeDTO = buTreeListByBuTreeCode.get(0);
            OrgBuTreeAndTreeDDTO orgBuTreeAndTreeDVO = OrgBuTreeConvert.INSTANCE.dtoToDDto(orgBuTreeDTO);
            if (orgBuTreeDTO != null) {
                orgBuTreeAndTreeDVO.setOrgBuTreeDVOList(orgBuTreeVersionService.getOrgBuTreeDByCache(orgBuTreeDTO.getId(), UtilEnum.CODE_TYPE.getValue(), null));
            }
            return orgBuTreeAndTreeDVO;
        } else {
            throw new BusinessException("出现重复版本数据请检查数据");
        }

    }

    /**
     * 发布组织树版本
     *
     * @param id 组织树id
     */
    @Override
    public void releaseBuTreeVersion(Long id) {
        // 获取当前发布的数据信息
        Optional<OrgBuTreeDO> nowVersionOptional = orgBuTreeRepo.findById(id);
        if (nowVersionOptional.isPresent()) {
            OrgBuTreeDO nowVersion = nowVersionOptional.get();
            List<OrgBuTreeDO> nowActiveBuTreeList = orgBuTreeRepoProc.getNowActiveBuTree(nowVersion.getBuTreeCode());
            LocalDateTime now = LocalDateTime.now();
            if (CollectionUtils.isEmpty(nowActiveBuTreeList)) {
                nowVersion.setReleaseTime(now);
                nowVersion.setReleaseUser(Objects.requireNonNull(UserUtil.getNowUser()).getUsername());
                nowVersion.setIsNowVersion(true);
                nowVersion.setBuTreeStatus(OrgUdcEnum.ORG_BUTREE_STATUS_ACTIVE.getUdcVal());
                orgBuTreeRepo.save(nowVersion);
            } else {
                // 判断组织树是否存在多个生效版本
                int size = nowActiveBuTreeList.size();
                if (size > 1) {
                    throw new BusinessException("组织树存在多个生效的版本，请检查数据");
                }
                OrgBuTreeDO oldVersion = nowActiveBuTreeList.get(0);


                // ------------------------ 旧版本 -------------------------
                // 设置失效时间
                // 开始更新状态
                oldVersion.setIsNowVersion(false);
                oldVersion.setDisableTime(now);
                oldVersion.setBuTreeStatus(OrgUdcEnum.ORG_BUTREE_STATUS_CLOSED.getUdcVal());

                // --------------------------- 新版本 ---------------------
                nowVersion.setReleaseTime(now);
                nowVersion.setReleaseUser(Objects.requireNonNull(UserUtil.getNowUser()).getUsername());
                nowVersion.setIsNowVersion(true);
                nowVersion.setBuTreeStatus(OrgUdcEnum.ORG_BUTREE_STATUS_ACTIVE.getUdcVal());

                // 更新旧版本
                orgBuTreeRepo.save(oldVersion);
                // 更新新版本
                orgBuTreeRepo.save(nowVersion);
            }

        }
    }

    /**
     * 根据param 获取组织树节点  仅获取节点数据 并无上下级信息
     *
     * @param param param
     * @return 组织树节点信息
     */
    @Override
    public List<OrgBuTreeDDTO> findBuTreeDByParam(OrgBuTreeDParam param) {
        return orgBuTreeDRepoProc.findBuTreeDByParam(param);
    }
}
