package com.elitescloud.cloudt.system.modules.orgtree.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.SysConstant;
import com.elitescloud.cloudt.system.modules.orgtree.common.OrgUdcEnum;
import com.elitescloud.cloudt.system.modules.orgtree.common.UtilEnum;
import com.elitescloud.cloudt.system.modules.orgtree.convert.OrgBuTreeConvert;
import com.elitescloud.cloudt.system.modules.orgtree.convert.OrgTreeConvert;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreeDDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreeDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreeDetailDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreePagingDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.entity.OrgBuTreeDO;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.*;
import com.elitescloud.cloudt.system.modules.orgtree.service.OrgBuTreeDDomainService;
import com.elitescloud.cloudt.system.modules.orgtree.service.OrgBuTreeDomainService;
import com.elitescloud.cloudt.system.modules.orgtree.service.OrgBuTreeVersionDomainService;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.MultiOrgTreeRepoProc;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeDRepo;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeRepo;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeRepoProc;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author Tristan
 * @date 2020/7/2
 */
@Service
@RequiredArgsConstructor
public class OrgBuTreeDomainServiceImpl implements OrgBuTreeDomainService {

    private final OrgBuTreeRepo orgBuTreeRepo;
    private final OrgBuTreeDRepo orgBuTreeDRepo;
    private final OrgBuTreeRepoProc orgBuTreeRepoProc;
    private final MultiOrgTreeRepoProc multiOrgTreeRepoProc;
    private final OrgBuTreeDDomainService orgBuTreeDService;
    private final OrgBuTreeVersionDomainService orgBuTreeVersionService;

    /**
     * 组织树新增、编辑
     *
     * @param param
     * @return
     */
    @Override
    @CacheEvict(cacheNames = "ORG_BU_TREE", key = "'BU_TREE_ID_' + #param.id", condition = "#param.id != null && !#param.id.isEmpty()")
    @Transactional(rollbackFor = Exception.class)
    public Long orgBuTreeSaveOrUpdate(OrgBuTreeSaveParam param) {
        if (SysConstant.ORG_TREE_EXECUTIVE_CODE.equals(param.getBuTreeCode())) {
            throw new BusinessException(SysConstant.ORG_TREE_EXECUTIVE_CODE + "是内置编码，不可用");
        }
        OrgBuTreeDO orgBuTreeDO;
        if (StringUtils.isNotBlank(param.getId())) {
            var id = Long.valueOf(param.getId());
            // 更新
            if (orgBuTreeRepo.existsByBuTreeCodeAndNowVersionAndIdNot(param.getBuTreeCode(), param.getNowVersion(), id)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "组织树编号:" + param.getBuTreeCode() + "，已存在，请勿重复添加");
            }

            //id不为空，即为更新操作
            var optional = orgBuTreeRepo.findById(id);
            if (optional.isEmpty()) {
                throw new BusinessException("组织树id：" + param.getId() + ", 不存在");
            }

            orgBuTreeDO = optional.get();

            OrgBuTreeConvert.INSTANCE.updateDoFromSaveParam(param, orgBuTreeDO);
        } else {
            // 新增
            if (orgBuTreeRepo.existsByBuTreeCode(param.getBuTreeCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "组织树编号:" + param.getBuTreeCode() + "，已存在，请勿重复添加");
            }
            orgBuTreeDO = OrgBuTreeConvert.INSTANCE.saveParamToDo(param);
        }

        orgBuTreeRepo.save(orgBuTreeDO);


        return orgBuTreeDO.getId();
    }

    /**
     * 通过id获取组织树详细信息，包含树状结构
     *
     * @param id
     * @return
     */
//    @Override
//    public OrgBuTreeDetailDTO getOrgBuTreeDetailById(Long id) {
//        var orgBuTreeDetailVO = orgBuTreeRepoProc.findBuTreeDetailById(id);
//        if (orgBuTreeDetailVO != null) {
//            orgBuTreeDetailVO.setOrgBuTreeDVOList(orgBuTreeVersionService.getOrgBuTreeDByCache(id, UtilEnum.CODE_TYPE.getValue(), null));
//        }
//        return orgBuTreeDetailVO;
//    }

    /**
     * 从系统域获取组织信息
     * @param id
     * @return
     */
    @Override
    public OrgBuTreeDetailDTO getOrgBuTreeDetailById(Long id) {
        var orgBuTreeDetailVO = orgBuTreeRepoProc.findBuTreeDetailById(id);
        if(ObjectUtil.isNotNull(orgBuTreeDetailVO)) {
            // orgBuTreeDetailVO.setOrgBuTreeDVOList(orgBuTreeVersionService.getOrgBuTreeDByCache(id, UtilEnum.CODE_TYPE.getValue(), null));
            List<OrgBuTreeDDTO> orgBuTreeDVOList = orgBuTreeVersionService.getOrgBuTreeDFromSystem(id, UtilEnum.CODE_TYPE.getValue(), null);
            orgBuTreeDetailVO.setOrgBuTreeDVOList(orgBuTreeDVOList);
        }
        return orgBuTreeDetailVO;
    }

    /**
     * 组织树分页条件查询
     *
     * @param param
     * @return
     */
    @Override
    @SysCodeProc
    public PagingVO<OrgBuTreePagingDTO> queryOrgBuTreeByParams(OrgBuTreePagingParam param) {
        return orgBuTreeRepoProc.searchByParams(param);
    }

    /**
     * 组织树确认、停用
     *
     * @param param 参数
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void updateStatus(OrgBuTreeStatusParam param) {

        if (CollectionUtils.isEmpty(param.getIds())) {
            return;
        }

        var ids = param.getIds().stream().distinct().filter(Objects::nonNull).map(Long::valueOf).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(ids)) {
            return;
        }

        if (StringUtils.isBlank(param.getBuTreeStatus())) {
            return;
        }

        orgBuTreeRepoProc.updateStatusByIds(ids, param.getBuTreeStatus());
    }

    /**
     * 根据param 获取组织树信息
     *
     * @param param param
     * @return list
     */
    @Override
    public List<OrgBuTreeDTO> findOrgBuTreeDTOByParam(OrgBuTreeParam param) {
        return orgBuTreeRepoProc.findOrgBuTreeVoByParam(param);
    }


    /**
     * 根据param获取组织树履历数据
     *
     * @param param param
     * @return 分页数据
     */
    @Override
    public PagingVO<OrgBuTreeDTO> getBuTreePagingByParam(OrgBuTreeVPagingParam param) {
        return orgBuTreeRepoProc.getBuTreePagingByParam(param);
    }

    /**
     * 根据组织树编码获取是否可以创建新版本组织树标识
     *
     * @param buTreeCode
     * @return
     */
    @Override
    public Boolean getCreateNewVersionFlag(String buTreeCode) {
        // 获取该组织树其他版本信息 确定版本号 确定组织树是否可以进行版本的新增
        List<OrgBuTreeDTO> buTreeVersionByBuTreeCode = orgBuTreeRepoProc.getBuTreeListByBuTreeCode(
                OrgBuTreeVListParam.builder().buTreeCode(buTreeCode).build());
        // 最大版本的数据 判断是否可以新增数据
        Map<Integer, String> collect = buTreeVersionByBuTreeCode.stream()
                .collect(HashMap::new, (m, v) -> m.put(Integer.valueOf(v.getNowVersion()), v.getBuTreeStatus()), HashMap::putAll);
        Optional<Integer> max = collect.keySet().stream().max(Integer::compareTo);
        if (max.isPresent()) {
            String status = collect.get(max.get());
            return !OrgUdcEnum.ORG_BUTREE_STATUS_DRAFT.getUdcVal().equals(status);
        }
        return true;
    }

    /**
     * 根据id删除草稿状态的组织树
     *
     * @param id 组织树id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBuTreeById(Long id) {
        // 校验改组织树状态是否为操作状态
        Optional<OrgBuTreeDO> byId = orgBuTreeRepo.findById(id);
        if (byId.isPresent()) {
            OrgBuTreeDO buTreeDO = byId.get();
            if (!OrgUdcEnum.ORG_BUTREE_STATUS_DRAFT.getUdcVal().equals(buTreeDO.getBuTreeStatus())) {
                throw new BusinessException("仅草稿状态的组织树可以删除");
            }
        } else {
            throw new BusinessException("组织树不存在，请检查数据");
        }
        // 删除组织树数据
        orgBuTreeRepo.deleteById(id);
        // 删除节点数据
        orgBuTreeDRepo.deleteByBuTreeId(id);
        multiOrgTreeRepoProc.deleteEmployeeOfTree(id);
    }

    @Override
    public OrgBuTreeDTO findByBuTreeCodeAndNowVersion(String buTreeCode, String nowVersion) {
        Optional<OrgBuTreeDO> result = orgBuTreeRepo.findByBuTreeCodeAndNowVersion(buTreeCode, nowVersion);
        if(result.isPresent()){
            OrgBuTreeDO orgBuTreeDO = result.get();
            return  OrgTreeConvert.INSTANCE.doToDTO(orgBuTreeDO);
        }
        return null;
    }
}
