package com.elitescloud.cloudt.system.provider.orgsync;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.boot.util.StrUtil;
import com.elitescloud.cloudt.system.modules.wecom.model.user.UserSaveParam;
import com.elitescloud.cloudt.system.modules.wecom.util.WeComTool;
import com.elitescloud.cloudt.system.service.callback.EmployeeChangedCallback;
import com.elitescloud.cloudt.system.service.common.constant.SyncDataType;
import com.elitescloud.cloudt.system.service.manager.EmployeeOrgManager;
import com.elitescloud.cloudt.system.service.manager.SysSyncManager;
import com.elitescloud.cloudt.system.service.model.bo.SysEmployeeSaveBO;
import com.elitescloud.cloudt.system.service.model.bo.SysSyncSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysEmployeeDO;
import com.elitescloud.cloudt.system.service.model.entity.SysEmployeeOrgDO;
import com.elitescloud.cloudt.system.service.repo.EmployeeOrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.SyncRepoProc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 企微通讯录的同步.
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/26
 */
@Component
public class WecomEmpSync extends BaseWecomSync implements EmployeeChangedCallback {
    private static final Logger logger = LoggerFactory.getLogger(WecomEmpSync.class);

    @Autowired
    private EmployeeRepoProc employeeRepoProc;
    @Autowired
    private TaskExecutor taskExecutor;

    @Override
    public void onUpsert(boolean add, SysEmployeeSaveBO saveBO, SysEmployeeDO employeeDO) {
        taskExecutor.execute(() -> {
            var needSync = employeeRepoProc.needSyncOuter(employeeDO.getId());
            if (!needSync) {
                logger.info("员工无需向企微同步");
                return;
            }

            var cfg = getConfig();
            if (cfg == null) {
                logger.info("企微未配置，忽略同步");
                return;
            }

            // 向企微同步
            syncToWeCom(employeeDO, cfg);
        });
    }

    @Override
    public void onEnabled(Long employeeId, boolean enabled) {
        taskExecutor.execute(() -> {
            var needSync = employeeRepoProc.needSyncOuter(employeeId);
            if (!needSync) {
                logger.info("员工无需向企微同步");
                return;
            }

            var cfg = getConfig();
            if (cfg == null) {
                logger.info("企微未配置，忽略同步");
                return;
            }

            // 向企微同步
            var employeeDO = employeeRepoProc.get(employeeId);
            if (employeeDO != null) {
                syncToWeCom(employeeDO, cfg);
            }
        });
    }

    @Override
    public void onDelete(Long employeeId) {
        taskExecutor.execute(() -> {
            var needSync = employeeRepoProc.needSyncOuter(employeeId);
            if (!needSync) {
                logger.info("员工无需向");
                return;
            }

            var cfg = getConfig();
            if (cfg == null) {
                logger.info("企微未配置，忽略同步");
                return;
            }
            logger.info("删除企微员工：{}", employeeId);

           syncDeleteToWecom(employeeId, cfg);
        });
    }

    static void syncDeleteToWecom(long employeeId, WecomConfig cfg) {
        SyncRepoProc syncRepoProc = SpringContextHolder.getBean(SyncRepoProc.class);
        var syncInfo = SpringContextHolder.getBean(SyncRepoProc.class).getIdCode(SyncDataType.EMPLOYEE.name(), employeeId + "", cfg.getCorpid());
        if (syncInfo == null) {
            // 不存在，无需处理
            return;
        }

        syncRepoProc.delete(syncInfo.getId());

        // 向企微同步
        if (CharSequenceUtil.isBlank(syncInfo.getCode())) {
            return;
        }
        var token = getAccessTokenOfWecom(cfg);
        WeComTool.userDelete(token, employeeId + "");
    }

    static void syncToWeCom(SysEmployeeDO employee, WecomConfig cfg) {
        logger.info("同步企微员工：{}", employee.getId());
        if (StrUtil.isAllBlank(employee.getMobile(), employee.getEmail())) {
            logger.info("邮箱和手机号都为空，忽略同步");
            return;
        }
        var syncInfo = SpringContextHolder.getBean(SyncRepoProc.class).getIdCode(SyncDataType.EMPLOYEE.name(), employee.getId() + "", cfg.getCorpid());
        var syncId = syncInfo == null ? null : syncInfo.getId();

        if (syncInfo == null) {
            // 初始化同步信息
            SysSyncSaveBO sysSyncSaveBO = initSyncSaveBO(employee.getId(), cfg.getCorpid());
            syncId = SpringContextHolder.getBean(SysSyncManager.class).save(sysSyncSaveBO);
        }

        // 同步给企微
        var empOrgList = SpringContextHolder.getBean(EmployeeOrgRepoProc.class).getEmployeeOrgByEmployeeId(employee.getId());
        List<Long> orgIds = Collections.emptyList();
        List<Long> leaderOrgIds = Collections.emptyList();
        Long leaderId = null;
        Long mainOrgId = null;
        // 过滤出行政组织
        var executiveOrgIds = empOrgList.stream().map(SysEmployeeOrgDO::getOrgId).filter(ObjUtil::isNotNull).collect(Collectors.toList());
        if (!executiveOrgIds.isEmpty()) {
            executiveOrgIds = SpringContextHolder.getBean(OrgRepoProc.class).filterExecutiveOrg(executiveOrgIds);
        }
        if (!executiveOrgIds.isEmpty()) {
            Map<Long, Long> wecomDepartmentIdMap = WecomOrgSync.getWecomIds(executiveOrgIds, cfg.getCorpid());
            orgIds = new ArrayList<>();
            leaderOrgIds = new ArrayList<>();
            for (var sysEmployeeOrgDO : empOrgList) {
                if (!executiveOrgIds.contains(sysEmployeeOrgDO.getOrgId())) {
                    continue;
                }
                var departmentId = wecomDepartmentIdMap.get(sysEmployeeOrgDO.getOrgId());
                if (departmentId == null) {
                    continue;
                }

                orgIds.add(departmentId);
                if (Boolean.TRUE.equals(sysEmployeeOrgDO.getAssignee())) {
                    leaderOrgIds.add(departmentId);
                }
                if (leaderId == null && sysEmployeeOrgDO.getLeaderEmployeeId() != null) {
                    leaderId = sysEmployeeOrgDO.getLeaderEmployeeId();
                }
                if (mainOrgId == null && Boolean.TRUE.equals(sysEmployeeOrgDO.getAssignee())) {
                    mainOrgId = departmentId;
                }
            }
        }
        var saveParam = convertSaveParam(employee, orgIds, leaderOrgIds, leaderId, mainOrgId);
        var token = getAccessTokenOfWecom(cfg);
        var syncResult = syncInfo != null && StringUtils.hasText(syncInfo.getCode()) ?
                WeComTool.userUpdate(token, saveParam) : WeComTool.userCreate(token, saveParam);
        SpringContextHolder.getBean(SysSyncManager.class).updateSyncResult(syncId, syncResult.isSuccess(), employee.getId().toString(),
                syncResult.isSuccess() ? null : syncResult.getErrcode() + ":" + syncResult.getErrmsg());
    }

    /**
     * 转换员工信息
     *
     * @param employeeDO 员工信息
     * @param departmentIds 部门ID
     * @param isLeaderDepartmentIds 是领导的部门ID
     * @param directLeader 直接领导的ID
     * @param mainDepartment 主部门ID
     * @return 员工信息
     */
    static UserSaveParam convertSaveParam(SysEmployeeDO employeeDO, List<Long> departmentIds, List<Long> isLeaderDepartmentIds,
                                          Long directLeader, Long mainDepartment) {
        UserSaveParam saveParam = new UserSaveParam();
        saveParam.setUserid(employeeDO.getId().toString());
        saveParam.setName(employeeDO.getFullName());
        saveParam.setMobile(employeeDO.getMobile());
        saveParam.setDepartment(departmentIds);
        saveParam.setPosition(employeeDO.getDuty());
        saveParam.setGender(Gender.FEMALE.getValue().equals(employeeDO.getGender()) ? 2 : 1);
        saveParam.setEmail(employeeDO.getEmail());
        saveParam.setTelephone(employeeDO.getPhone());
        if (CollUtil.isNotEmpty(departmentIds)) {
            List<Integer> leaderDepartmentIds = new ArrayList<>();
            for (Long departmentId : departmentIds) {
                leaderDepartmentIds.add(isLeaderDepartmentIds != null && isLeaderDepartmentIds.contains(departmentId) ? 1 : 0);
            }
            saveParam.setIs_leader_in_dept(leaderDepartmentIds);
        }
        saveParam.setDirect_leader(directLeader == null ? null : List.of(directLeader));
        saveParam.setEnable(Boolean.FALSE.equals(employeeDO.getEnabled()) ? 0 : 1);
        saveParam.setTo_invite(true);
        saveParam.setNickname(employeeDO.getNickName());
        saveParam.setAddress(employeeDO.getAddress());
        saveParam.setMain_department(mainDepartment);

        return saveParam;
    }

    static SysSyncSaveBO initSyncSaveBO(Long employeeId, String corpid) {
        SysSyncSaveBO sysSyncSaveBO = new SysSyncSaveBO();
        sysSyncSaveBO.setDataType(SyncDataType.EMPLOYEE.name());
        sysSyncSaveBO.setDataKey(employeeId.toString());
        sysSyncSaveBO.setSyncOuter(true);
        sysSyncSaveBO.setOuterApp(corpid);
        sysSyncSaveBO.setSyncDataTime(LocalDateTime.now());
        sysSyncSaveBO.setManual(true);
        sysSyncSaveBO.setSyncSuccess(false);

        return sysSyncSaveBO;
    }
}
