package com.elitescloud.cloudt.system.service.alert;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.nacos.common.codec.Base64;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.cloudt.system.constant.MsgSendStateEnum;
import com.elitescloud.cloudt.system.constant.SysAlertType;
import com.elitescloud.cloudt.system.model.vo.resp.extend.AlertConfigDingdingRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.AlertConfigDingdingSaveVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.Serial;
import java.io.Serializable;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 钉钉机器人.
 *
 * <p>
 * <a href = "https://open.dingtalk.com/document/orgapp/custom-bot-to-send-group-chat-messages">开发指南</a>，
 * <a href = "https://open.dingtalk.com/document/orgapp/custom-robots-send-group-messages">API文档</a>
 *
 * @author Kaiser（wang shao）
 * @date 2025/10/22 周三
 */
public class DingdingAlertProvider extends AbstractAlertProvider<AlertConfigDingdingSaveVO, AlertConfigDingdingRespVO> {
    private static final Logger logger = LoggerFactory.getLogger(DingdingAlertProvider.class);

    private final RestTemplate restTemplate;

    public DingdingAlertProvider(RedisUtils redisUtils) {
        super(redisUtils);
        this.restTemplate = RestTemplateFactory.instance();
    }

    @Override
    public SysAlertType alertType() {
        return SysAlertType.DINGDING;
    }

    @Override
    public String toString(AlertConfigDingdingSaveVO saveVO) {
        if (saveVO == null) {
            return null;
        }
        Assert.notEmpty(saveVO.getWebhookUrls(), "地址不能为空");

        var cfg = new Cfg(saveVO.getWebhookUrls(), saveVO.getTmplContent());
        cfg.setSecret(saveVO.getSecret());

        // 去重配置
        boolean deduplicate = ObjectUtil.defaultIfNull(saveVO.getDeduplicate(), false);
        cfg.setDeduplicate(deduplicate);
        cfg.setDeduplicateFields(saveVO.getDeduplicateFields());
        cfg.setDeduplicateIntervals(ObjectUtil.defaultIfNull(saveVO.getDeduplicateIntervals(), 0));
        if (deduplicate) {
            Assert.notEmpty(cfg.getDeduplicateFields(), "请选择去重字段");
            Assert.isTrue(cfg.getDeduplicateIntervals() > 0, "去重间隔必须大于0");
        }

        return JSONUtil.toJsonString(cfg);
    }

    @Override
    public AlertConfigDingdingRespVO parse(String cfgJson) {
        if (CharSequenceUtil.isBlank(cfgJson)) {
            return null;
        }

        var cfg = this.parseCfg(cfgJson);

        AlertConfigDingdingRespVO respVO = new AlertConfigDingdingRespVO();
        respVO.setWebhookUrls(cfg.getWebhookUrlMap());
        respVO.setTmplContent(cfg.getTmplContent());
        respVO.setSecret(cfg.getSecret());

        // 去重
        boolean deduplicate = ObjectUtil.defaultIfNull(cfg.getDeduplicate(), false);
        respVO.setDeduplicate(deduplicate);
        respVO.setDeduplicateFields(ObjectUtil.defaultIfNull(cfg.getDeduplicateFields(), Collections.emptyList()));
        respVO.setDeduplicateIntervals(ObjectUtil.defaultIfNull(cfg.getDeduplicateIntervals(), 0));

        return respVO;
    }

    @Override
    public boolean send(String cfgJson, String content, String category) {
        if (CharSequenceUtil.isBlank(cfgJson)) {
            logger.info("企业微信配置内容为空");
            return false;
        }
        var cfg = this.parseCfg(cfgJson);

        // 发送
        return this.execute(cfg, content, category);
    }

    @Override
    public boolean sendByTmpl(String cfgJson, Map<String, Object> tmplParams, String category) {
        if (CharSequenceUtil.isBlank(cfgJson)) {
            logger.info("企业微信配置内容为空");
            return false;
        }
        var cfg = this.parseCfg(cfgJson);

        var tmplContent = cfg.getTmplContent();
        if (CharSequenceUtil.isBlank(tmplContent)) {
            logger.info("模板内容未配置：{}", cfgJson);
            return false;
        }

        // 判重
        if (isDuplicate(cfg, tmplParams, category)) {
            logger.info("消息重复：{}", JSONUtil.toJsonString(cfgJson));
            return false;
        }

        var content = StrUtil.format(tmplContent, tmplParams, true);

        // 发送
        return this.execute(cfg, content, category);
    }

    private boolean execute(Cfg cfg, String content, String category) {
        // 根据分类获取
        List<String> urls = CollUtil.isEmpty(cfg.getWebhookUrlMap()) ? null :
                cfg.getWebhookUrlMap().get(CharSequenceUtil.blankToDefault(category, CATEGORY_DEFAULT));
        if (urls != null) {
            urls = urls.stream().filter(StringUtils::hasText).collect(Collectors.toList());
        }

        if (CollUtil.isEmpty(urls)) {
            logger.info("钉钉地址配置未配置：{}，{}", category, JSONUtil.toJsonString(cfg));
            return false;
        }

        // 限制字数
//        content = CharSequenceUtil.sub(content, 0, 3000);

        for (String webhookUrl : urls) {
            this.execute(webhookUrl, cfg.getSecret(), content);
        }

        return true;
    }

    private void execute(String url, String secret, String content) {
        // 秘钥签名
        if (StrUtil.isNotBlank(secret)) {
            Long timestamp = System.currentTimeMillis();
            String sign = null;
            try {
                String stringToSign = timestamp + "\n" + secret;
                Mac mac = Mac.getInstance("HmacSHA256");
                mac.init(new SecretKeySpec(secret.getBytes(StandardCharsets.UTF_8), "HmacSHA256"));
                byte[] signData = mac.doFinal(stringToSign.getBytes(StandardCharsets.UTF_8));
                sign = URLEncoder.encode(new String(Base64.encodeBase64(signData)), StandardCharsets.UTF_8);
            } catch (Exception e) {
                logger.error("签名异常：", e);
                throw new BusinessException("签名异常：" + e.getMessage());
            }
            url = url + "&timestamp=" + timestamp + "&sign=" + sign;
        }

        logger.info("【钉钉机器人】发送预警：{}, {}", url, content);

        Map<String, Object> body = new HashMap<>();
        body.put("msgtype", "markdown");
        body.put("markdown", Map.of("title", "预警提醒", "text", content));

        MsgSendStateEnum state = MsgSendStateEnum.FAILED;
        String msg = "发送失败";
        String receipt = null;
        ResponseEntity<String> resp = null;
        try {
            resp = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(body), String.class);
            receipt = resp.getBody();
            if (resp.getStatusCode().is2xxSuccessful()) {
                logger.info("发送成功！");
            }
        } catch (RestClientException e) {
            logger.error("发送预警异常：", e);
            return;
        }

        logger.info("发送失败：{}, {}", resp.getStatusCodeValue(), resp.getBody());
    }

    private Cfg parseCfg(String cfgJson) {
        var cfg = JSONUtil.json2Obj(cfgJson, Cfg.class);
        if (CollUtil.isEmpty(cfg.getWebhookUrlMap()) && CollUtil.isNotEmpty(cfg.getWebhookUrls())) {
            // 兼容性升级
            cfg.setWebhookUrlMap(Map.of(CATEGORY_DEFAULT, cfg.getWebhookUrls()));
        }
        return cfg;
    }

    static class Cfg extends BaseCfg implements Serializable {
        @Serial
        private static final long serialVersionUID = 4537456626636877998L;
        private List<String> webhookUrls;
        private Map<String, List<String>> webhookUrlMap;

        /**
         * 秘钥
         */
        private String secret;

        public Cfg(Map<String, List<String>> webhookUrlMap, String tmplContent) {
            super(tmplContent);
            this.webhookUrlMap = webhookUrlMap;
        }

        public Cfg() {
            super(null);
        }

        public String getSecret() {
            return secret;
        }

        public void setSecret(String secret) {
            this.secret = secret;
        }

        public Map<String, List<String>> getWebhookUrlMap() {
            return webhookUrlMap;
        }

        public void setWebhookUrlMap(Map<String, List<String>> webhookUrlMap) {
            this.webhookUrlMap = webhookUrlMap;
        }

        public List<String> getWebhookUrls() {
            return webhookUrls;
        }

        public void setWebhookUrls(List<String> webhookUrls) {
            this.webhookUrls = webhookUrls;
        }
    }
}
