package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.common.param.CodeNameParam;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 基础数据初始化提供者工厂.
 *
 * @author Kaiser（wang shao）
 * @date 2024/1/18
 */
public class BasicDataInitProviderFactory {

    private static final List<BasicDataInitProviderFactory.Group> EXPORT_PROVIDER_GROUP = new ArrayList<>(16);
    private static final List<BasicDataInitProviderFactory.Group> IMPORT_PROVIDER_GROUP = new ArrayList<>(16);

    private BasicDataInitProviderFactory() {
    }

    static {
        // 导出
        EXPORT_PROVIDER_GROUP.add(new Group("app", "应用信息", 1
                , new PlatformAppInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("menu", "系统菜单", 2
                , new PlatformMenusInitProvider(), new PlatformAdminMenusInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("api", "API接口", 3
                , new PlatformApiManageInitProvider(), new PlatformApiParameterInitProvider(), new PlatformMenusApiInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("udc", "UDC", 4
                , new PlatformUdcInitProvider(), new PlatformUdcValueInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("nextNumberRule", "发号器", 5
                , new PlatformNextNumberInitProvider(), new PlatformNumberRuleInitProvider(), new PlatformNumberRuleDtlInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("area", "行政区域", 6
                , new PlatformAreaInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("currency", "货币", 7
                , new PlatformCurrencyInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("currencyRate", "汇率", 8
                , new CurrencyRateInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("taxRate", "税率", 9
                , new TaxRateInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("setting", "系统设置", 10
                , new SettingInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("msgTmpl", "消息模板", 11
                , new MsgTemplateInitProvider(), new MsgTemplateConfigInitProvider()));
        EXPORT_PROVIDER_GROUP.add(new Group("excelTmpl", "导入导出模板", 12
                , new TmplInitProvider()));

        // 导入
        IMPORT_PROVIDER_GROUP.add(new Group("app", "应用信息", 1
                , new PlatformAppInitProvider()));
        IMPORT_PROVIDER_GROUP.add(new Group("menu", "系统菜单", 2
                , new PlatformMenusInitProvider(), new PlatformAdminMenusInitProvider()));
        IMPORT_PROVIDER_GROUP.add(new Group("api", "API接口", 3
                , new PlatformApiManageInitProvider().thenImport(new PlatformApiParameterInitProvider(), new PlatformMenusApiInitProvider())));
        IMPORT_PROVIDER_GROUP.add(new Group("udc", "UDC", 4
                , new PlatformUdcInitProvider(), new PlatformUdcValueInitProvider()));
        IMPORT_PROVIDER_GROUP.add(new Group("nextNumberRule", "发号器", 5
                , new PlatformNextNumberInitProvider(), new PlatformNumberRuleInitProvider().thenImport(new PlatformNumberRuleDtlInitProvider())));
        IMPORT_PROVIDER_GROUP.add(new Group("area", "行政区域", 6
                , new PlatformAreaInitProvider()));
        IMPORT_PROVIDER_GROUP.add(new Group("currency", "货币", 7
                , new PlatformCurrencyInitProvider()));
        IMPORT_PROVIDER_GROUP.add(new Group("currencyRate", "汇率", 8
                , new CurrencyRateInitProvider()));
        IMPORT_PROVIDER_GROUP.add(new Group("taxRate", "税率", 9
                , new TaxRateInitProvider()));
        IMPORT_PROVIDER_GROUP.add(new Group("setting", "系统设置", 10
                , new SettingInitProvider()));
        IMPORT_PROVIDER_GROUP.add(new Group("msgTmpl", "消息模板", 11
                , new MsgTemplateInitProvider().thenImport(new MsgTemplateConfigInitProvider())));
        IMPORT_PROVIDER_GROUP.add(new Group("excelTmpl", "导入导出模板", 12
                , new TmplInitProvider()));
    }

    /**
     * 获取导出分组列表
     *
     * @return 分组列表
     */
    public static List<CodeNameParam> getExportProviderGroupList() {
        return EXPORT_PROVIDER_GROUP.stream()
                .map(group -> new CodeNameParam(group.getCode(), group.getName()))
                .collect(Collectors.toList());
    }

    /**
     * 获取导入分组列表
     *
     * @return 分组列表
     */
    public static List<CodeNameParam> getImportProviderGroupList() {
        return IMPORT_PROVIDER_GROUP.stream()
                .map(group -> new CodeNameParam(group.getCode(), group.getName()))
                .collect(Collectors.toList());
    }

    /**
     * 获取初始化导出提供者列表
     *
     * @param groupCodes 分组编码列表
     * @return 初始化提供者列表
     */
    public static List<List<AbstractBasicDataInitProvider>> getExportProviderList(Collection<String> groupCodes) {
        var isAllGroup = CollUtil.isEmpty(groupCodes);

        return EXPORT_PROVIDER_GROUP.stream()
                .filter(t -> isAllGroup || groupCodes.contains(t.getCode()))
                .map(t -> Collections.unmodifiableList(t.getProviderList()))
                .collect(Collectors.toList());
    }

    /**
     * 获取初始化导入提供者列表
     *
     * @param groupCodes 分组编码列表
     * @return 初始化提供者列表
     */
    public static List<List<AbstractBasicDataInitProvider>> getImportProviderList(Collection<String> groupCodes) {
        var isAllGroup = CollUtil.isEmpty(groupCodes);

        return IMPORT_PROVIDER_GROUP.stream()
                .filter(t -> isAllGroup || groupCodes.contains(t.getCode()))
                .map(t -> Collections.unmodifiableList(t.getProviderList()))
                .collect(Collectors.toList());
    }

    private static class Group {

        /**
         * 分组编码
         */
        private final String code;
        /**
         * 分组名称
         */
        private final String name;
        /**
         * 分组排序
         */
        private final int order;
        /**
         * 提供者列表
         */
        private final List<AbstractBasicDataInitProvider> providerList;

        public Group(String code, String name, int order, AbstractBasicDataInitProvider... provider) {
            this.code = code;
            this.name = name;
            this.order = order;

            Assert.notBlank(code, "分组编码为空");
            Assert.notBlank(name, "分组名称为空");
            Assert.notEmpty(provider, "提供者为空");

            this.providerList = Arrays.stream(provider).collect(Collectors.toList());
            this.checkProviders();
        }

        public String getCode() {
            return code;
        }


        public String getName() {
            return name;
        }

        public int getOrder() {
            return order;
        }

        public List<AbstractBasicDataInitProvider> getProviderList() {
            return providerList;
        }

        private void checkProviders() {
            Set<String> existsNames = new HashSet<>(providerList.size());
            for (AbstractBasicDataInitProvider provider : providerList) {
                if (existsNames.contains(provider.typeName())) {
                    throw new IllegalArgumentException("分组[" + code + "]下存在重复的类型[" + provider.typeName() + "]");
                }
                existsNames.add(provider.typeName());
            }
        }
    }
}
