package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.DataRelationConvert;
import com.elitescloud.cloudt.system.model.bo.DataSelectorSimpleBO;
import com.elitescloud.cloudt.system.model.vo.query.datarelation.DataRelationPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.datarelation.DataRelationEditRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.datarelation.DataRelationPagedRespVO;
import com.elitescloud.cloudt.system.model.vo.save.datarelation.DataRelationSaveVO;
import com.elitescloud.cloudt.system.service.DataRelationMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysDataRelationCategoryDO;
import com.elitescloud.cloudt.system.service.model.entity.SysDataRelationDO;
import com.elitescloud.cloudt.system.service.repo.BusinessObjectRepoProc;
import com.elitescloud.cloudt.system.service.repo.DataRelationCategoryRepoProc;
import com.elitescloud.cloudt.system.service.repo.DataRelationRepoProc;
import com.elitescloud.cloudt.system.service.repo.DataSelectorRepoProc;
import com.google.common.base.Functions;
import io.jsonwebtoken.lang.Assert;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/14
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class DataRelationMngServiceImpl implements DataRelationMngService {
    @Autowired
    private DataRelationRepoProc repoProc;
    @Autowired
    private DataRelationCategoryRepoProc categoryRepoProc;
    @Autowired
    private BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    private DataSelectorRepoProc dataSelectorRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(DataRelationSaveVO saveVO) {
        try {
            checkForSave(saveVO);
        } catch (Exception e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // 保存基本信息
        var data = this.saveDataRelation(saveVO);
        // 保存分类
        this.saveCat(data, saveVO.getCats());

        return ApiResult.ok(data.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> delete(List<Long> ids) {
        Assert.notEmpty(ids, "ID为空");

        categoryRepoProc.deleteByDrIds(ids);
        repoProc.delete(ids);

        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id, Boolean enabled) {
        Assert.notNull(id, "ID为空");
        Assert.notNull(enabled, "启用状态为空");

        repoProc.updateEnabled(id, enabled);

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<DataRelationEditRespVO> get(Long id) {
        Assert.notNull(id, "ID为空");
        var drDO = repoProc.get(id);
        if (drDO == null) {
            return ApiResult.fail("数据不存在");
        }
        var respVO = DataRelationConvert.INSTANCE.do2EditRespVO(drDO);

        var boCodes = Stream.of(respVO.getBoCode(), respVO.getRefBoCode()).filter(StringUtils::hasText).collect(Collectors.toSet());
        if (!boCodes.isEmpty()) {
            var boCodeNameMap = businessObjectRepoProc.getCodeAndNames(boCodes);
            respVO.setBoName(boCodeNameMap.get(respVO.getBoCode()));
            respVO.setRefBoName(boCodeNameMap.get(respVO.getRefBoCode()));
        }

        // 获取分类
        var catList = categoryRepoProc.listByDrCode(drDO.getCode()).stream()
                .map(DataRelationConvert.INSTANCE::do2EditRespVO)
                .collect(Collectors.toList());
        respVO.setCats(catList);

        // 获取前端配置
        if (StringUtils.hasText(respVO.getDataSelectorCode())) {
            respVO.setDataSelectorName(dataSelectorRepoProc.getName(respVO.getDataSelectorCode()));
        }

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<DataRelationPagedRespVO>> page(DataRelationPageQueryVO queryVO) {
        var pageData = repoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        // 业务对象信息
        var boCodes = pageData.getRecords().stream().flatMap(t -> Stream.of(t.getBoCode(), t.getRefBoCode()))
                .filter(StringUtils::hasText).collect(Collectors.toSet());
        Map<String, String> boCodeNameMap = boCodes.isEmpty() ? Collections.emptyMap() :
                businessObjectRepoProc.getCodeAndNames(boCodes);
        // 数据选择器信息
        var dataSelectorsCodes = pageData.getRecords().stream().map(DataRelationPagedRespVO::getDataSelectorCode).filter(StringUtils::hasText).collect(Collectors.toSet());
        Map<String, DataSelectorSimpleBO> dataSelectorMap = dataSelectorsCodes.isEmpty() ? Collections.emptyMap() :
                dataSelectorRepoProc.listByCodes(dataSelectorsCodes).stream().collect(Collectors.toMap(DataSelectorSimpleBO::getSelectorCode, Functions.identity(), (t1, t2) -> t1));
        pageData.each(t -> {
            t.setBoName(boCodeNameMap.get(t.getBoCode()));
            t.setRefBoName(boCodeNameMap.get(t.getRefBoCode()));

            ObjUtil.ifNotNull(dataSelectorMap.get(t.getDataSelectorCode()), ds -> {
                t.setDataSelectorName(ds.getSelectorName());
                t.setComponent(ds.getComponent());
                t.setComponentCode(ds.getComponentCode());
                t.setDataApi(ds.getDataApi());
                t.setDataApiMethod(ds.getDataApiMethod());
            });
        });

        return ApiResult.ok(pageData);
    }

    private void saveCat(SysDataRelationDO drDO, List<DataRelationSaveVO.Cat> cats) {
        if (CollUtil.isEmpty(cats)) {
            categoryRepoProc.deleteByDrCode(drDO.getCode());
            return;
        }
        Map<String, SysDataRelationCategoryDO> existsMap = categoryRepoProc.listByDrCode(drDO.getCode()).stream()
                .collect(Collectors.toMap(SysDataRelationCategoryDO::getCatCode, Function.identity(), (t1, t2) -> t1));

        List<SysDataRelationCategoryDO> categoryDoList = new ArrayList<>();
        Set<String> savedCodes = new HashSet<>(cats.size());
        SysDataRelationCategoryDO categoryDO = null;
        int i = 0;
        for (var cat : cats) {
            savedCodes.add(cat.getCatCode());

            categoryDO = existsMap.get(cat.getCatCode());
            if (categoryDO == null) {
                categoryDO = new SysDataRelationCategoryDO();
                categoryDO.setDrCode(drDO.getCode());
                categoryDO.setCatCode(cat.getCatCode());
            }
            categoryDO.setCatName(cat.getCatName());
            categoryDO.setEnabled(cat.getEnabled());
            categoryDO.setSortNo(i++);

            categoryDoList.add(categoryDO);
        }
        categoryRepoProc.save(categoryDoList);

        List<Long> toDelIds = new ArrayList<>(cats.size());
        for (var entry : existsMap.entrySet()) {
            if (savedCodes.contains(entry.getKey())) {
                continue;
            }
            toDelIds.add(entry.getValue().getId());
        }
        if (!toDelIds.isEmpty()) {
            categoryRepoProc.delete(toDelIds);
        }
    }

    private SysDataRelationDO saveDataRelation(DataRelationSaveVO saveVO) {
        SysDataRelationDO relationDO = null;
        if (saveVO.getId() == null) {
            relationDO = new SysDataRelationDO();
        } else {
            relationDO = repoProc.get(saveVO.getId());
            Assert.notNull(relationDO, "修改的数据不存在");
        }
        DataRelationConvert.INSTANCE.copy(saveVO, relationDO);

        repoProc.save(relationDO);
        return relationDO;
    }

    private void checkForSave(DataRelationSaveVO saveVO) {
        Assert.hasText(saveVO.getCode(), "数据关系编码为空");
        if (saveVO.getId() == null) {
            if (repoProc.existsCode(saveVO.getCode(), null)) {
                throw new BusinessException("编码" + saveVO.getCode() + "已存在");
            }
        } else {
            var code = repoProc.getCode(saveVO.getId());
            Assert.isTrue(code.equals(saveVO.getCode()), "编码不可修改");
        }
        ObjUtil.ifNull(saveVO.getEnabled(), true, saveVO::setEnabled);

        // 业务对象编码
        var boCodes = Stream.of(saveVO.getBoCode(), saveVO.getRefBoCode()).filter(StringUtils::hasText).collect(Collectors.toSet());
        if (!boCodes.isEmpty()) {
            var existsBusinessCodes = businessObjectRepoProc.filterCodes(Set.of(saveVO.getBoCode(), saveVO.getRefBoCode()));
            if (existsBusinessCodes.size() != boCodes.size()) {
                if (StringUtils.hasText(saveVO.getBoCode())) {
                    Assert.isTrue(existsBusinessCodes.contains(saveVO.getBoCode()), "业务对象编码不存在");
                }
                if (StringUtils.hasText(saveVO.getRefBoCode())) {
                    Assert.isTrue(existsBusinessCodes.contains(saveVO.getRefBoCode()), "关联的业务对象编码不存在");
                }
            }
        }

        // 分类
        if (CollUtil.isNotEmpty(saveVO.getCats())) {
            Set<String> existsCodes = new HashSet<>(saveVO.getCats().size());
            List<DataRelationSaveVO.Cat> cats = new ArrayList<>(saveVO.getCats().size());
            for (var cat : saveVO.getCats()) {
                if (existsCodes.contains(cat.getCatCode())) {
                    throw new BusinessException(cat.getCatCode() + "已存在");
                }
                existsCodes.add(cat.getCatCode());

                ObjUtil.ifNull(cat.getEnabled(), true, cat::setEnabled);
                cats.add(cat);
            }
            saveVO.setCats(cats);
        }
    }
}
