package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.MenuConvert;
import com.elitescloud.cloudt.system.dto.resp.SysPlatformMenusDTO;
import com.elitescloud.cloudt.system.model.bo.MenuBO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonMenuTreeNodeRespVO;
import com.elitescloud.cloudt.system.service.MenuQueryService;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.manager.PermissionQueryManager;
import com.elitescloud.cloudt.system.service.model.bo.AppBO;
import com.elitescloud.cloudt.system.service.repo.MenuRepoProc;
import com.google.common.base.Functions;
import io.jsonwebtoken.lang.Assert;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/4/28
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class MenuQueryServiceImpl implements MenuQueryService {

    @Autowired
    private MenuRepoProc menuRepoProc;
    @Autowired
    private PermissionQueryManager permissionQueryManager;

    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;

    @Override
    public ApiResult<List<CommonMenuTreeNodeRespVO>> menuTreeSystem(String appCode, Boolean withSys, Boolean withBusiness,
                                                                    Boolean withMenu, Boolean withAction, Boolean defTree, Boolean tree) {
        var appMap = permissionQueryManager.tenantApps(appCode);
        if (appMap.isEmpty()) {
            // 没有授权的应用
            return ApiResult.ok(Collections.emptyList());
        }

        withMenu = withMenu == null || withMenu;
        withAction = withMenu && withAction != null && withAction;
        defTree = defTree != null && defTree;

        // 查询惨淡列表
        var menuList = this.queryOriginalMenus(appMap, ObjUtil.defaultIfNull(withSys, false)
                , ObjUtil.defaultIfNull(withBusiness, true), withMenu, withAction);
        if (menuList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 加入应用8
        if (CharSequenceUtil.isBlank(appCode)) {
            var appList = this.convertApp2MenuTreeNode(appMap);
            menuList.addAll(appList);
        }

        // 转为树
        TreeDataUtil<CommonMenuTreeNodeRespVO> treeDataUtil = new TreeDataUtil<>(menuList, CommonMenuTreeNodeRespVO::getId, CommonMenuTreeNodeRespVO::getParentId
                , CommonMenuTreeNodeRespVO::setChildren, Comparator.comparingInt(CommonMenuTreeNodeRespVO::getSortNo));
        List<CommonMenuTreeNodeRespVO> resultList = tree == null || tree ? (List<CommonMenuTreeNodeRespVO>) treeDataUtil.getRoots()
                : CollectionUtil.expandTree((List<CommonMenuTreeNodeRespVO>) treeDataUtil.getRoots(), CommonMenuTreeNodeRespVO::getChildren);

        return ApiResult.ok(resultList);
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<SysPlatformMenusDTO> getMenuInfo(String menuCode) {
        Assert.hasText(menuCode, "菜单编码为空");
        var menuDO = menuRepoProc.getByMenuCode(menuCode);
        if (menuDO == null) {
            return ApiResult.ok();
        }

        var menuDto = MenuConvert.INSTANCE.do2Dto(menuDO);
        return ApiResult.ok(menuDto);
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<List<SysPlatformMenusDTO>> getMenuInfo(Set<String> menuCodes) {
        Assert.notEmpty(menuCodes, "菜单编码为空");

        var menuList = menuRepoProc.listByMenuCode(menuCodes).stream()
                .map(MenuConvert.INSTANCE::do2Dto)
                .collect(Collectors.toList());
        return ApiResult.ok(menuList);
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<List<IdCodeNameParam>> getMenuWithParents(String menuCode) {
        Assert.hasText(menuCode, "菜单编码为空");

        List<String> menuCodeList = new ArrayList<>(8);
        menuCodeList.add(menuCode);
        String parentCode = menuRepoProc.getParentCode(menuCode);
        while (StringUtils.hasText(parentCode)) {
            menuCode = parentCode;
            menuCodeList.add(parentCode);
            parentCode = menuRepoProc.getParentCode(menuCode);
        }
        var menuMap = menuRepoProc.getMenuCodeAndNames(menuCodeList).stream()
                .collect(Collectors.toMap(IdCodeNameParam::getCode, Functions.identity(), (t1, t2) -> t1));
        if (menuMap.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        List<IdCodeNameParam> menuList = new ArrayList<>(menuCodeList.size());
        for (int i = menuCodeList.size() - 1; i >= 0; i--) {
            var menu = menuMap.get(menuCodeList.get(i));
            if (menu != null) {
                menuList.add(menu);
            }
        }
        return ApiResult.ok(menuList);
    }

    private List<CommonMenuTreeNodeRespVO> convertApp2MenuTreeNode(Map<String, AppBO> appMap) {
        return appMap.values().stream()
                .map(t -> {
                    CommonMenuTreeNodeRespVO respVO = new CommonMenuTreeNodeRespVO();
                    respVO.setNodeType(MenuTreeNodeType.APP.getValue());
                    respVO.setAppCode(t.getAppCode());
                    respVO.setMenusIcon(t.getIcon());
                    respVO.setHidden(false);
                    respVO.setEnabled(true);
                    respVO.setId(t.getId());
                    respVO.setCode(t.getAppCode());
                    respVO.setName(t.getAppName());
                    respVO.setSortNo(ObjUtil.defaultIfNull(t.getAppOrder(), 0));

                    return respVO;
                }).collect(Collectors.toList());
    }

    private List<CommonMenuTreeNodeRespVO> queryOriginalMenus(Map<String, AppBO> appMap, boolean withSys, boolean withBusiness,
                                                              boolean withMenu, boolean withAction) {
        var menuBoList = tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryMenu(appMap.keySet(), withSys, withBusiness, !withAction, true));
        if (menuBoList.isEmpty()) {
            return Collections.emptyList();
        }
        var menuCodeIdMap = menuBoList.stream().collect(Collectors.toMap(MenuBO::getMenusCode, MenuBO::getId, (t1, t2) -> t1));
        return menuBoList.stream()
                .filter(t -> {
                    if (withMenu) {
                        return true;
                    }
                    // 没有路由就是分组
                    return CharSequenceUtil.isBlank(t.getMenusRoute());
                })
                .map(t -> {
                    CommonMenuTreeNodeRespVO respVO = new CommonMenuTreeNodeRespVO();
                    respVO.setMenusType(t.getMenusType());
                    respVO.setNodeType(t.getNodeType());
                    respVO.setAppCode(t.getMenusAppCode());
                    respVO.setMenusIcon(t.getMenusIcon());
                    respVO.setMenusRoute(t.getMenusRoute());
                    respVO.setHidden(Boolean.FALSE.equals(t.getDisplay()));
                    respVO.setEnabled(Boolean.TRUE.equals(t.getMenusState()));
                    respVO.setId(t.getId());
                    respVO.setCode(t.getMenusCode());
                    respVO.setName(t.getMenusName());
                    respVO.setSortNo(ObjectUtil.defaultIfNull(t.getMenusOrder(), 0));
                    if (StringUtils.hasText(t.getMenusParentCode())) {
                        respVO.setParentId(menuCodeIdMap.get(t.getMenusParentCode()));
                        respVO.setParentCode(t.getMenusParentCode());
                    } else {
                        // 顶级菜单，则设置上级为应用app
                        respVO.setParentCode(t.getMenusAppCode());
                        com.elitescloud.boot.util.ObjUtil.ifNotNull(appMap.get(t.getMenusAppCode()), app -> respVO.setParentId(app.getId()));
                    }

                    return respVO;
                })
                .collect(Collectors.toList());
    }
}
