package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.convert.UnifyQueryConvert;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.model.vo.query.sys.UnifyQueryPagedQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.UnifyQueryDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.UnifyQueryPagedRespVO;
import com.elitescloud.cloudt.system.service.SysUnifyQueryMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysUnifyQueryRecordDO;
import com.elitescloud.cloudt.system.service.repo.AppRepoProc;
import com.elitescloud.cloudt.system.service.repo.BusinessObjectRepoProc;
import com.elitescloud.cloudt.system.service.repo.UnifyQueryRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Functions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/6/7
 */
@Service
public class SysUnifyQueryMngServiceImpl implements SysUnifyQueryMngService {

    @Autowired
    private UnifyQueryRepoProc repoProc;
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;

    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;

    @Override
    public ApiResult<PagingVO<UnifyQueryPagedRespVO>> pageMng(UnifyQueryPagedQueryVO queryVO) {
        var pageData = repoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData.map(UnifyQueryConvert.INSTANCE::do2PagedRespVO));
        }

        var userIds = pageData.stream().map(SysUnifyQueryRecordDO::getUserId).filter(ObjUtil::isNotNull).collect(Collectors.toSet());
        var businessObjectCodes = pageData.stream().map(SysUnifyQueryRecordDO::getBusinessObjectCode).filter(ObjUtil::isNotNull).collect(Collectors.toSet());

        Map<Long, SysUserBasicDTO> userMap = new HashMap<>(pageData.size());
        Map<String, String> businessObjectMap = new HashMap<>(pageData.size());
        Map<String, IdCodeNameParam> appMap = new HashMap<>(pageData.size());
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            if (!userIds.isEmpty()) {
                userMap.putAll(userRepoProc.getBasicDto(userIds).stream().collect(Collectors.toMap(SysUserBasicDTO::getId, Functions.identity(), (t1, t2) -> t1)));
            }
            if (!businessObjectMap.isEmpty()) {
                businessObjectMap.putAll(businessObjectRepoProc.getCodeAndNames(businessObjectCodes));
            }
            appMap.putAll(appRepoProc.allParams(null).stream().collect(Collectors.toMap(IdCodeNameParam::getCode, Functions.identity(), (t1, t2) -> t1)));
            return null;
        });

        var pageVO = pageData.map(t -> {
            var vo = UnifyQueryConvert.INSTANCE.do2PagedRespVO(t);
            var user = userMap.get(t.getUserId());
            if (user != null) {
                vo.setUsername(user.getUsername());
                vo.setFullName(user.getFullName());
            }

            vo.setBusinessObjectName(businessObjectMap.get(t.getBusinessObjectCode()));

            var app = appMap.get(t.getAppCode());
            if (app != null) {
                vo.setAppName(app.getName());
            }

            return vo;
        });
        return ApiResult.ok(pageVO);
    }

    @Override
    public ApiResult<UnifyQueryDetailRespVO> get(Long id) {
        Assert.notNull(id, "ID为空");

        var recordDO = repoProc.get(id);
        if (recordDO == null) {
            return ApiResult.fail("数据不存在");
        }

        var detailVO = UnifyQueryConvert.INSTANCE.do2DetailRespVO(recordDO);
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            if (recordDO.getUserId() != null) {
                var user = userRepoProc.getBasicDto(recordDO.getUserId());
                com.elitescloud.boot.util.ObjUtil.ifNotNull(user, u -> {
                    detailVO.setUsername(u.getUsername());
                    detailVO.setFullName(u.getFullName());
                });

                if (StringUtils.hasText(recordDO.getBusinessObjectCode())) {
                    detailVO.setBusinessObjectName(businessObjectRepoProc.getName(recordDO.getBusinessObjectCode()));
                }

                if (StringUtils.hasText(recordDO.getAppCode())) {
                    detailVO.setAppName(appRepoProc.getNameByCode(recordDO.getAppCode()));
                }
            }
            return null;
        });

        if (StringUtils.hasText(recordDO.getBusinessFields())) {
            detailVO.setBusinessFieldsList(Arrays.asList(recordDO.getBusinessFields().split(",")));
        }
        detailVO.setBusinessParamsMap(JSONUtil.json2Obj(recordDO.getBusinessParams(), new TypeReference<>() {
        }));
        detailVO.setSqlParamsMap(JSONUtil.json2Obj(recordDO.getSqlParams(), new TypeReference<>() {
        }));
        return ApiResult.ok(detailVO);
    }
}
