package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.dto.SysUserTerminalSaveDTO;
import com.elitescloud.cloudt.system.model.entity.SysUserTerminalDO;
import com.elitescloud.cloudt.system.service.SysUserTerminalService;
import com.elitescloud.cloudt.system.service.repo.SysUserTerminalRepo;
import com.elitescloud.cloudt.system.service.repo.SysUserTerminalRepoProc;
import com.elitescloud.cloudt.system.service.repo.old.SysUserRepoProc;
import com.elitescloud.cloudt.system.service.vo.save.SysUserTerminalSaveVO;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2021/09/28
 */
@Data
@Slf4j
@AllArgsConstructor
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT_USER)
@TenantOrgTransaction(useTenantOrg = false)
public class SysUserTerminalServiceImpl implements SysUserTerminalService {

    private final SysUserTerminalRepo userTerminalRepo;
    private final SysUserTerminalRepoProc userTerminalRepoProc;
    private final SysUserRepoProc userRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.TENANT)
    @Override
    public ApiResult<Long> save(SysUserTerminalSaveVO saveVO) {
        updateByUser(saveVO.getUserId(), ObjectUtil.defaultIfNull(saveVO.getTerminals(), Collections.emptySet()));
        return ApiResult.ok(saveVO.getUserId());
    }

    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.TENANT)
    @Override
    public ApiResult<Long> save(SysUserTerminalSaveDTO saveParam) {
        updateByUser(saveParam.getUserId(), ObjectUtil.defaultIfNull(saveParam.getTerminals(), Collections.emptySet()));
        return ApiResult.ok(saveParam.getUserId());
    }

    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.TENANT)
    @Override
    public ApiResult<Long> save(Long userId, Set<Terminal> terminals) {
        Assert.notNull(userId, "用户ID为空");
        updateByUser(userId, ObjectUtil.defaultIfNull(terminals, Collections.emptySet()));
        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.TENANT)
    @Override
    public ApiResult<Long> saveForAdd(SysUserTerminalSaveDTO saveParam) {
        var terminals = saveParam.getTerminals();
        var oldTerminals = userTerminalRepoProc.getTerminalByUserId(saveParam.getUserId());
        if (!oldTerminals.isEmpty()) {
            terminals = terminals.stream().filter(t -> !oldTerminals.contains(t)).collect(Collectors.toSet());
        }

        if (terminals.isEmpty()) {
            return ApiResult.ok(saveParam.getUserId());
        }

        add(saveParam.getUserId(), terminals);

        return ApiResult.ok(saveParam.getUserId());
    }

    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.TENANT)
    @Override
    public ApiResult<Long> saveForAdd(Long userId, Set<Terminal> terminals) {
        Assert.notNull(userId, "未知用户ID");
        if (CollUtil.isEmpty(terminals)) {
            return ApiResult.fail("终端为空");
        }

        var oldTerminals = userTerminalRepoProc.getTerminalByUserId(userId);
        if (!oldTerminals.isEmpty()) {
            terminals = terminals.stream().filter(t -> !oldTerminals.contains(t)).collect(Collectors.toSet());
        }

        if (terminals.isEmpty()) {
            return ApiResult.ok(userId);
        }

        add(userId, terminals);

        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.TENANT)
    @Override
    public ApiResult<Long> saveForReduce(SysUserTerminalSaveDTO saveParam) {
        var terminals = saveParam.getTerminals().stream().map(Terminal::name).collect(Collectors.toSet());
        if (terminals.isEmpty()) {
            return ApiResult.ok(saveParam.getUserId());
        }

        var userTerminals = userTerminalRepoProc.queryByUserId(saveParam.getUserId());
        if (userTerminals.isEmpty()) {
            return ApiResult.ok(saveParam.getUserId());
        }
        var toDelIds = userTerminals.stream().filter(t -> terminals.contains(t.getTerminalCode())).map(SysUserTerminalDO::getId).collect(Collectors.toList());
        if (!toDelIds.isEmpty()) {
            userTerminalRepoProc.deleteById(toDelIds);
        }
        return ApiResult.ok(saveParam.getUserId());
    }

    @Override
    @TenantTransaction(isolateType = TenantIsolateType.TENANT)
    public ApiResult<List<Terminal>> getByUser(Long userId) {
        var terminals = userTerminalRepoProc.getTerminalByUserId(userId);
        return ApiResult.ok(terminals);
    }

    @Override
    public ApiResult<List<Terminal>> getByUser(String username) {
        var userId = userRepoProc.getIdByUsername(username);
        if (userId == null) {
            return ApiResult.fail("未查询到用户信息");
        }

        var terminals = userTerminalRepoProc.getTerminalByUserId(userId);
        return ApiResult.ok(terminals);
    }

    @Override
    public ApiResult<Map<Long, List<Terminal>>> getByUser(List<Long> userIds) {
        var userTerminal = userTerminalRepoProc.getTerminalByUserId(userIds);
        return ApiResult.ok(userTerminal);
    }

    /**
     * 根据用户
     *
     * @param userId
     * @param terminals
     */
    private void updateByUser(Long userId, Set<Terminal> terminals) {
        var oldDOList = userTerminalRepoProc.queryByUserId(userId);

        if (CollUtil.isNotEmpty(oldDOList)) {
            // 判断是否有需要删除的
            Set<String> finalNewTerminals = terminals.stream().map(Terminal::name).collect(Collectors.toSet());
            var toDelIds = oldDOList.stream()
                    .filter(t -> !finalNewTerminals.contains(t.getTerminalCode()))
                    .map(SysUserTerminalDO::getId)
                    .collect(Collectors.toList());
            if (!toDelIds.isEmpty()) {
                userTerminalRepoProc.deleteById(toDelIds);
            }

            // 筛选已存在的进行忽略
            if (!terminals.isEmpty()) {
                var oldTerminalCode = oldDOList.stream().map(SysUserTerminalDO::getTerminalCode).collect(Collectors.toSet());
                terminals = terminals.stream().filter(t -> !oldTerminalCode.contains(t.name())).collect(Collectors.toSet());
            }
        }

        add(userId, terminals);
    }

    private void add(Long userId, Set<Terminal> terminals) {
        if (!terminals.isEmpty()) {
            var nowTime = LocalDateTime.now();
            var newDataList = terminals.stream().map(t -> {
                var terminalDO = new SysUserTerminalDO();
                terminalDO.setTerminalCode(t.name());
                terminalDO.setUserId(userId);
                terminalDO.setTimeBind(nowTime);
                return terminalDO;
            }).collect(Collectors.toList());
            userTerminalRepo.saveAll(newDataList);
        }
    }
}
