package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.InfinityPlatformConvert;
import com.elitescloud.cloudt.system.model.entity.InfinityPlatformDO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.*;
import com.elitescloud.cloudt.system.service.AuthMethodService;
import com.elitescloud.cloudt.system.service.ThirdApiService;
import com.elitescloud.cloudt.system.service.repo.InfinityPlatformRepo;
import com.elitescloud.cloudt.system.service.repo.InfinityPlatformRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @Auther: Mark
 * @Date: 2024/12/13 13:53
 * @Description:
 */
@Slf4j
@Service
@RequiredArgsConstructor
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class ThirdApiServiceImpl implements ThirdApiService {

    private final AuthMethodService authMethodService;
    private final InfinityPlatformRepo platformRepo;
    private final InfinityPlatformRepoProc platformRepoProc;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    private long currentTenantId() {
        var tenant = tenantClientProvider.getSessionTenant();
        if (tenant == null) {
            tenant = tenantClientProvider.getCurrentTenant();
        }
        return tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId();
    }

    /**
     * 根据查询参数获取第三方API分页数据
     * <p>
     * 此方法首先根据查询参数计算总记录数，然后根据查询参数获取分页数据
     * 如果存在记录，将这些记录转换为分页视图对象并返回；如果没有记录，则返回一个空的分页视图对象
     *
     * @param query 分页查询参数，包含分页信息和筛选条件
     * @return 返回一个包含第三方API分页数据的PagingVO对象
     */
    @Override
    public PagingVO<ThirdApiPagingVO> ThirdApiPageBy(ThirdApiPagingParamVO query) {
        // 计算满足查询条件的总记录数
        long count = platformRepoProc.platformCountBy(query);

        // 如果存在记录，则获取分页数据
        if (count > 0) {
            var list = platformRepoProc.platformPageBy(query);
            // 将分页数据转换为分页视图对象并返回
            return new PagingVO<>(count, InfinityPlatformConvert.INSTANCE.toPageVO(list));
        }
        // 如果没有记录，返回一个空的分页视图对象
        return new PagingVO<>();
    }

    /**
     * 根据参数查询第三方API信息
     * <p>
     * 此方法通过接收一个查询参数对象，从平台仓库中获取符合条件的平台列表，
     * 并将列表转换为第三方API响应对象列表此外，它还负责获取认证方法的名称，
     * 并将其添加到响应对象中，以便于后续使用
     *
     * @param query 查询参数对象，包含查询第三方API所需的参数
     * @return 包含第三方API信息的响应对象列表
     */
    @Override
    public List<ThirdApiRespVO> ThirdApiByParam(ThirdApiQueryParamVO query) {
        // 根据查询参数获取平台列表
        List<InfinityPlatformDO> list = platformRepoProc.platformByParam(query);

        // 将平台列表转换为第三方API响应对象列表
        List<ThirdApiRespVO> platforms = InfinityPlatformConvert.INSTANCE.doToVO(list);

        // 提取所有认证方法代码
        List<String> authMethodCodes =
                platforms.stream().map(ThirdApiRespVO::getAuthMethod).collect(Collectors.toList());

        // 获取认证方法名称映射
        Map<String, String> authMethodNames = authMethodService.fromCode(authMethodCodes);

        // 为每个平台响应对象设置认证方法名称
        for (var platform : platforms) {
            String authMethodName = authMethodNames.getOrDefault(platform.getAuthMethod(), platform.getAuthMethod());
            platform.setAuthMethodName(authMethodName);
        }

        // 返回填充了认证方法名称的平台列表
        return platforms;
    }

    /**
     * 根据ID获取第三方API响应信息
     * <p>
     * 此方法首先尝试通过提供的ID在平台上查找相应的数据如果找不到任何记录，
     * 则抛出一个业务异常，表明数据未找到如果找到了对应的数据，则将其转换为
     * ThirdApiRespVO对象并返回这个方法主要用于响应第三方API请求，需要处理的情况包括
     * 数据不存在时的异常处理，以及数据转换为响应对象
     *
     * @param id 第三方API请求中包含的ID，用于查找平台上的数据
     * @return 返回一个ThirdApiRespVO对象，包含从平台上查找到的数据信息
     * @throws BusinessException 当提供的ID在平台上找不到对应数据时抛出此异常
     */
    @Override
    public ThirdApiRespVO ThirdApiById(Long id) {
        // 尝试通过ID查找平台上的数据
        Optional<InfinityPlatformDO> infinityPlatformDO = platformRepo.findById(id);
        // 如果找不到数据，抛出业务异常
        if (infinityPlatformDO.isEmpty()) {
            throw new BusinessException("Not Found Data");
        }
        // 找到数据后，将其转换为响应对象并返回
        return InfinityPlatformConvert.INSTANCE.doToVO(infinityPlatformDO.get());
    }

    /**
     * 保存第三方API信息
     *
     * @param saveParam 保存参数对象，包含需要保存的第三方API信息
     * @return 保存后的第三方API ID
     * @throws BusinessException 当数据不存在时抛出业务异常
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long save(ThirdApiSaveParamVO saveParam) {
        // 检查编码是否存在
        boolean exists = platformRepoProc.existsByCode(saveParam.getId(), saveParam.getPlatformCode());
        Assert.isFalse(exists, "编码已存在");

        // 新增
        if (saveParam.isNew()) {
            // 将保存参数转换为实体对象
            InfinityPlatformDO infinityPlatformDO = InfinityPlatformConvert.INSTANCE.saveParamToDO(saveParam);
            infinityPlatformDO.setTenantId(currentTenantId());
            // 保存到数据库
            platformRepo.save(infinityPlatformDO);
            return infinityPlatformDO.getId();
        }
        // 修改
        else {
            // 根据ID查询现有数据
            Optional<InfinityPlatformDO> infinityPlatformDO = platformRepo.findById(saveParam.getId());
            if (infinityPlatformDO.isEmpty()) {
                throw new BusinessException("Not Found Data");
            }
            InfinityPlatformDO infinityPlatform = infinityPlatformDO.get();
            // 将保存参数合并到现有数据中
            InfinityPlatformConvert.INSTANCE.saveParamMergeToDO(saveParam, infinityPlatform);
            infinityPlatform.setTenantId(currentTenantId());
            // 保存更新后的数据
            platformRepo.save(infinityPlatform);
            return infinityPlatform.getId();
        }
    }

    /**
     * 删除指定的平台及其下所有目录和接口
     * 此方法使用了事务管理，当操作抛出异常时，会回滚事务，确保数据一致性
     *
     * @param ids 平台ID列表，表示要删除的平台
     * @return 返回删除的平台数量
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public long delete(List<Long> ids) {
        // TODO 删除该平台下，所有目录及接口
        return platformRepoProc.delete(ids);
    }
}
