package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.provider.TenantOrgDataIsolateProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.platform.service.repo.MenuOperationRepoProc;
import com.elitescloud.cloudt.security.common.InnerUserEnum;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.model.bo.MasterUserBO;
import com.elitescloud.cloudt.system.service.common.constant.SubUserPermissionTypeEnum;
import com.elitescloud.cloudt.system.service.model.bo.AppBO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.*;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/25
 */
abstract class BasePermissionManager {
    @Autowired
    protected AppRepoProc appRepoProc;
    @Autowired
    protected MenuRepoProc menuRepoProc;
    @Autowired
    protected MenuOperationRepoProc menuOperationRepoProc;
    @Autowired
    protected RoleDataPermissionRepoProc dataPermissionRepoProc;
    @Autowired
    protected RoleFieldPermissionRepoProc fieldPermissionRepoProc;
    @Autowired
    protected BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    protected BusinessOperationRepoProc businessOperationRepoProc;
    @Autowired
    protected BusinessObjectRefRepoProc refRepoProc;
    @Autowired
    protected BusinessParamRepoProc businessParamRepoProc;
    @Autowired
    protected RoleRepoProc roleRepoProc;
    @Autowired
    protected UserRoleRepoProc userRoleRepoProc;
    @Autowired
    protected PermissionResRepoProc permissionResRepoProc;

    @Autowired
    protected UserQueryManager userQueryManager;

    @Autowired
    protected TenantDataIsolateProvider tenantDataIsolateProvider;
    @Autowired
    protected TenantOrgDataIsolateProvider tenantOrgDataIsolateProvider;
    @Autowired
    protected TenantClientProvider tenantClientProvider;

    @Autowired
    protected TaskExecutor taskExecutor;
    @Autowired
    protected RedisUtils redisUtils;

    /**
     * 查询租户的应用列表
     *
     * @param appCodeFilter 过滤的应用编码
     * @return 应用信息
     */
    public Map<String, AppBO> tenantApps(String appCodeFilter) {
        return tenantApps(appCodeFilter, null);
    }

    /**
     * 查询租户的应用列表
     *
     * @param appCodeFilter 过滤的应用编码
     * @return 应用信息
     */
    public Map<String, AppBO> tenantApps(String appCodeFilter, String terminal) {
        var tenant = tenantClientProvider.getCurrentTenant();
        if (tenant != null && CollUtil.isEmpty(tenant.getAppCodes())) {
            return Collections.emptyMap();
        }
        var needFilterAppCode = StringUtils.hasText(appCodeFilter);
        var needFilterTerminal = StringUtils.hasText(terminal);

        var appList = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.allBO(true));
        var tenantAppCodes = tenant == null ? null : tenant.getAppCodes();
        Map<String, AppBO> appsMap = new LinkedHashMap<>(appList.size());
        for (var param : appList) {
            if (needFilterAppCode && !appCodeFilter.equals(param.getAppCode())) {
                continue;
            }
            if (needFilterTerminal && !terminal.equals(param.getAdaptedTerminal())) {
                continue;
            }
            if (tenantAppCodes == null || tenantAppCodes.contains(param.getAppCode())) {
                appsMap.put(param.getAppCode(), param);
            }
        }
        return appsMap;
    }

    /**
     * 获取当前用户授权的应用编码
     *
     * @param currentUser   当前用户
     * @param defaultAllApp 默认是否返回所有应用
     * @return 授权的应用编码，null表示拥有所有权限
     */
    public Set<String> tenantAppCodes(@NotNull GeneralUserDetails currentUser, boolean defaultAllApp) {
        // 获取租户授权的应用
        if (currentUser.getTenant() != null) {
            if (CollectionUtils.isEmpty(currentUser.getTenant().getAppCodes())) {
                return Collections.emptySet();
            }
            return currentUser.getTenant().getAppCodes();
        }

        if (!defaultAllApp) {
            return null;
        }
        // 返回所有应用
        return new HashSet<>(tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.allCodes(true)));
    }

    /**
     * 获取租户ID
     *
     * @return
     */
    protected Long obtainTenantId() {
        var tenant = tenantClientProvider.getSessionTenant();
        if (tenant == null) {
            return TenantConstant.DEFAULT_TENANT_ID;
        }
        return tenant.getId();
    }

    /**
     * 是否是系统管理员
     *
     * @param currentUser
     * @param masterUser
     * @return
     */
    protected boolean isSystemAdmin(@NotNull GeneralUserDetails currentUser, MasterUserBO masterUser) {
        if (currentUser.isSystemAdmin()) {
            return true;
        }

        // 主账号
        return isSystemAdmin(masterUser);
    }

    protected boolean isSystemAdmin(@NotNull SysUserDTO user, MasterUserBO masterUser) {
        if (isSystemAdmin(user.getUsername())) {
            return true;
        }
        // 主账号
        return isSystemAdmin(masterUser);
    }

    private boolean isSystemAdmin(MasterUserBO masterUser) {
        if (masterUser != null && isSystemAdmin(masterUser.getUsername())) {
            if (SubUserPermissionTypeEnum.EXTENDS_ALL == masterUser.getPermissionType()) {
                return true;
            }
        }

        return false;
    }

    private boolean isSystemAdmin(@NotBlank String username) {
        return InnerUserEnum.ADMIN.getUsername().equals(username);
    }

    /**
     * 是否是租户管理员
     *
     * @param currentUser
     * @param masterUser
     * @return
     */
    protected boolean isTenantAdmin(@NotNull GeneralUserDetails currentUser, MasterUserBO masterUser) {
        return isTenantAdmin(currentUser.getUser(), masterUser);
    }

    protected boolean isTenantAdmin(@NotNull SysUserDTO user, MasterUserBO masterUser) {
        var tenant = user.getSysTenantVO();
        if (tenant == null) {
            return false;
        }
        if (Objects.equals(user.getId(), tenant.getSysUserId())) {
            return true;
        }

        // 主账号
        if (masterUser != null && Objects.equals(tenant.getSysUserId(), masterUser.getMasterUserId())) {
            if (SubUserPermissionTypeEnum.EXTENDS_ALL == masterUser.getPermissionType()) {
                return true;
            }
        }

        return false;
    }

    /**
     * 是否是租户组织管理员
     *
     * @param currentUser
     * @param masterUser
     * @return
     */
    protected boolean isTenantOrgAdmin(@NotNull GeneralUserDetails currentUser, MasterUserBO masterUser) {
        return isTenantOrgAdmin(currentUser.getUser(), masterUser);
    }

    protected boolean isTenantOrgAdmin(@NotNull SysUserDTO user, MasterUserBO masterUser) {
        var tenantOrgAdminId = user.getTenantOrgAdminId();
        if (tenantOrgAdminId == null) {
            return false;
        }
        if (Objects.equals(user.getId(), tenantOrgAdminId)) {
            return true;
        }

        // 主账号
        if (masterUser != null && Objects.equals(tenantOrgAdminId, masterUser.getMasterUserId())) {
            if (SubUserPermissionTypeEnum.EXTENDS_ALL == masterUser.getPermissionType()) {
                return true;
            }
        }

        return false;
    }
}
