package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.service.repo.EmployeeOrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/24
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@Log4j2
public class EmployeeQueryManager {

    @Autowired
    private EmployeeRepoProc repoProc;
    @Autowired
    private EmployeeOrgRepoProc employeeOrgRepoProc;
    @Autowired
    private OrgRepoProc orgRepoProc;

    /**
     * 根据账号ID获取组织
     *
     * @param userIds 账号ID
     * @return 账号ID与对应的组织
     */
    public Map<Long, List<IdCodeNameParam>> getEmployeeOrgsByUserIds(Collection<Long> userIds) {
        if (CollUtil.isEmpty(userIds)) {
            return Collections.emptyMap();
        }

        return employeeOrgRepoProc.getOrgsByUserId(userIds, true);
    }

    /**
     * 获取员工的所在组织（包含上级组织）
     *
     * @param employeeId 员工ID
     * @return 组织列表
     */
    public List<List<IdCodeNameParam>> getEmployeeOrgsWithParent(long employeeId) {
        // 查询出员工的组织ID
        List<Long> orgIds = employeeOrgRepoProc.getOrgIdByEmployeeId(employeeId);
        if (orgIds.isEmpty()) {
            return Collections.emptyList();
        }

        // 查询组织及其父组织
        List<List<IdCodeNameParam>> respList = new ArrayList<>(orgIds.size());
        for (Long orgId : orgIds) {
            var orgs = orgRepoProc.getBasicDtoOfParents(orgId);

            // 过滤掉禁用的组织
            var existsDisabled = orgs.stream().anyMatch(t -> !t.getEnabled());
            if (existsDisabled) {
                continue;
            }

            var paramList = orgs.stream()
                    .map(t -> {
                        IdCodeNameParam param = new IdCodeNameParam();
                        param.setId(t.getId());
                        param.setCode(t.getCode());
                        param.setName(t.getPrettyName());

                        return param;
                    }).collect(Collectors.toList());
            respList.add(paramList);
        }

        return respList;
    }

    /**
     * 获取员工的领导
     *
     * @param employeeId 员工ID
     * @return 组织ID与领导信息
     */
    public Map<Long, SysEmployeeBasicDTO> getEmployeeLeader(long employeeId) {
        // 查询领导ID
        var leaderIdMap = employeeOrgRepoProc.getEmployeeOrgIdAndLeaderId(employeeId);
        if (leaderIdMap.isEmpty()) {
            return Collections.emptyMap();
        }
        var leaderUserIds = leaderIdMap.values();
        var emaployeeMap = repoProc.queryBasicByUserIds(leaderUserIds).stream().collect(Collectors.toMap(SysEmployeeBasicDTO::getUserId, t -> t, (t1, t2) -> t1));

        Map<Long, SysEmployeeBasicDTO> result = new HashMap<>();
        for (Map.Entry<Long, Long> entry : leaderIdMap.entrySet()) {
            var emp = emaployeeMap.get(entry.getValue());
            if (emp != null) {
                result.put(entry.getKey(), emp);
            }
        }
        return result;
    }
}
