package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.model.entity.QSysMqConsumeDO;
import com.elitescloud.cloudt.system.model.entity.QSysMqMessageDO;
import com.elitescloud.cloudt.system.model.entity.QSysMqRetryDO;
import com.elitescloud.cloudt.system.model.entity.SysMqMessageDO;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/16
 */
@Repository
public class MqMessageRepoProc extends BaseRepoProc<SysMqMessageDO> {
    private static final QSysMqMessageDO QDO = QSysMqMessageDO.sysMqMessageDO;

    public MqMessageRepoProc() {
        super(QDO);
    }

    public void deleteExpiredMessage(LocalDateTime expiredTime) {
        super.delete(QDO.createTime.before(expiredTime));
    }

    /**
     * 更新发送结果
     *
     * @param messageId
     * @param success
     * @param failReason
     */
    public void updateSendResult(@NotBlank String messageId, boolean success, String failReason) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalMessageId.eq(messageId)).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.success, success)
                .set(QDO.failReason, failReason)
                .set(QDO.finishTime, LocalDateTime.now())
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.messageId.eq(messageId))
                .execute();
    }

    /**
     * 更新重试失败的结果
     *
     * @param messageId
     * @param reason
     */
    public void updateRetryFailResult(@NotBlank String messageId, String reason) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalMessageId.eq(messageId)).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.remark, reason)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.messageId.eq(messageId))
                .execute();
    }

    /**
     * 更新消息的重试发送时间
     *
     * @param messageId
     * @param sendTime
     */
    public void updateRetrySendTime(@NotBlank String messageId, LocalDateTime sendTime) {
        jpaQueryFactory.update(QDO)
                .set(QDO.sendTime, sendTime)
                .where(QDO.messageId.eq(messageId))
                .execute();
    }

    /**
     * 更新消息的重试发送时间
     *
     * @param messageId
     * @param lastRetryMessageId
     */
    public void updateLastRetryMessageId(@NotBlank String messageId, String lastRetryMessageId) {
        jpaQueryFactory.update(QDO)
                .set(QDO.lastRetryMessageId, lastRetryMessageId)
                .where(QDO.messageId.eq(messageId))
                .execute();
    }

    /**
     * 更新消费时间
     *
     * @param id
     * @param consumeTime
     */
    public void updateConsumed(long id, LocalDateTime consumeTime) {
        var sub = new QSysMqMessageDO("sub");
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalMessageId.eq(
                        JPAExpressions.select(sub.messageId).from(sub).where(sub.id.eq(id))
                )).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.consumed, true)
                .set(QDO.consumeTime, consumeTime)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新消费时间
     *
     * @param messageId
     * @param consumeTime
     */
    public void updateConsumedByMessageId(@NotBlank String messageId, LocalDateTime consumeTime) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalMessageId.eq(messageId
                )).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.consumed, true)
                .set(QDO.consumeTime, consumeTime)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.messageId.eq(messageId))
                .execute();
    }

    /**
     * 根据消息ID删除
     *
     * @param messageId
     */
    public void deleteByMessageId(@NotBlank String messageId) {
        super.deleteByValue(QDO.messageId, messageId);
    }

    /**
     * 获取原始消息ID
     *
     * @param messageId
     * @return
     */
    public String getOriginalMessageId(@NotBlank String messageId) {
        return super.getValueByValue(QDO.originalMessageId, QDO.messageId, messageId);
    }

    /**
     * 获取版本号
     *
     * @param messageId
     * @return
     */
    public Integer getVersionByMessageId(@NotBlank String messageId) {
        return super.getValueByValue(QDO.auditDataVersion, QDO.messageId, messageId);
    }

    /**
     * 根据消息ID获取ID
     *
     * @param messageId
     * @return
     */
    public Long getIdByMessageId(@NotBlank String messageId) {
        return super.getIdByValue(QDO.messageId, messageId);
    }

    /**
     * 根据消息ID查询消息
     *
     * @param messageId
     * @return
     */
    public SysMqMessageDO getByMessageId(@NotBlank String messageId) {
        return super.getOneByValue(QDO.messageId, messageId);
    }

    /**
     * 获取消息列表
     *
     * @param messageIds
     * @return
     */
    public List<SysMqMessageDO> listByMessageId(@NotEmpty Collection<String> messageIds) {
        return super.getListByValue(QDO.messageId, messageIds);
    }
}
