package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.constant.DataPermissionType;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiDataRuleListQueryDTO;
import com.elitescloud.cloudt.system.model.bo.RolePermissionRuleBO;
import com.elitescloud.cloudt.system.service.model.entity.QSysRoleDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysRoleDataPermissionDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleDataPermissionDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/25
 */
@Repository
public class RoleDataPermissionRepoProc extends BaseRepoProc<SysRoleDataPermissionDO> {
    private static final QSysRoleDataPermissionDO QDO = QSysRoleDataPermissionDO.sysRoleDataPermissionDO;
    private static final QSysRoleDO QDO_ROLE = QSysRoleDO.sysRoleDO;

    public RoleDataPermissionRepoProc() {
        super(QDO);
    }

    public void deleteByRole(@NotBlank String roleCode) {
        super.deleteByValue(QDO.roleCode, roleCode);
    }

    public void deleteByRoleForMenuOperation(@NotBlank String roleCode, @NotBlank String menuCode, @NotBlank String operationCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.menuCode.eq(menuCode)).and(QDO.operationCode.eq(operationCode))
                .and(QDO.permissionType.eq(DataPermissionType.MENU_OPERATION_RULE.name()));
        super.delete(predicate);
    }

    public void deleteByRoleForBusinessOperation(@NotBlank String roleCode, @NotBlank String operationCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.operationCode.eq(operationCode))
                .and(QDO.permissionType.eq(DataPermissionType.BUSINESS_OPERATION_RULE.name()));
        super.delete(predicate);
    }

    public void deleteByRoleForBusinessObject(@NotBlank String roleCode, @NotBlank String appCode, @NotBlank String businessObjectCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.businessObjectCode.eq(businessObjectCode)).and(QDO.appCode.eq(appCode))
                .and(QDO.permissionType.eq(DataPermissionType.BUSINESS_OBJECT_RULE.name()));
        super.delete(predicate);
    }

    public void deleteByRoleForRefResourceBusinessObject(@NotBlank String roleCode, @NotBlank String appCode, @NotBlank String businessObjectCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.businessObjectCode.eq(businessObjectCode)).and(QDO.appCode.eq(appCode))
                .and(QDO.permissionType.eq(DataPermissionType.BUSINESS_RESOURCE_RULE.name()));
        super.delete(predicate);
    }

    public List<SysRoleDataPermissionDO> listByRoleForMenuOperation(@NotBlank String roleCode, @NotBlank String menuCode, @NotBlank String operationCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.menuCode.eq(menuCode)).and(QDO.operationCode.eq(operationCode))
                .and(QDO.permissionType.eq(DataPermissionType.MENU_OPERATION_RULE.name()));
        return super.getList(predicate, QDO.ruleOrder.asc());
    }

    public List<SysRoleDataPermissionDO> listByRoleForBusinessOperation(@NotBlank String roleCode, @NotBlank String operationCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.operationCode.eq(operationCode))
                .and(QDO.permissionType.eq(DataPermissionType.BUSINESS_OPERATION_RULE.name()));
        return super.getList(predicate, QDO.ruleOrder.asc());
    }

    public List<SysRoleDataPermissionDO> listByRoleAndBusinessObjectForBusinessOperation(@NotBlank String roleCode, @NotEmpty Collection<String> businessObjectCodes) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.businessObjectCode.in(businessObjectCodes))
                .and(QDO.permissionType.eq(DataPermissionType.BUSINESS_OPERATION_RULE.name()));
        return super.getList(predicate);
    }

    public List<SysRoleDataPermissionDO> listByRoleAndBusinessObjectForMenuBusinessOperation(@NotBlank String roleCode, @NotEmpty Collection<String> businessObjectCodes) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.businessObjectCode.in(businessObjectCodes))
                .and(QDO.permissionType.eq(DataPermissionType.MENU_OPERATION_RULE.name()));
        return super.getList(predicate);
    }

    public List<SysRoleDataPermissionDO> listByRoleForBusinessObject(@NotBlank String roleCode, @NotBlank String businessObjectCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.businessObjectCode.eq(businessObjectCode))
                .and(QDO.permissionType.eq(DataPermissionType.BUSINESS_OBJECT_RULE.name()));
        return super.getList(predicate, QDO.ruleOrder.asc());
    }

    public List<SysRoleDataPermissionDO> listByRoleForBusinessObjectResource(@NotBlank String roleCode, @NotBlank String businessObjectCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.businessObjectCode.eq(businessObjectCode))
                .and(QDO.permissionType.eq(DataPermissionType.BUSINESS_RESOURCE_RULE.name()));
        return super.getList(predicate, QDO.ruleOrder.asc());
    }

    public List<String> listFieldByRoleForBusinessObjectResource(@NotBlank String roleCode, @NotBlank String businessObjectCode) {
        var predicate = QDO.roleCode.eq(roleCode).and(QDO.businessObjectCode.eq(businessObjectCode))
                .and(QDO.permissionType.eq(DataPermissionType.BUSINESS_RESOURCE_RULE.name()))
                .and(QDO.ruleField.isNotNull());
        return super.getValueList(QDO.ruleField, predicate, QDO.ruleOrder.asc());
    }

    public List<SysRoleDataPermissionDO> listByRole(@NotBlank String roleCode) {
        return super.getListByValue(QDO.roleCode, roleCode);
    }

    public List<RolePermissionRuleBO> listRolePermissionRuleByRole(@NotBlank String roleCode, DataPermissionType permissionType) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.roleCode, roleCode)
                .andEq(QDO.permissionType, permissionType == null ? null : permissionType.name())
                .build();
        return super.jpaQueryFactory.select(qBeanRolePermissionRule())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    public List<RolePermissionRuleBO> listRolePermissionRuleByRole(long roleId) {
        return super.jpaQueryFactory.select(qBeanRolePermissionRule())
                .from(QDO)
                .where(QDO.roleCode.eq(JPAExpressions.select(QDO_ROLE.code).from(QDO_ROLE).where(QDO_ROLE.id.eq(roleId))))
                .fetch();
    }

    public List<SysDprRoleApiDataRuleListQueryDTO> queryDataPermissionByRoles(@NotEmpty Collection<String> roleCodes) {
        return super.jpaQueryFactory.select(qBeanDataPermission())
                .from(QDO)
                .where(QDO.roleCode.in(roleCodes))
                .fetch();
    }

    public List<SysDprRoleApiDataRuleListQueryDTO> queryForRefBusinessObjectResourceByRole(@NotBlank String roleCode, @NotBlank String businessObjectCode) {
        return super.jpaQueryFactory.select(qBeanDataPermission())
                .from(QDO)
                .where(QDO.roleCode.eq(roleCode).and(QDO.permissionType.eq(DataPermissionType.BUSINESS_RESOURCE_RULE.name())).and(QDO.businessObjectCode.eq(businessObjectCode)))
                .fetch();
    }

    public Map<String, String> queryBusinessObjectRef(@NotBlank String roleCode) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.roleCode, roleCode)
                .andIn(QDO.permissionType, Set.of(DataPermissionType.BUSINESS_RESOURCE_RULE.name(), DataPermissionType.BUSINESS_OBJECT_RULE.name()))
                .and(QDO.businessObjectCode.isNotEmpty().and(QDO.refBusinessObject.isNotNull()))
                .build();

        return jpaQueryFactory.select(QDO.businessObjectCode, QDO.refBusinessObject)
                .from(QDO)
                .where(predicate)
                .fetch()
                .stream()
                .filter(t -> StrUtil.isAllNotBlank(t.get(QDO.businessObjectCode), t.get(QDO.refBusinessObject)))
                .collect(Collectors.toMap(t -> t.get(QDO.refBusinessObject), t -> t.get(QDO.businessObjectCode), (t1, t2) -> t1));
    }

    private QBean<SysDprRoleApiDataRuleListQueryDTO> qBeanDataPermission() {
        return Projections.bean(SysDprRoleApiDataRuleListQueryDTO.class, QDO.ruleGroup, QDO.roleCode, QDO.permissionType, QDO.ruleGroupCode,
                QDO.appCode, QDO.menuCode.as("menusCode"), QDO.businessObjectCode, QDO.operationCode.as("apiPermissionCode"),
                QDO.id.as("dprRuleId"), QDO.ruleOrder.as("order"), QDO.dataSet, QDO.ruleRelation.as("dprRuleRelation"),
                QDO.ruleName.as("dprRuleName"), QDO.ruleField.as("dprRuleField"), QDO.refResource, QDO.refBusinessObject, QDO.refField,
                QDO.ruleFieldType.as("dprRuleFieldType"), QDO.fieldValueCondition.as("dprRuleCondition"), QDO.ruleValueType.as("dprRuleValueType"),
                QDO.ruleValue.as("dprRuleValue"), QDO.bs1, QDO.bs2, QDO.bs3);
    }

    private QBean<RolePermissionRuleBO> qBeanRolePermissionRule() {
        return Projections.bean(RolePermissionRuleBO.class, QDO.id, QDO.roleCode, QDO.permissionType, QDO.appCode, QDO.menuCode,
                QDO.businessObjectCode, QDO.operationCode);
    }
}
