package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.entity.QSysVersionDO;
import com.elitescloud.cloudt.system.model.entity.SysVersionDO;
import com.elitescloud.cloudt.system.model.vo.query.version.SysVersionPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.version.SysVersionRespVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

@Repository
public class SysVersionRepoProc extends BaseRepoProc<SysVersionDO> {

    private static final QSysVersionDO QDO = QSysVersionDO.sysVersionDO;

    @Autowired
    private SysVersionRepo sysVersionRepo;

    public SysVersionRepoProc() {
        super(QDO);
    }


    public QBean<SysVersionRespVO> sysVersionBean() {
        return Projections.bean(SysVersionRespVO.class,
                QDO.id,
                QDO.versionNo,
                QDO.describeMsg,
                QDO.changeContent,
                QDO.status,
                QDO.createTime
        );
    }

    /**
     * 获取当前版本号
     *
     * @return 版本号
     */
    public String getActiveVersion() {
        return super.jpaQueryFactory.select(QDO.versionNo)
                .from(QDO)
                .where(QDO.status.eq(Boolean.TRUE))
                .orderBy(QDO.createTime.desc())
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取版本列表
     *
     * @return 版本列表
     */
    public List<String> queryVersionList() {
        return super.getValueList(QDO.versionNo);
    }

    public SysVersionRespVO getActiveVersionDetail() {
        Predicate predicate = QDO.status.eq(Boolean.TRUE);
        JPAQuery<SysVersionRespVO> jpaQuery = super.jpaQueryFactory.select(sysVersionBean())
                .from(QDO)
                .where(predicate);
        return jpaQuery.fetchOne();
    }

    public PagingVO<SysVersionRespVO> page(SysVersionPageQueryVO query) {
        Predicate predicate = this.pageWhere(query);
        var jpaQuery = super.jpaQueryFactory.select(sysVersionBean())
                .from(QDO)
                .where(predicate);
        return super.queryByPage(jpaQuery, query.getPageRequest(), QDO.createTime.desc());
    }

    public Predicate pageWhere(SysVersionPageQueryVO query) {
        List<Predicate> list = new ArrayList<>();
        if (StringUtils.isNotBlank(query.getVersionNo())) {
            list.add(QDO.versionNo.contains(query.getVersionNo()));
        }
        if (StringUtils.isNotBlank(query.getDescribeMsg())) {
            list.add(QDO.describeMsg.contains(query.getDescribeMsg()));
        }
        if (StringUtils.isNotBlank(query.getChangeContent())) {
            list.add(QDO.changeContent.contains(query.getChangeContent()));
        }
        if (ObjectUtils.isNotEmpty(query.getStatus())) {
            list.add(QDO.status.eq(query.getStatus()));
        }
        return ExpressionUtils.allOf(list);
    }

    public Long getStatusCount(Long id, Boolean status) {
        BooleanExpression e = QDO.status.eq(status);
        if (ObjectUtils.isNotEmpty(id)) {
            e = e.and(QDO.id.ne(id));
        }
        return sysVersionRepo.count(e);
    }

    public Long getVersionNoCount(Long id, String versionNo) {
        BooleanExpression e = QDO.versionNo.eq(versionNo);
        if (ObjectUtils.isNotEmpty(id)) {
            e = e.and(QDO.id.ne(id));
        }
        return sysVersionRepo.count(e);
    }

    @Transactional(rollbackFor = Exception.class)
    public SysVersionDO saveVersion(SysVersionDO sysVersionDO) {
        return sysVersionRepo.save(sysVersionDO);
    }


    @Transactional(rollbackFor = Exception.class)
    public void del(Long id) {
        sysVersionRepo.deleteById(id);
    }


    public SysVersionDO getById(Long id) {
        Optional<SysVersionDO> optional = sysVersionRepo.findById(id);
        return optional.get();
    }

}
