package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.model.vo.query.org.TenantOrgPageQueryVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysOrgDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantOrgDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantOrgDO;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.SubQueryExpression;
import com.querydsl.jpa.JPAExpressions;
import com.querydsl.jpa.impl.JPAQuery;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/30
 */
@Repository
public class TenantOrgRepoProc extends BaseRepoProc<SysTenantOrgDO> {
    private static final QSysTenantOrgDO QDO = QSysTenantOrgDO.sysTenantOrgDO;
    private static final QSysOrgDO QDO_ORG = QSysOrgDO.sysOrgDO;

    public TenantOrgRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用状态
     *
     * @param id
     * @param enabled
     */
    public void updateEnabled(Long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 更新管理员ID
     *
     * @param id
     * @param adminId
     */
    public void updateAdminId(Long id, Long adminId) {
        super.updateValue(QDO.adminId, adminId, id);
    }

    /**
     * 根据机构获取租户组织信息
     *
     * @param orgId
     * @return
     */
    public SysTenantOrgDO getByOrgId(Long orgId) {
        return super.getOneByValue(QDO.orgId, orgId);
    }

    /**
     * 获取是否启用
     *
     * @param id
     * @return
     */
    public Boolean getEnabled(Long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 获取管理员ID
     *
     * @param id
     * @return
     */
    public Long getAdminId(Long id) {
        return super.getValue(QDO.adminId, id);
    }

    /**
     * 查询组织的管理员ID
     *
     * @param orgIds
     * @return
     */
    public Map<Long, Long> queryAdmin(Collection<Long> orgIds) {
        var recordList = jpaQueryFactory.select(QDO.orgId, QDO.adminId)
                .from(QDO)
                .where(QDO.orgId.in(orgIds).and(QDO.enabled.eq(true)))
                .fetch();
        if (recordList.isEmpty()) {
            return Collections.emptyMap();
        }

        Map<Long, Long> result = new HashMap<>(recordList.size());
        for (Tuple tuple : recordList) {
            result.put(tuple.get(QDO.orgId), tuple.get(QDO.adminId));
        }
        return result;
    }

    /**
     * 分页查询租户组织
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysTenantOrgDO> pageMng(TenantOrgPageQueryVO queryVO) {
        SubQueryExpression<Long> subQueryExpression = null;
        if (!CharSequenceUtil.isAllBlank(queryVO.getOrgCode(), queryVO.getOrgName())) {
            Predicate subPredicate = PredicateBuilder.builder()
                    .andLike(StringUtils.hasText(queryVO.getOrgCode()), QDO_ORG.code, queryVO.getOrgCode())
                    .and(StringUtils.hasText(queryVO.getOrgName()), () -> {
                        String keyword = "%" + queryVO.getOrgName() + "%";
                        return QDO_ORG.name.like(keyword).or(QDO_ORG.shortName.like(keyword));
                    })
                    .build();
            subQueryExpression = JPAExpressions.select(QDO_ORG.id)
                    .from(QDO_ORG)
                    .where(subPredicate);
        }

        JPAQuery<SysTenantOrgDO> jpaQuery = jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(subQueryExpression == null ? null : QDO.orgId.in(subQueryExpression));
        return super.queryByPage(jpaQuery, queryVO.getPageRequest());
    }

    /**
     * 根据组织编码获取ID
     *
     * @param code            组织编码
     * @param includeDisabled 是否包含禁用的
     * @return
     */
    public Long getOrgIdByCode(@NotBlank String code, boolean includeDisabled) {
        if (includeDisabled) {
            return jpaQueryFactory.select(QDO.orgId)
                    .from(QDO)
                    .where(QDO.orgId.eq(
                            JPAExpressions.select(QDO_ORG.id)
                                    .from(QDO_ORG)
                                    .where(QDO_ORG.code.eq(code))
                                    .limit(1)
                    ))
                    .limit(1)
                    .fetchOne();
        }

        return jpaQueryFactory.select(QDO.orgId)
                .from(QDO)
                .where(QDO.orgId.eq(
                        JPAExpressions.select(QDO_ORG.id)
                                .from(QDO_ORG)
                                .where(QDO_ORG.code.eq(code).and(QDO_ORG.enabled.eq(true)))
                                .limit(1)
                ).and(QDO.enabled.eq(true)))
                .limit(1)
                .fetchOne();
    }
}
