package com.elitescloud.cloudt.ucenter.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.dto.req.msg.MsgRecipientUserDTO;
import com.elitescloud.cloudt.system.dto.req.msg.MsgSendEmployeeUserDTO;
import com.elitescloud.cloudt.system.dto.req.msg.custom.EmployeeSendParamDTO;
import com.elitescloud.cloudt.system.dto.req.msg.custom.EmployeeSentMsgParamDTO;
import com.elitescloud.cloudt.system.param.SysMsgViewResultDTO;
import com.elitescloud.cloudt.system.service.SysMsgCustomSendRpcService;
import com.elitescloud.cloudt.system.service.SysMsgInteriorRpcService;
import com.elitescloud.cloudt.system.vo.SysUserDTO;

import com.elitescloud.cloudt.ucenter.api.vo.param.MessagePagingParam;
import com.elitescloud.cloudt.ucenter.api.vo.param.MessageReceiverPagingParam;
import com.elitescloud.cloudt.ucenter.api.vo.param.PublishParam;
import com.elitescloud.cloudt.ucenter.api.vo.resp.AttachFileInfoVO;
import com.elitescloud.cloudt.ucenter.api.vo.resp.MessageDetailRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.resp.MessageReceiverRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.resp.MessageRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.save.FileInfoSaveVO;
import com.elitescloud.cloudt.ucenter.api.vo.save.MessageSaveVO;
import com.elitescloud.cloudt.ucenter.common.constant.UdcEnum;
import com.elitescloud.cloudt.ucenter.convert.MessageManageConvert;
import com.elitescloud.cloudt.ucenter.entity.FileInfoDO;
import com.elitescloud.cloudt.ucenter.entity.MessageInfoDO;
import com.elitescloud.cloudt.ucenter.entity.MessageReceiverDO;
import com.elitescloud.cloudt.ucenter.repo.*;
import com.elitescloud.cloudt.ucenter.service.MessageManageService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
@AllArgsConstructor
public class MessageManageServiceImpl implements MessageManageService {

    private final MessageInfoRepo messageInfoRepo;
    private final MessageInfoRepoProc messageInfoRepoProc;

    private final MessageReceiverRepoProc messageReceiverRepoProc;
    private final MessageReceiverRepo messageReceiverRepo;

    private final SysMsgInteriorRpcService sysMsgInteriorRpcService;

    private final SysMsgCustomSendRpcService sysMsgCustomSendRpcService;

    private final FileInfoRepo fileInfoRepo;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdate(MessageSaveVO saveVO) {

        if (ObjectUtil.isNotNull(saveVO.getId())) {
            // 编辑
            Optional<MessageInfoDO> optional = messageInfoRepo.findById(saveVO.getId());
            if (optional.isEmpty()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未找到消息信息");
            }
            if (StrUtil.equals(optional.get().getPublishStatus(), UdcEnum.MESSAGE_CATEGORY_PUBLISH_STATUS_1.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能编辑没有发布的消息");
            }
            MessageInfoDO messageInfoDO = optional.get();
            // 删除参与人信息以及附件信息
            messageReceiverRepo.deleteByMessageId(messageInfoDO.getId());
            fileInfoRepo.deleteAllBySourceId(messageInfoDO.getId());
        }

        // 保存消息信息
        MessageInfoDO messageInfoDO = MessageManageConvert.INSTANCE.messageSaveVo2Do(saveVO);
        messageInfoDO.setTotalCount(saveVO.getReceiverList().size());
        MessageInfoDO saveDo = messageInfoRepo.save(messageInfoDO);

        // 保存附件信息
        List<FileInfoSaveVO> fileList = saveVO.getFileCodeList();
        List<FileInfoDO> fileInfoDoList = MessageManageConvert.INSTANCE.fileVos2Dos(fileList);
        fileInfoDoList.stream().forEach(item -> item.setSourceId(saveDo.getId()).setSourceType("自定义消息表"));
        fileInfoRepo.saveAll(fileInfoDoList);

        // 保存接收人信息
        List<MessageReceiverDO> receiverDOList = MessageManageConvert.INSTANCE.receiverVos2Dos(saveVO.getReceiverList());
        receiverDOList.stream().forEach(item -> item.setMessageId(saveDo.getId()));
        messageReceiverRepo.saveAll(receiverDOList);
        // 立即发布
        if (StrUtil.equals(saveVO.getPublishStatus(), UdcEnum.MESSAGE_CATEGORY_PUBLISH_STATUS_1.getValueCode())) {
            PublishParam param = new PublishParam();
            param.setMessageId(saveDo.getId());
            param.setImmediatelyPublish(Boolean.TRUE);
            try {
                publish(param);
            } catch (Exception e) {
                log.error("saveOrUpdate 立即发布失败，参数 {} , 失败原因：{} \t 失败信息：{}", saveVO, e.getMessage(), e);
                throw new BusinessException("立即发布失败！请稍后发布");
            }
        }

        return saveDo.getId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object delete(List<Long> ids) {

        List<MessageInfoDO> messageInfoList = messageInfoRepo.findAllById(ids);
        if (CollectionUtil.isEmpty(messageInfoList)) {
            return null;
        }
        // 已发布id列表
        List<Long> publishIdList = messageInfoList.stream()
                .filter(item -> ObjectUtil.equals(item.getPublishStatus(), UdcEnum.MESSAGE_CATEGORY_PUBLISH_STATUS_1.getValueCode()))
                .map(item -> item.getId()).collect(Collectors.toList());

        // 未发布id列表
        List<Long> unPublishIdList = messageInfoList.stream()
                .filter(item -> ObjectUtil.equals(item.getPublishStatus(), UdcEnum.MESSAGE_CATEGORY_PUBLISH_STATUS_0.getValueCode()))
                .map(item -> item.getId()).collect(Collectors.toList());

        // 已发布 逻辑删除
        if (CollectionUtil.isNotEmpty(publishIdList)) {
            messageInfoRepo.updateByIdInBatch(publishIdList);
            // 删除文件信息以及接收者信息
            fileInfoRepo.updateByMessageIdInBatch(publishIdList);
            messageReceiverRepo.updateByMessageIdInBatch(publishIdList);
        }
        // 未发布 物理删除
        if (CollectionUtil.isNotEmpty(unPublishIdList)) {
            messageInfoRepo.deleteAllByIdInBatch(unPublishIdList);
            fileInfoRepo.deleteByMessageIdInBatch(publishIdList);
            messageReceiverRepo.deleteByMessageIdInBatch(publishIdList);
        }
        return null;
    }

    @Override
    @SysCodeProc
    @Transactional(rollbackFor = Exception.class)
    public PagingVO<MessageRespVO> search(MessagePagingParam param) {
        PagingVO<MessageRespVO> results = messageInfoRepoProc.search(param);
        List<MessageRespVO> records = results.getRecords();

        if (CollectionUtil.isNotEmpty(records)) {
            List<MessageInfoDO> vos = MessageManageConvert.INSTANCE.messageRespVos2Dos(records);
            // 已经发布的 需要查询阅读人数
            List<MessageInfoDO> publishList = vos.stream()
                    .filter(item -> StrUtil.equals(item.getPublishStatus(), UdcEnum.MESSAGE_CATEGORY_PUBLISH_STATUS_1.getValueCode()))
                    .collect(Collectors.toList());
            try {
                // 防止更新消息阅读人数失败 导致查询不到数据
                publishList.stream().forEach(item -> updateReceiverReadStatus(item));
                // 设置已读人数
                Map<Long, Integer> map = publishList.stream().collect(Collectors.toMap(MessageInfoDO::getId, MessageInfoDO::getReadCount));
                results.getRecords().stream().forEach(item -> item.setReadCount(map.get(item.getId())));
            } catch (Exception e) {
                log.error("更新阅读状态失败,失败原因 {}， {}", e.getMessage(), e.getCause());
            }
        }
        return results;
    }

    @Override
    @SysCodeProc
    public MessageDetailRespVO queryDetail(Long id) {
        // 查询消息信息
        Optional<MessageInfoDO> byId = messageInfoRepo.findById(id);
        if (byId.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "没有查询到该消息详情信息");
        }
        // 查询人员信息
        List<MessageReceiverDO> receiverDoList = messageReceiverRepo.findAllByMessageId(id);
        List<MessageReceiverRespVO> receiverRespVOList = MessageManageConvert.INSTANCE.receiverDos2Vos(receiverDoList);
        MessageDetailRespVO respVO = MessageManageConvert.INSTANCE.messageDo2Vo(byId.get());

        // 查询附件信息
        List<FileInfoDO> doList = fileInfoRepo.findAllBySourceId(id);
        List<AttachFileInfoVO> attachFileInfoVOS = MessageManageConvert.INSTANCE.fileDos2Vos(doList);

        respVO.setFileInfoList(attachFileInfoVOS);
        respVO.setReceiverList(receiverRespVOList);
        return respVO;
    }

    @Override
    public PagingVO<MessageReceiverRespVO> searchReceiver(MessageReceiverPagingParam param) {

        // 查询消息记录id
        Optional<MessageInfoDO> byId = messageInfoRepo.findById(param.getMessageId());
        if (byId.isEmpty()) {
            throw new BusinessException("未查询到自定义消息的信息！");
        }
        try {
            // 查询消息接收者是否阅读 并更新阅读状态
            updateReceiverReadStatus(byId.get());
        } catch (Exception e) {
            log.error("更新阅读状态失败,失败原因 {}， {}", e.getMessage(), e.getCause());
        }
        // 查询
        PagingVO<MessageReceiverRespVO> pagingVO = messageReceiverRepoProc.search(param);
        return pagingVO;
    }

    @Transactional(rollbackFor = Exception.class)
    MessageInfoDO updateReceiverReadStatus(MessageInfoDO messageInfoDO) {
        // 查询消息接收者
        List<MessageReceiverDO> receiverDOList = messageReceiverRepo.findAllByMessageId(messageInfoDO.getId());

        // 消息没有接收人
        if (CollectionUtil.isEmpty(receiverDOList)) {
            return null;
        }

        // 获取还未阅读的
        List<MessageReceiverDO> unReadList = receiverDOList.stream()
                .filter(item -> StrUtil.equals(item.getReadStatus(), UdcEnum.MESSAGE_CATEGORY_READ_STATUS_0.getValueCode()))
                .collect(Collectors.toList());

        // 设置消息阅读情况 有还未完成阅读的需要远程查询
        if (CollectionUtil.isNotEmpty(unReadList)) {
            // 查询阅读情况
            ApiResult<List<SysMsgViewResultDTO>> viewedResult = null;
            List<SysMsgViewResultDTO> data;
            try {
                viewedResult = sysMsgInteriorRpcService.getViewedResult(messageInfoDO.getRecordId());
                if (ObjectUtil.notEqual(viewedResult.getCode(), ApiCode.SUCCESS.getCode())) {
                    log.error("publish 调用系统域发送自定义消息失败 失败原因:{} ", viewedResult.getErrorMsg());
                    throw new BusinessException(viewedResult.getMsg());
                }
                data = viewedResult.getData();
            } catch (Exception e) {
                log.error("远程调用系统域发送消息相关服务异常:{} ", e);
                throw new BusinessException("远程调用系统域发送自定义消息接口失败", e);
            }
            if (CollectionUtil.isNotEmpty(data)) {
                Map<Long, SysMsgViewResultDTO> map = data.stream().collect(Collectors.toMap(SysMsgViewResultDTO::getUserId, t -> t, (t1, t2) -> t1));
                unReadList.stream().forEach(item -> {
                    SysMsgViewResultDTO dto = map.get(item.getEmpId());
                    item.setReadStatus(dto.getViewed() ? UdcEnum.MESSAGE_CATEGORY_READ_STATUS_1.getValueCode() : UdcEnum.MESSAGE_CATEGORY_READ_STATUS_0.getValueCode());
                    item.setReadTime(dto.getViewTime());
                });
                List<MessageReceiverDO> readList = unReadList.stream().filter(item -> StrUtil.equals(item.getReadStatus(), UdcEnum.MESSAGE_CATEGORY_READ_STATUS_1.getValueCode())).collect(Collectors.toList());
                messageReceiverRepo.saveAll(unReadList);
                messageInfoDO.setReadCount(messageInfoDO.getReadCount() + readList.size());
                messageInfoRepo.updateReadCount(messageInfoDO.getReadCount(),messageInfoDO.getId());
            }
        }
        return messageInfoDO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object publish(PublishParam param) {
        EmployeeSendParamDTO dto = getEmployeeSendParamDTO(param);
        String recordId = null;
        try {
            ApiResult<String> apiResult = sysMsgCustomSendRpcService.msgCustomEmployeeSend(dto);
            log.info("publish方法调用系统域发送自定义消息返回的结果：{}", apiResult);
            if (ObjectUtil.notEqual(apiResult.getCode(), ApiCode.SUCCESS.getCode())) {
                log.error("publish自定义发布消息 调用系统域发送自定义消息失败 失败原因:{} ", apiResult.getErrorMsg());
                throw new BusinessException(apiResult.getMsg());
            }
            recordId = apiResult.getData();
        } catch (Exception e) {
            log.error("远程调用系统域发送消息相关服务异常:{} ", e);
            throw new BusinessException("远程调用系统域发送自定义消息接口失败");
        }

        messageInfoRepo.updateById(param.getMessageId(), recordId, UdcEnum.MESSAGE_CATEGORY_PUBLISH_STATUS_1.getValueCode());
        messageReceiverRepo.updateReadStatusByMessageId(param.getMessageId());
        return recordId;
    }

    /**
     * 封装发布参数
     *
     * @param param
     * @return
     */
    @NotNull
    private EmployeeSendParamDTO getEmployeeSendParamDTO(PublishParam param) {
        EmployeeSendParamDTO dto = new EmployeeSendParamDTO();
        SysUserDTO user = new SysUserDTO();
        try {
            user = SecurityContextUtil.currentUser().getUser();
        } catch (Exception e) {
            log.error("获取登录人信息失败{}", e);
        }
        Optional<MessageInfoDO> byId = messageInfoRepo.findById(param.getMessageId());
        if (byId.isEmpty()) {
            throw new BusinessException("未查询到自定义消息信息！");
        }
        if (ObjectUtil.equals(param.getImmediatelyPublish(), Boolean.FALSE) && StrUtil.equals(byId.get().getPublishStatus(), UdcEnum.MESSAGE_CATEGORY_PUBLISH_STATUS_1.getValueCode())) {
            throw new BusinessException("该自定义消息已经发布，请勿重复发布！");
        }
        // 发送人对象
        MsgSendEmployeeUserDTO sendUser = new MsgSendEmployeeUserDTO();
        sendUser.setUserId(user.getId());
        sendUser.setUserName(user.getLastName());
        sendUser.setUserCode(user.getUsername());

        // 接收人对象
        List<MessageReceiverDO> receiverDOList = messageReceiverRepo.findAllByMessageId(param.getMessageId());
        if (CollectionUtil.isEmpty(receiverDOList)) {
            throw new BusinessException("该自定义消息还未选择接收人，请选择参与人后进行发布");
        }
        List<MsgRecipientUserDTO> msgRecipientUserDTOS = receiverDOList.stream().map(item -> {
            MsgRecipientUserDTO msgRecipientUser = new MsgRecipientUserDTO();
            msgRecipientUser.setUserId(item.getEmpId());
            msgRecipientUser.setUserName(item.getEmpName());
            return msgRecipientUser;
        }).collect(Collectors.toList());

        // 消息发送渠道类型与内容
        List<EmployeeSentMsgParamDTO> sendTypeMessageMap = new ArrayList<>();
        EmployeeSentMsgParamDTO sentMsgParamDTO = new EmployeeSentMsgParamDTO();
        sentMsgParamDTO.setMsgSendTypeEnum(MsgSendTypeEnum.SYS_INTERIOR);
        sentMsgParamDTO.setTitle(byId.get().getMessageTitle());
        sentMsgParamDTO.setContent(byId.get().getMessageDetail());
        sentMsgParamDTO.setExternalTemplateParams(new HashMap<>());
        sendTypeMessageMap.add(sentMsgParamDTO);

        dto.setSendUser(sendUser);
        dto.setMsgRecipientUserDTO(msgRecipientUserDTOS);
        dto.setSendTypeMessageMap(sendTypeMessageMap);
        Map<String, String> mesJoinParamMap = new HashMap<>();
        // 消息中有附件定义
        mesJoinParamMap.put("msgType", "files");
        List<FileInfoDO> fileList = fileInfoRepo.findAllBySourceId(byId.get().getId());
        if (CollectionUtil.isNotEmpty(fileList)) {
            fileList.stream().forEach(item -> {
                mesJoinParamMap.put(item.getFileCode(), item.getOriginalName());
            });
        }
        dto.setMesJoinParamMap(mesJoinParamMap);
        return dto;
    }
}
