package com.elitescloud.cloudt.system.cas;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.RandomUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.provider.provider.user.AbstractUserDetailManager;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import com.elitescloud.cloudt.system.constant.UserType;
import com.elitescloud.cloudt.system.service.UserQueryService;
import com.elitescloud.cloudt.system.service.manager.UserMngManager;
import com.elitescloud.cloudt.system.service.model.bo.SysUserSaveBO;
import com.elitescloud.cloudt.system.service.model.bo.SysUserTypeBO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.support.TransactionTemplate;

import javax.servlet.http.HttpServletRequest;
import javax.validation.constraints.NotBlank;
import java.util.Set;
import java.util.function.Supplier;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/12/20
 */
@SuppressWarnings({"deprecation", "removal"})
public class SystemUserDetailManager extends AbstractUserDetailManager {
    private static final Logger log = LoggerFactory.getLogger(SystemUserDetailManager.class);

    private final UserQueryService userQueryService;
    private final UserMngManager userMngManager;

    public SystemUserDetailManager(AuthorizationProperties authorizationProperties, UserQueryService userQueryService,
                                   UserMngManager userMngManager) {
        super(authorizationProperties);
        this.userQueryService = userQueryService;
        this.userMngManager = userMngManager;
    }

    @Override
    public GeneralUserDetails loadUserByUsername(@NotBlank String s) {
        return loadUser(() -> userQueryService.getUserByUsername(s));
    }

    @Override
    public GeneralUserDetails loadUserByMobile(String s) {
        return loadUser(() -> userQueryService.getUserByMobile(s));
    }

    @Override
    public GeneralUserDetails loadUserByEmail(String s) {
        return loadUser(() -> userQueryService.getUserByEmail(s));
    }

    @Override
    public GeneralUserDetails loadUserById(String s) {
        return loadUser(() -> userQueryService.getUserById(Long.parseLong(s)));
    }

    @Override
    public GeneralUserDetails loadUserByAccount(String s) {
        return loadUser(() -> userQueryService.getUserByAccount(s));
    }

    @Override
    public GeneralUserDetails loadUserByWechatOpenid(String s) {
        return loadUser(() -> userQueryService.getUserByWechatOpenid(s));
    }

    @Override
    public GeneralUserDetails loadUserByOpenId(String appId, String openId) {
        return loadUser(() -> userQueryService.getUserByOpenId(appId, openId));
    }

    @Override
    public GeneralUserDetails registerOnLoadUserByMobileNotFound(@NotBlank String mobile) {
        var user = userQueryService.getUserByMobile(mobile);
        if (user != null) {
            return loadUser(() -> user);
        }

        log.info("自动注册用户：{}", mobile);
        SysUserSaveBO saveBO = new SysUserSaveBO();
        saveBO.setUsername(generateUsername());
        saveBO.setMobile(mobile);
        saveBO.setEnabled(true);
        saveBO.setSourceType(UserSourceType.REGISTER.getValue());

        // 获取用户类型
        var userType = obtainRegisteredUserType();
        if (CharSequenceUtil.isNotBlank(userType)) {
            saveBO.setUserTypesAdd(Set.of(new SysUserTypeBO(userType)));
        }

        SysUserDO userDO = SpringContextHolder.getBean(TransactionTemplate.class).execute(status -> userMngManager.upsert(saveBO));
        return loadUser(() -> userQueryService.getUserById(userDO.getId()));
    }

    private GeneralUserDetails loadUser(Supplier<SysUserDTO> userDTOSupplier) {
        // 查询用户信息
        SysUserDTO user = userDTOSupplier.get();
        if (user == null) {
            return null;
        }

        return super.wrapUserDetails(user);
    }

    private String generateUsername() {
        while (true) {
            String username = "auto_" + RandomUtil.randomString(15);
            if (userQueryService.getIdByUsername(username).computeData() == null) {
                return username;
            }
        }
    }

    private String obtainRegisteredUserType() {
        HttpServletRequest request = HttpServletUtil.currentRequest();
        if (request == null) {
            return null;
        }

        String userType = request.getParameter("user_type");
        if (CharSequenceUtil.isBlank(userType)) {
            return null;
        }
        var udcMap = SpringContextHolder.getBean(UdcProvider.class).getValueMapByUdcCode(UserType.ADMIN_SYSTEM.getAppCode(), UserType.ADMIN_SYSTEM.getUdcCode());
        if (udcMap.containsKey(userType)) {
            return userType;
        }

        log.warn("用户类型不存在：{}", userType);
        return null;
    }
}
