package com.elitescloud.cloudt.system.controller.mng.org;

import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmpGroupEmpPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmpGroupPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupEmpPageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.EmpGroupSaveVO;
import com.elitescloud.cloudt.system.service.EmpGroupMngService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectEnum;
import com.elitescloud.cloudt.system.service.model.entity.SysEmpGroupDO;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * 员工组管理.
 *
 * @author Kaiser（wang shao）
 * 2023/1/16
 */
@Api(tags = "员工组管理")
@RestController
@RequestMapping(value = "/mng/empGroup", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectEnum.SYS_EMP_GROUP, businessDoClass = SysEmpGroupDO.class)
public class EmpGroupMngController {

    private final EmpGroupMngService service;

    public EmpGroupMngController(EmpGroupMngService service) {
        this.service = service;
    }

    /**
     * 分页查询员工组
     *
     * @param queryVO 查询参数
     * @return 员工组列表
     */
    @PostMapping(value = "/page")
    @ApiOperation(value = "分页查询员工组")
    @ApiOperationSupport(order = 1)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询员工组", dataPermissionEnabled = true)
    public ApiResult<PagingVO<EmpGroupPageRespVO>> page(@RequestBody EmpGroupPageQueryVO queryVO) {
        return service.page(queryVO);
    }

    /**
     * 切换启用禁用
     *
     * @param id 员工组ID
     * @return 切换结果
     */
    @PatchMapping(value = "/switch/{id}/enabled")
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "员工组ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换启用禁用"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return service.updateEnabled(id);
    }

    /**
     * 获取员工组详情
     *
     * @param id 员工组ID
     * @return 员工组详情
     */
    @GetMapping(value = "/{id}")
    @ApiOperation(value = "获取员工组详情")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "id", value = "员工组ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取员工组详情"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<EmpGroupDetailRespVO> get(@PathVariable("id") Long id) {
        return service.get(id);
    }

    /**
     * 保存员工组
     *
     * @param saveVO 员工组信息
     * @return 员工组ID
     */
    @PostMapping(value = "/save")
    @ApiOperation(value = "保存员工组")
    @ApiOperationSupport(order = 4)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存员工组"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> save(@RequestBody @Valid EmpGroupSaveVO saveVO) {
        return service.save(saveVO);
    }

    /**
     * 删除员工组信息
     *
     * @param id 员工组ID
     * @return 员工组ID
     */
    @DeleteMapping(value = "/{id}")
    @ApiOperation(value = "删除员工组信息")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "员工组ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除员工组"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return service.delete(id);
    }

    /**
     * 分页查询员工组的员工
     *
     * @param queryVO 查询参数
     * @return 员工组列表
     */
    @PostMapping(value = "/employee/page")
    @ApiOperation(value = "分页查询员工组的员工")
    @ApiOperationSupport(order = 11)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询员工组的员工")
    public ApiResult<PagingVO<EmpGroupEmpPageRespVO>> pageEmp(@RequestBody EmpGroupEmpPageQueryVO queryVO) {
        return service.pageEmp(queryVO);
    }

    /**
     * 删除员工组的指定员工
     *
     * @param id         员工组ID
     * @param employeeId 员工ID
     * @return 员工组ID
     */
    @DeleteMapping(value = "/{id}/employee/{employeeId}")
    @ApiOperation(value = "删除员工组的指定员工")
    @ApiOperationSupport(order = 12)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "员工组ID", required = true),
            @ApiImplicitParam(name = "employeeId", value = "员工ID", required = true),
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除员工组的指定员工"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> delete(@PathVariable("id") Long id, @PathVariable("employeeId") Long employeeId) {
        return service.deleteEmployee(id, employeeId);
    }

    /**
     * 新增员工组的员工
     *
     * @param id         员工组ID
     * @param employeeId 员工ID
     * @return 员工组ID
     */
    @PostMapping(value = "/{id}/employee/{employeeId}")
    @ApiOperation(value = "新增员工组的员工")
    @ApiOperationSupport(order = 13)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "员工组ID", required = true),
            @ApiImplicitParam(name = "employeeId", value = "员工ID", required = true),
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.ADD, operationDescription = "新增员工组的员工"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> addEmployee(@PathVariable("id") Long id, @PathVariable("employeeId") Long employeeId) {
        return service.addEmployee(id, employeeId);
    }
}
