package com.elitescloud.cloudt.system.controller.mng.sys;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.swagger.openapi.swagger3.core.util.Json;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.controller.mng.org.EmployeeMngController;
import com.elitescloud.cloudt.system.model.vo.query.sys.OpenApiOperationPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.sys.OpenApiPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.OpenApiOperationPageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.OpenApiPageRespVO;
import com.elitescloud.cloudt.system.service.OpenApiMngService;
import com.elitescloud.cloudt.system.service.OpenApiQueryService;
import com.elitescloud.cloudt.system.service.common.constant.OpenApiSourceEnum;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * OpenApi管理.
 *
 * @author Kaiser（wang shao）
 * @date 2024/1/9
 */
@Api(tags = "OpenApi管理")
@RestController
@RequestMapping(value = "/mng/openapi", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class OpenApiMngController {

    private OpenApiMngService mngService;
    private OpenApiQueryService queryService;

    @ApiOperation(value = "【测试】测试保存OpenApi")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParam(name = "controllerClass", value = "controller类的唯一限定名称")
    @GetMapping(value = "/testSave")
    public Object testSave(@RequestParam(name = "controllerClass", required = false) String controllerClass,
                           @RequestParam(name = "save", required = false, defaultValue = "false") Boolean save) throws Exception {
        if (CharSequenceUtil.isBlank(controllerClass)) {
            controllerClass = EmployeeMngController.class.getName();
        }
        Class<?> controller = Class.forName(controllerClass);
        var openApi = mngService.testAnalyzeController(controller).getData();
        if (Boolean.TRUE.equals(save)) {
            return mngService.updateOpenApi(openApi, OpenApiSourceEnum.GATHER_PULL);
        }

        return Json.pretty(openApi);
    }

    @ApiOperation(value = "根据服务名称拉取最新OpenAPI")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "appCode", value = "应用编码")
    @PostMapping(value = "/pullOpenApiByAppCode")
    public ApiResult<Long> pullOpenApiByAppCode(@RequestParam(name = "appCode") @NotBlank(message = "应用编码为空") String appCode) {
        return mngService.updateOpenApi(appCode);
    }

    /**
     * 分页查询OpenAPI
     *
     * @param queryVO 查询参数
     * @return 分页结果
     */
    @ApiOperation(value = "分页查询OpenAPI")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/pageOpenApi")
    public ApiResult<PagingVO<OpenApiPageRespVO>> pageMng(@RequestBody OpenApiPageQueryVO queryVO) {
        return mngService.pageMng(queryVO);
    }

    /**
     * 分页查询OpenAPI下的操作
     *
     * @param queryVO 查询参数
     * @return 分页结果
     */
    @ApiOperation(value = "分页查询OpenAPI下的操作")
    @ApiOperationSupport(order = 4)
    @PostMapping(value = "/pageOpenApiOperation")
    public ApiResult<PagingVO<OpenApiOperationPageRespVO>> pageMng(@RequestBody OpenApiOperationPageQueryVO queryVO) {
        return mngService.pageMng(queryVO);
    }

    @ApiOperation(value = "根据ID拉取最新OpenAPI")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "记录ID")
    @PutMapping(value = "/pullOpenApi")
    public ApiResult<Long> pullOpenApi(@RequestParam(name = "id") @NotNull(message = "记录ID为空") Long id) {
        return mngService.updateOpenApi(id);
    }

    @GetMapping(value = "/downloadOpenApi")
    @ApiOperation(value = "下载OpenApi", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperationSupport(order = 6)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "OpenApi的唯一标识", required = true, paramType = "query")
    })
    public HttpEntity<StreamingResponseBody> download(@RequestParam(name = "id") @NotNull(message = "记录ID为空") Long id) {
        return queryService.downloadOpenApi(id);
    }

    @GetMapping(value = "/downloadOpenApi/rpc")
    @ApiOperation(value = "根据应用编码远程下载OpenApi", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperationSupport(order = 7)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "appCode", value = "应用编码", required = true, paramType = "query")
    })
    public HttpEntity<StreamingResponseBody> downloadByRpc(@RequestParam(name = "appCode") @NotBlank(message = "应用编码为空") String appCode) {
        return queryService.downloadOpenApiByRpc(appCode);
    }

    @GetMapping(value = "/selectDistinctOperationGroupByApiType")
    @ApiOperation(value = "去重获取全部的接口分组数据，用于内部接口的下拉查询")
    public ApiResult<List<String>>  selectDistinctOperationGroupByApiType() {
        return mngService.selectDistinctOperationGroupByApiType();
    }

    @Autowired
    public void setMngService(OpenApiMngService mngService) {
        this.mngService = mngService;
    }

    @Autowired
    public void setQueryService(OpenApiQueryService queryService) {
        this.queryService = queryService;
    }



}
