package com.elitescloud.cloudt.system.modules.orgtree.service.repo;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.modules.orgtree.common.OrgUdcEnum;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreeDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreeDetailDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreePagingDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.entity.OrgBuTreeDO;
import com.elitescloud.cloudt.system.modules.orgtree.model.entity.QOrgBuTreeDO;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.OrgBuTreePagingParam;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.OrgBuTreeParam;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.OrgBuTreeVListParam;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.OrgBuTreeVPagingParam;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import lombok.val;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author Tristan
 * @date 2020/7/2
 */
@Component
@RequiredArgsConstructor
public class OrgBuTreeRepoProc {

    private static final String DRAFT = OrgUdcEnum.ORG_BUTREE_STATUS_DRAFT.getUdcVal();
    private static final String ACTIVE = OrgUdcEnum.ORG_BUTREE_STATUS_ACTIVE.getUdcVal();

    private final JPAQueryFactory jpaQueryFactory;

    private final QOrgBuTreeDO orgBuTreeDO = QOrgBuTreeDO.orgBuTreeDO;

    private JPAQuery<OrgBuTreePagingDTO> select(){
        var orgBuTreeDO = QOrgBuTreeDO.orgBuTreeDO;
        return jpaQueryFactory.select(
                Projections.bean(
                        OrgBuTreePagingDTO.class,
                        orgBuTreeDO.id,
                        orgBuTreeDO.buTreeCode,
                        orgBuTreeDO.buTreeName,
                        orgBuTreeDO.buTreeType,
                        orgBuTreeDO.buTreeStatus,
                        orgBuTreeDO.nowVersion,
                        orgBuTreeDO.createTime,
                        orgBuTreeDO.modifyTime,
                        orgBuTreeDO.updater,
                        orgBuTreeDO.creator,
                        orgBuTreeDO.versionInstruction
                )
        )
                .from(orgBuTreeDO);
    }

    private Predicate where(OrgBuTreePagingParam param){
        var orgBuTreeDO = QOrgBuTreeDO.orgBuTreeDO;
        Predicate predicate = orgBuTreeDO.deleteFlag.eq(0).or(orgBuTreeDO.deleteFlag.isNull());
        predicate = StringUtils.isBlank(param.getBuTreeCode()) ? predicate : ExpressionUtils.and(predicate, orgBuTreeDO.buTreeCode.like("%"+param.getBuTreeCode()+"%"));
        predicate = StringUtils.isBlank(param.getBuTreeName()) ? predicate : ExpressionUtils.and(predicate, orgBuTreeDO.buTreeName.like("%"+param.getBuTreeName()+"%"));
        predicate = StringUtils.isBlank(param.getBuTreeType()) ? predicate : ExpressionUtils.and(predicate, orgBuTreeDO.buTreeType.eq(param.getBuTreeType()));
        predicate = StringUtils.isBlank(param.getBuTreeStatus()) ? predicate : ExpressionUtils.and(predicate, orgBuTreeDO.buTreeStatus.eq(param.getBuTreeStatus()));
        return predicate;
    }

    public PagingVO<OrgBuTreePagingDTO> searchByParams(OrgBuTreePagingParam param){
        var orgBuTreeDO = QOrgBuTreeDO.orgBuTreeDO;
        val query = this.select().where(this.where(param));
        param.fillOrders(query, orgBuTreeDO);
        param.setPaging(query);
        return PagingVO.<OrgBuTreePagingDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }
    /**
     * 根据组织号获取组织所有版本信息
     *
     * @param param 组织树编码
     * @return list
     */
    public List<OrgBuTreeDTO> getBuTreeListByBuTreeCode(OrgBuTreeVListParam param) {
        List<Predicate> predicates = new ArrayList<>();
        if (param.getId() != null) {
            predicates.add(orgBuTreeDO.id.eq(param.getId()));
        }
        if (StringUtils.isNotBlank(param.getBuTreeCode())) {
            predicates.add(orgBuTreeDO.buTreeCode.eq(param.getBuTreeCode()));
        }
        if (StringUtils.isNotBlank(param.getVersion())) {
            predicates.add(orgBuTreeDO.nowVersion.eq(param.getVersion()));
        }

        if (null != param.getWhetherViewResumeData() && param.getWhetherViewResumeData()) {
            predicates.add(orgBuTreeDO.buTreeStatus.eq(OrgUdcEnum.ORG_BUTREE_STATUS_ACTIVE.getUdcVal())
                    .or(orgBuTreeDO.buTreeStatus.eq(OrgUdcEnum.ORG_BUTREE_STATUS_CLOSED.getUdcVal())));
        }

        return jpaQueryFactory.select(
                Projections.bean(
                        OrgBuTreeDTO.class,
                        orgBuTreeDO.id,
                        orgBuTreeDO.buTreeCode,
                        orgBuTreeDO.buTreeName,
                        orgBuTreeDO.buTreeType,
                        orgBuTreeDO.buTreeStatus,
                        orgBuTreeDO.nowVersion,
                        orgBuTreeDO.createTime,
                        orgBuTreeDO.creator,
                        orgBuTreeDO.releaseTime,
                        orgBuTreeDO.releaseUser,
                        orgBuTreeDO.isNowVersion,
                        orgBuTreeDO.versionInstruction,
                        orgBuTreeDO.disableTime
                )
        )
                .from(orgBuTreeDO)
                .where(ExpressionUtils.allOf(predicates))
                .fetch();
    }

    /**
     * 根据组织树编码获取该组织生效的编码
     *
     * @param buTreeCode 组织树编码
     * @return list
     */
    public List<OrgBuTreeDO> getNowActiveBuTree(String buTreeCode) {
        return jpaQueryFactory
                .select(orgBuTreeDO)
                .from(orgBuTreeDO)
                .where(orgBuTreeDO.buTreeCode.eq(buTreeCode).and(orgBuTreeDO.isNowVersion.eq(true)))
                .fetch();
    }

    /**
     * 根据组织树编码 获取当前生效的版本号
     *
     * @param buTreeCode
     * @return
     */
    public List<String> findActiveVersionByBuTreeCode(String buTreeCode) {
        return jpaQueryFactory.select(
                orgBuTreeDO.nowVersion
        ).from(orgBuTreeDO)
                .where(orgBuTreeDO.buTreeCode.eq(buTreeCode).and(orgBuTreeDO.isNowVersion.eq(true))).fetch();

    }

    /**
     * 根据param 获取组织树信息
     *
     * @param param param
     * @return list
     */
    public List<OrgBuTreeDTO> findOrgBuTreeVoByParam(OrgBuTreeParam param) {
        List<Predicate> predicates = new ArrayList<>();
        if (StringUtils.isNotBlank(param.getBuTreeCode())) {
            predicates.add(orgBuTreeDO.buTreeCode.eq(param.getBuTreeCode()));
        }
        if (StringUtils.isNotBlank(param.getNowVersion())) {
            predicates.add(orgBuTreeDO.nowVersion.eq(param.getNowVersion()));
        }
        if (null != param.getIsNowVersion()) {
            predicates.add(orgBuTreeDO.isNowVersion.eq(param.getIsNowVersion()));
        }
        if (CollectionUtils.isNotEmpty(param.getBuTreeCodes())){
            predicates.add(orgBuTreeDO.buTreeCode.in(param.getBuTreeCodes()));
        }


        if (CollectionUtils.isEmpty(predicates)) {
            return Collections.emptyList();
        }

        return jpaQueryFactory.select(
                Projections.bean(
                        OrgBuTreeDTO.class,
                        orgBuTreeDO.id,
                        orgBuTreeDO.buTreeCode,
                        orgBuTreeDO.buTreeName,
                        orgBuTreeDO.buTreeType,
                        orgBuTreeDO.buTreeStatus,
                        orgBuTreeDO.nowVersion,
                        orgBuTreeDO.createTime,
                        orgBuTreeDO.creator,
                        orgBuTreeDO.releaseTime,
                        orgBuTreeDO.releaseUser,
                        orgBuTreeDO.isNowVersion,
                        orgBuTreeDO.versionInstruction,
                        orgBuTreeDO.disableTime
                )
        )
                .from(orgBuTreeDO)
                .where(ExpressionUtils.allOf(predicates))
                .fetch();
    }

    /**
     * 获取组织树履历分页数据
     *
     * @param param
     * @return
     */
    public PagingVO<OrgBuTreeDTO> getBuTreePagingByParam(OrgBuTreeVPagingParam param) {
        List<Predicate> predicates = new ArrayList<>();
        if (StringUtils.isNotBlank(param.getBuTreeCode())) {
            predicates.add(orgBuTreeDO.buTreeCode.eq(param.getBuTreeCode()));
        }
        // 过滤条件   正常和已停用的组织树
        predicates.add(orgBuTreeDO.buTreeStatus.eq(OrgUdcEnum.ORG_BUTREE_STATUS_ACTIVE.getUdcVal())
                .or(orgBuTreeDO.buTreeStatus.eq(OrgUdcEnum.ORG_BUTREE_STATUS_CLOSED.getUdcVal())));

        JPAQuery<OrgBuTreeDTO> query = jpaQueryFactory.select(
                Projections.bean(
                        OrgBuTreeDTO.class,
                        orgBuTreeDO.id,
                        orgBuTreeDO.buTreeCode,
                        orgBuTreeDO.buTreeName,
                        orgBuTreeDO.buTreeType,
                        orgBuTreeDO.buTreeStatus,
                        orgBuTreeDO.nowVersion,
                        orgBuTreeDO.createTime,
                        orgBuTreeDO.creator,
                        orgBuTreeDO.releaseTime,
                        orgBuTreeDO.releaseUser,
                        orgBuTreeDO.isNowVersion,
                        orgBuTreeDO.versionInstruction,
                        orgBuTreeDO.disableTime
                )
        )
                .from(orgBuTreeDO)
                .where(ExpressionUtils.allOf(predicates));
        param.fillOrders(query, orgBuTreeDO);
        param.setPaging(query);
        return PagingVO.<OrgBuTreeDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }
    /**
     * 根据组织号获取组织版本信息
     *
     * @param buTreeCode 组织树编码
     * @return list
     */
    public List<String> getBuTreeVersionByBuTreeCode(String buTreeCode) {
        return jpaQueryFactory.select(orgBuTreeDO.nowVersion).from(orgBuTreeDO).where(orgBuTreeDO.buTreeCode.eq(buTreeCode)).fetch();
    }
    /**
     * 根据ID查组织树详情
     *
     * @param id id
     * @return 详情信息
     */
    public OrgBuTreeDetailDTO findBuTreeDetailById(Long id) {
        return jpaQueryFactory
                .select(Projections.bean(
                        OrgBuTreeDetailDTO.class,
                        orgBuTreeDO.id,
                        orgBuTreeDO.buTreeType,
                        orgBuTreeDO.buTreeStatus,
                        orgBuTreeDO.buTreeCode,
                        orgBuTreeDO.buTreeName,
                        orgBuTreeDO.buTreeVersion,
                        orgBuTreeDO.nowVersion,
                        orgBuTreeDO.versionInstruction
                ))
                .from(orgBuTreeDO)
                .where(orgBuTreeDO.id.eq(id))
                .fetchOne();
    }
    /**
     * 批量更新组织树状态
     *
     * @param ids          id列表
     * @param buTreeStatus 状态
     */
    public void updateStatusByIds(List<Long> ids, String buTreeStatus) {
        jpaQueryFactory
                .update(orgBuTreeDO)
                .set(orgBuTreeDO.buTreeStatus, buTreeStatus)
                .set(orgBuTreeDO.isNowVersion,false)
                .set(orgBuTreeDO.disableTime, LocalDateTime.now())
                .where(orgBuTreeDO.id.in(ids))
                .execute();
    }
}
