package com.elitescloud.cloudt.system.modules.warning.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.common.param.IdNameParam;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgTemplateRepoProc;
import com.elitescloud.cloudt.system.modules.warning.common.*;
import com.elitescloud.cloudt.system.modules.warning.service.SysWarningRuleMngService;
import com.elitescloud.cloudt.system.modules.warning.service.model.entity.SysWarningRuleDO;
import com.elitescloud.cloudt.system.modules.warning.service.model.entity.SysWarningSceneCondDO;
import com.elitescloud.cloudt.system.modules.warning.service.repo.SysWarningRuleRepoProc;
import com.elitescloud.cloudt.system.modules.warning.service.repo.SysWarningScendCondRepoProc;
import com.elitescloud.cloudt.system.modules.warning.web.model.vo.query.SysWarningRulePageQueryVO;
import com.elitescloud.cloudt.system.modules.warning.web.model.vo.resp.SysWarningRuleDetailRespVO;
import com.elitescloud.cloudt.system.modules.warning.web.model.vo.resp.SysWarningRulePageRespVO;
import com.elitescloud.cloudt.system.modules.warning.web.model.vo.save.SysWarningRuleSaveVO;
import com.elitescloud.cloudt.system.service.repo.RoleRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2025/8/4 周一
 */
@Service
public class SysWarningRuleMngServiceImpl extends AbstractSysWarningService implements SysWarningRuleMngService {

    @Autowired
    private SysWarningRuleRepoProc warningRuleRepoProc;
    @Autowired
    private SysWarningScendCondRepoProc warningScendCondRepoProc;
    @Autowired
    private SysMsgTemplateRepoProc msgTemplateRepoProc;
    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;
    @Autowired
    private SeqNumProvider seqNumProvider;
    @Autowired
    private UdcProvider udcProvider;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(SysWarningRuleSaveVO saveVO) {
        SysWarningRuleDO ruleDO = this.checkAndConvertRule(saveVO);
        warningRuleRepoProc.save(ruleDO);

        // 场景，先删后增
        List<SysWarningSceneCondDO> sceneCondDOList = this.checkAndConvertScene(ruleDO.getRuleCode(), saveVO.getSceneConditionList());
        warningScendCondRepoProc.deleteByRuleCode(ruleDO.getRuleCode());
        if (!sceneCondDOList.isEmpty()) {
            warningScendCondRepoProc.save(sceneCondDOList);
        }
        return ApiResult.ok(ruleDO.getId());
    }

    @Override
    public ApiResult<PagingVO<SysWarningRulePageRespVO>> pageMng(SysWarningRulePageQueryVO queryVO) {
        var pageData = warningRuleRepoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        var udcSceneMap = udcProvider.getValueMapByUdcCode(Application.NAME, SysWarningSceneUdc.UDC_NAME);
        pageData.each(respVO -> {
            respVO.setSceneName(udcSceneMap.get(respVO.getScene()));

            var execTimeType = SysWarningTimeEnum.getByValue(respVO.getExecTimeType());
            respVO.setExecTimeTypeName(execTimeType == null ? null : execTimeType.getDescription());
            respVO.setExecTimePretty(convertPrettyExecType(execTimeType, respVO.getExecTimeValues(), respVO.getExecStartTime()));

            var noticeRate = SysWarningNoticeRateEnum.getByValue(respVO.getNoticeRateValue());
            respVO.setNoticeRateValueName(noticeRate == null ? null : noticeRate.getDescription());

            if (StrUtil.isNotBlank(respVO.getSendTypes())) {
                respVO.setSendTypeList(Arrays.stream(respVO.getSendTypes().split(",")).toList());
                respVO.setSendTypeList(Arrays.stream(respVO.getSendTypes().split(",")).map(t -> {
                    var sendType = MsgSendTypeEnum.parse(t);
                    return sendType == null ? null : sendType.getDescription();
                }).toList());
            }
        });
        return ApiResult.ok(pageData);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> updateEnabled(List<Long> ids, Boolean enabled) {
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.ok(ids);
        }

        warningRuleRepoProc.updateEnabled(ids, ObjUtil.defaultIfNull(enabled, false));
        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> delete(List<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.ok(ids);
        }

        warningScendCondRepoProc.deleteByRuleIds(ids);
        warningRuleRepoProc.delete(ids);

        return ApiResult.ok(ids);
    }

    @Override
    public ApiResult<SysWarningRuleDetailRespVO> getDetail(Long id) {
        if (id == null) {
            return ApiResult.fail("ID为空");
        }
        SysWarningRuleDO ruleDO = warningRuleRepoProc.get(id);
        if (ruleDO == null) {
            return ApiResult.ok();
        }

        SysWarningRuleDetailRespVO respVO = new SysWarningRuleDetailRespVO();
        respVO.setId(ruleDO.getId());
        respVO.setRuleCode(ruleDO.getRuleCode());
        respVO.setRuleName(ruleDO.getRuleName());
        respVO.setEnabled(ruleDO.getEnabled());
        respVO.setDescription(ruleDO.getDescription());

        var sceneMap = udcProvider.getValueMapByUdcCode(Application.NAME, SysWarningSceneUdc.UDC_NAME);
        respVO.setScene(ruleDO.getScene());
        respVO.setSceneName(sceneMap.get(ruleDO.getScene()));

        respVO.setSceneConditionList(getSceneConditionList(id));
        respVO.setTimeCondition(getTimeConditionList(ruleDO));
        respVO.setNotice(getNoticeMsg(ruleDO));

        return ApiResult.ok(respVO);
    }

    private SysWarningRuleDetailRespVO.NoticeMsg getNoticeMsg(SysWarningRuleDO ruleDO) {
        SysWarningRuleDetailRespVO.NoticeMsg noticeMsg = new SysWarningRuleDetailRespVO.NoticeMsg();
        noticeMsg.setTmplName(StrUtil.isBlank(ruleDO.getNoticeTmplCode()) ? null : msgTemplateRepoProc.getNameByTemplateCode(ruleDO.getNoticeTmplCode()));
        if (noticeMsg.getTmplName() != null) {
            noticeMsg.setTmplCode(ruleDO.getNoticeTmplCode());
        }

        if (StrUtil.isBlank(ruleDO.getSendTypes())) {
            noticeMsg.setSendTypeList(Collections.emptyList());
            noticeMsg.setSendTypeNameList(Collections.emptyList());
        } else {
            noticeMsg.setSendTypeList(Arrays.stream(ruleDO.getSendTypes().split(",")).map(MsgSendTypeEnum::parse).filter(Objects::nonNull).toList());
            noticeMsg.setSendTypeNameList(noticeMsg.getSendTypeList().stream().map(MsgSendTypeEnum::getDescription).toList());
        }

        noticeMsg.setEnabledRecvByRole(ruleDO.getEnabledRecvByRole());

        List<String> roleCodes = StrUtil.isBlank(ruleDO.getRecvRoleCodes()) ? Collections.emptyList() : Arrays.stream(ruleDO.getRecvRoleCodes().split(",")).toList();
        if (roleCodes.isEmpty()) {
            noticeMsg.setRecvRoleCodes(Collections.emptyList());
            noticeMsg.setRecvRoleNames(Collections.emptyList());
        } else {
            var roles = roleRepoProc.queryIdCodeNamesByCode(roleCodes);
            noticeMsg.setRecvRoleCodes(roles.stream().map(IdCodeNameParam::getCode).toList());
            noticeMsg.setRecvRoleNames(roles.stream().map(IdCodeNameParam::getName).toList());
        }

        noticeMsg.setOnlyPushWarningObj(ruleDO.getOnlyPushWarningObj());
        noticeMsg.setEnabledRecvByUser(ruleDO.getEnabledRecvByUser());

        List<Long> userIds = StrUtil.isBlank(ruleDO.getRecvUserIds()) ? Collections.emptyList() : Arrays.stream(ruleDO.getRecvUserIds().split(",")).map(Long::parseLong).toList();
        if (userIds.isEmpty()) {
            noticeMsg.setRecvUserIds(Collections.emptyList());
            noticeMsg.setRecvUserNames(Collections.emptyList());
        } else {
            var users = tenantDataIsolateProvider.byDefaultDirectly(() -> userRepoProc.queryUserName(userIds));
            noticeMsg.setRecvUserIds(users.stream().map(IdNameParam::getId).toList());
            noticeMsg.setRecvUserNames(users.stream().map(IdNameParam::getName).toList());
        }

        return noticeMsg;
    }

    private SysWarningRuleDetailRespVO.ExecTime getTimeConditionList(SysWarningRuleDO ruleDO) {
        SysWarningRuleDetailRespVO.ExecTime execTime = new SysWarningRuleDetailRespVO.ExecTime();

        SysWarningTimeEnum timeType = SysWarningTimeEnum.getByValue(ruleDO.getExecTimeType());
        execTime.setTimeType(timeType);
        execTime.setTimeTypeName(timeType == null ? null : timeType.getDescription());

        execTime.setValueList(convertExecType(timeType, ruleDO.getExecTimeValues()));
        execTime.setStartTime(ruleDO.getExecStartTime());
        execTime.setExcludedWorkday(ruleDO.getExcludedWorkday());
        execTime.setWorkdayStartTime(ruleDO.getWorkdayStartTime());
        execTime.setWorkdayEndTime(ruleDO.getWorkdayEndTime());
        execTime.setExcludedHolidays(ruleDO.getExcludedHolidays());

        SysWarningNoticeRateEnum noticeRate = SysWarningNoticeRateEnum.getByValue(ruleDO.getNoticeRateValue());
        execTime.setNoticeRate(noticeRate);
        execTime.setNoticeRateName(noticeRate == null ? null : noticeRate.getDescription());

        execTime.setNoticeRateValue(ruleDO.getCustomNoticeRate());

        return execTime;
    }

    private List<SysWarningRuleDetailRespVO.SceneCondition> getSceneConditionList(Long ruleId) {
        var sceneCondList = warningScendCondRepoProc.listByRuleId(ruleId);
        if (sceneCondList.isEmpty()) {
            return Collections.emptyList();
        }

        var termMap = udcProvider.getValueMapByUdcCode(Application.NAME, SysWarningSceneTermUdc.UDC_NAME);
        return sceneCondList.stream().map(t -> {
            SysWarningRuleDetailRespVO.SceneCondition cond = new SysWarningRuleDetailRespVO.SceneCondition();
            cond.setCondTerm(t.getCondTerm());
            cond.setCondTermName(termMap.get(t.getCondTerm()));

            var opr = SysWarningOprEnum.getByValue(t.getOprValue());
            cond.setOpr(opr);
            cond.setOprName(opr == null ? null : opr.getDescription());

            cond.setValueList(JSONUtil.json2Obj(t.getValueListJson(), new TypeReference<>() {
            }));

            SysWarningRelEnum rel = SysWarningRelEnum.getByValue(t.getRelValue());
            cond.setRel(rel);
            cond.setRelName(rel == null ? null : rel.getDescription());

            return cond;
        }).toList();
    }

    private List<SysWarningSceneCondDO> checkAndConvertScene(String ruleCode, List<SysWarningRuleSaveVO.SceneCondition> sceneConditionList) {
        if (CollUtil.isEmpty(sceneConditionList)) {
            return Collections.emptyList();
        }

        AtomicInteger seqNum = new AtomicInteger(0);
        return sceneConditionList.stream().map(t -> {
            SysWarningSceneCondDO sceneCondDO = new SysWarningSceneCondDO();
            sceneCondDO.setRuleCode(ruleCode);
            sceneCondDO.setCondTerm(t.getCondTerm());

            Assert.notNull(t.getOpr(), "条件值不能为空");
            sceneCondDO.setOprValue(t.getOpr().name());

            Assert.notNull(t.getValueList(), "数据范围为空");
            sceneCondDO.setValueListJson(JSONUtil.toJsonString(t.getValueList()));

            sceneCondDO.setRelValue(t.getRel() == null ? null : t.getRel().name());

            sceneCondDO.setSortNo(seqNum.incrementAndGet());

            return sceneCondDO;
        }).collect(Collectors.toList());
    }

    private SysWarningRuleDO checkAndConvertRule(SysWarningRuleSaveVO saveVO) {
        SysWarningRuleDO ruleDO = saveVO.getId() == null ? new SysWarningRuleDO() : warningRuleRepoProc.get(saveVO.getId());

        boolean isNew = saveVO.getId() == null;
        if (isNew) {
            if (CharSequenceUtil.isBlank(saveVO.getRuleCode())) {
                String ruleCode = seqNumProvider.generateCode(Application.NAME, "sys_warning_rule_code", null);
                Assert.notBlank(ruleCode, "");
                ruleDO.setRuleCode(ruleCode);
            } else {
                ruleDO.setRuleCode(saveVO.getRuleCode());
            }
            Assert.isFalse(warningRuleRepoProc.existsRuleCode(saveVO.getRuleCode()), "编码已存在");
        } else {
            Assert.notNull(ruleDO, "修改的数据不存在");
        }

        ruleDO.setRuleName(saveVO.getRuleName());
        ruleDO.setEnabled(ObjectUtil.defaultIfNull(saveVO.getEnabled(), false));
        ruleDO.setDescription(saveVO.getDescription());
        ruleDO.setScene(saveVO.getScene());

        // 执行时间
        checkAndConvertExecTimeCond(ruleDO, saveVO.getTimeCondition());

        // 推送用户
        checkAndConvertExecNotice(ruleDO, saveVO.getNotice());

        return ruleDO;
    }

    private void checkAndConvertExecTimeCond(SysWarningRuleDO ruleDO, SysWarningRuleSaveVO.ExecTime execTimeCond) {
        Assert.notNull(execTimeCond, "执行条件时间为空");

        SysWarningTimeEnum timeType = execTimeCond.getTimeType();
        Assert.notNull(timeType, "执行条件的时间类型为空");

        ruleDO.setExecTimeType(timeType.name());
        if (SysWarningTimeEnum.DAY == timeType) {
            // nothing to do
        } else if (SysWarningTimeEnum.WEEK == timeType) {
            Assert.notEmpty(execTimeCond.getValueList(), "执行时间的每周的时间为空");

            String values = execTimeCond.getValueList().stream()
                    .map(t -> super.obtainDayOfWeek(t) + "").collect(Collectors.joining(","));
            ruleDO.setExecTimeValues(values);
        } else if (SysWarningTimeEnum.MONTH == timeType) {
            Assert.notEmpty(execTimeCond.getValueList(), "执行时间的每月的时间为空");

            String values = execTimeCond.getValueList().stream()
                    .map(t -> super.obtainDayOfMonth(t) + "").collect(Collectors.joining(","));
            ruleDO.setExecTimeValues(values);
        } else if (SysWarningTimeEnum.CUSTOM == timeType) {
            Assert.notEmpty(execTimeCond.getValueList(), "执行时间的自定义的时间区间为空");

            String values = execTimeCond.getValueList().stream().limit(2)
                    .peek(t -> {
                        try {
                            DatetimeUtil.parseLocalDate(t);
                        } catch (Exception e) {
                            throw new IllegalArgumentException("日期格式为yyyy-MM-dd");
                        }

                    }).collect(Collectors.joining(","));
            ruleDO.setExecTimeValues(values);
        } else {
            throw new BusinessException("暂不支持的时间类型");
        }

        Assert.notNull(execTimeCond.getStartTime(), "开始时间为空");
        ruleDO.setExecStartTime(execTimeCond.getStartTime());

        // 免打扰
        ruleDO.setExcludedWorkday(ObjectUtil.defaultIfNull(execTimeCond.getExcludedWorkday(), false));
        if (ruleDO.getExcludedWorkday()) {
            Assert.notNull(execTimeCond.getWorkdayStartTime(), "工作日开始时间为空");
            Assert.notNull(execTimeCond.getWorkdayEndTime(), "工作日结束时间为空");
            ruleDO.setWorkdayStartTime(execTimeCond.getWorkdayStartTime());
            ruleDO.setWorkdayEndTime(execTimeCond.getWorkdayEndTime());
        }
        ruleDO.setExcludedHolidays(ObjectUtil.defaultIfNull(execTimeCond.getExcludedHolidays(), false));

        Assert.notNull(execTimeCond.getNoticeRate(), "通知频率为空");
        ruleDO.setNoticeRateValue(execTimeCond.getNoticeRate().name());
        ruleDO.setCustomNoticeRate(execTimeCond.getNoticeRateValue());
    }

    private void checkAndConvertExecNotice(SysWarningRuleDO ruleDO, SysWarningRuleSaveVO.NoticeMsg noticeMsg) {
        Assert.notNull(noticeMsg, "消息通知为空");

        ruleDO.setNoticeTmplCode(noticeMsg.getTmplCode());

        // 推送类型
        Assert.notEmpty(noticeMsg.getSendTypeList(), "推送方式为空");
        var sendTypes = noticeMsg.getSendTypeList().stream().map(MsgSendTypeEnum::name).collect(Collectors.joining(","));
        ruleDO.setSendTypes(sendTypes);

        // 推送角色
        ruleDO.setEnabledRecvByRole(ObjectUtil.defaultIfNull(noticeMsg.getEnabledRecvByRole(), false));
        if (ruleDO.getEnabledRecvByRole()) {
            Assert.notEmpty(noticeMsg.getRecvRoleCodes(), "推送对象角色为空");
            ruleDO.setRecvRoleCodes(noticeMsg.getRecvRoleCodes().stream().distinct().collect(Collectors.joining(",")));
        }
        ruleDO.setOnlyPushWarningObj(ObjectUtil.defaultIfNull(noticeMsg.getOnlyPushWarningObj(), false));

        // 推送用户
        ruleDO.setEnabledRecvByUser(ObjectUtil.defaultIfNull(noticeMsg.getEnabledRecvByUser(), false));
        if (ruleDO.getEnabledRecvByUser()) {
            Assert.notNull(noticeMsg.getRecvUserIds(), "推送用户不能为空");
            ruleDO.setRecvUserIds(noticeMsg.getRecvUserIds().stream().distinct().map(Object::toString).collect(Collectors.joining(",")));
        }
    }
}
