package com.elitescloud.cloudt.system.rpc;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.req.UserRoleSaveDTO;
import com.elitescloud.cloudt.system.param.AuthUserIdQueryDTO;
import com.elitescloud.cloudt.system.provider.SysUserPermissionRpcService;
import com.elitescloud.cloudt.system.service.AuthUserService;
import com.elitescloud.cloudt.system.service.PermissionMngService;
import com.elitescloud.cloudt.system.service.PermissionQueryService;
import com.elitescloud.cloudt.system.service.UserQueryService;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.security.crypto.encrypt.TextEncryptor;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Set;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/23/2023
 */
@RestController
@RequestMapping(value = SysUserPermissionRpcService.URI, produces = MediaType.APPLICATION_JSON_VALUE)
public class SysUserPermissionRpcServiceImpl implements SysUserPermissionRpcService {
    private static final Logger logger = LoggerFactory.getLogger(SysUserPermissionRpcServiceImpl.class);

    private PermissionMngService permissionMngService;
    private PermissionQueryService permissionQueryService;
    private AuthUserService authUserService;
    private TextEncryptor encryptor;
    private UserRepoProc userRepoProc;
    private TenantDataIsolateProvider tenantDataIsolateProvider;

    @Override
    public ApiResult<List<IdCodeNameParam>> listAllRoles(String tenantCode) {
        return permissionQueryService.listAllRoles(tenantCode);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> listAllRolesByTenantOrg(String tenantOrgCode) {
        return permissionQueryService.listAllRolesByTenantOrg(tenantOrgCode);
    }

    @Override
    public ApiResult<Long> saveUserRoles(Long userId, List<Long> roleIds) {
        return permissionMngService.saveUserRoles(userId, roleIds);
    }

    @Override
    public ApiResult<List<Long>> saveUserRoles(List<UserRoleSaveDTO> saveDTOList) {
        return permissionMngService.saveUserRoles(saveDTOList);
    }

    @Override
    public ApiResult<Set<Long>> getUserIdByRoleCode(String roleCode) {
        return permissionQueryService.getUserIdByRoleCode(roleCode);
    }

    @Override
    public ApiResult<List<SysUserBasicDTO>> listUserByRole(String roleCode) {
        return permissionQueryService.listUserByRole(roleCode);
    }

    @Override
    public ApiResult<OAuthToken> authByUserId(AuthUserIdQueryDTO queryDTO) {
        if (authUserService == null) {
            return ApiResult.fail("暂不支持认证");
        }

        String userIdTxt = null;
        try {
            userIdTxt = this.decrypt(queryDTO.getUserIdCypher());
        } catch (Exception e) {
            return ApiResult.fail("解密用户ID失败，请确认加密方式正确");
        }
        if (CharSequenceUtil.isBlank(userIdTxt)) {
            return ApiResult.fail("用户标识为空");
        }
        return authUserService.authenticate(null, null, InternalAuthenticationGranter.IdType.USER_ID.name(), userIdTxt);
    }

    @Override
    public ApiResult<OAuthToken> authByUserMobile(AuthUserIdQueryDTO queryDTO) {
        if (authUserService == null) {
            return ApiResult.fail("暂不支持认证");
        }

        String mobileTxt = null;
        try {
            mobileTxt = this.decrypt(queryDTO.getUserIdCypher());
        } catch (Exception e) {
            return ApiResult.fail("解密用户手机号失败，请确认加密方式正确");
        }
        if (CharSequenceUtil.isBlank(mobileTxt)) {
            return ApiResult.fail("用户手机号为空");
        }
        return authUserService.authenticate(null, null, InternalAuthenticationGranter.IdType.MOBILE.name(), mobileTxt);
    }

    @Override
    public ApiResult<OAuthToken> authByUserEmail(AuthUserIdQueryDTO queryDTO) {
        if (authUserService == null) {
            return ApiResult.fail("暂不支持认证");
        }

        String emailTxt = null;
        try {
            emailTxt = this.decrypt(queryDTO.getUserIdCypher());
        } catch (Exception e) {
            return ApiResult.fail("解密用户邮箱失败，请确认加密方式正确");
        }
        if (CharSequenceUtil.isBlank(emailTxt)) {
            return ApiResult.fail("用户邮箱为空");
        }
        return authUserService.authenticate(null, null, InternalAuthenticationGranter.IdType.EMAIL.name(), emailTxt);
    }

    @Override
    public ApiResult<OAuthToken> authByOuterKey(AuthUserIdQueryDTO queryDTO) {
        if (authUserService == null) {
            return ApiResult.fail("暂不支持认证");
        }

        String outerKeyTxt = null;
        try {
            outerKeyTxt = this.decrypt(queryDTO.getUserIdCypher());
        } catch (Exception e) {
            return ApiResult.fail("解密用户外部标识失败，请确认加密方式正确");
        }
        if (CharSequenceUtil.isBlank(outerKeyTxt)) {
            return ApiResult.fail("用户外部标识为空");
        }

        var ids = userRepoProc.getIdByOuterKey(outerKeyTxt);
        if (ids.isEmpty()) {
            return ApiResult.fail("账户不存在");
        }
        if (ids.size() > 1) {
            return ApiResult.fail("存在重复的账号标识");
        }
        return authUserService.authenticate(null, null, InternalAuthenticationGranter.IdType.USER_ID.name(), ids.get(0).toString());
    }

    @Override
    public ApiResult<String> getTokenByTicket(String ticket) {
        if (authUserService == null) {
            return ApiResult.fail("暂不支持ticket换token");
        }
        return authUserService.ticket2Token(ticket);
    }

    private String decrypt(String cipher) {
        if (CharSequenceUtil.isBlank(cipher)) {
            return null;
        }

        try {
            return encryptor.decrypt(cipher);
        } catch (Exception e) {
            logger.info("解密失败，密文：{}，异常：", cipher, e);
            throw new IllegalArgumentException("解密失败：" + e.getMessage());
        }
    }

    @Autowired
    public void setPermissionMngService(PermissionMngService permissionMngService) {
        this.permissionMngService = permissionMngService;
    }

    @Autowired
    public void setPermissionQueryService(PermissionQueryService permissionQueryService) {
        this.permissionQueryService = permissionQueryService;
    }

    @Autowired(required = false)
    public void setAuthUserService(AuthUserService authUserService) {
        this.authUserService = authUserService;
    }

    @Autowired
    public void setEncryptor(TextEncryptor encryptor) {
        this.encryptor = encryptor;
    }

    @Autowired
    public void setUserRepoProc(UserRepoProc userRepoProc) {
        this.userRepoProc = userRepoProc;
    }

    @Autowired
    public void setTenantDataIsolateProvider(TenantDataIsolateProvider tenantDataIsolateProvider) {
        this.tenantDataIsolateProvider = tenantDataIsolateProvider;
    }
}
