package com.elitescloud.cloudt.system.service;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.InfinityApiConvert;
import com.elitescloud.cloudt.system.convert.InfinityFolderConvert;
import com.elitescloud.cloudt.system.model.entity.InfinityApiDO;
import com.elitescloud.cloudt.system.model.entity.InfinityFolderDO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.*;
import com.elitescloud.cloudt.system.service.repo.InfinityApiRepo;
import com.elitescloud.cloudt.system.service.repo.InfinityApiRepoProc;
import com.elitescloud.cloudt.system.service.repo.InfinityFolderRepoProc;
import com.elitescloud.cloudt.system.util.BeanUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Auther: Mark
 * @Date: 2024/12/13 16:20
 * @Description:
 */

@Slf4j
@Service
@RequiredArgsConstructor
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class InfinityApiServiceImpl implements InfinityApiService {

    private final InfinityApiRepo infinityApiRepo;
    private final InfinityApiRepoProc infinityApiRepoProc;
    private final InfinityFolderRepoProc folderRepoProc;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    private long currentTenantId() {
        var tenant = tenantClientProvider.getSessionTenant();
        if (tenant == null) {
            tenant = tenantClientProvider.getCurrentTenant();
        }
        return tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId();
    }

    /**
     * 根据参数获取无限API列表，并按照目录进行分组和排序
     *
     * @param query API查询参数对象，包含查询所需的信息
     * @return 返回一个包含API信息的列表，这些API根据目录进行了分组和排序
     */
    @Override
    public List<InfinityApiRespVO> infinityApiByParam(ApiQueryParamVO query) {
        // 所有接口
        List<InfinityApiRespVO> responseVO = infinityApiRepoProc.infinityApiByParam(query);
        // 所有目录
        Map<Long, InfinityApiRespVO> folderMap = queryFolderByParam(query);
        // 按目录分组
        Map<Long, List<InfinityApiRespVO>> apiMap = responseVO.stream()
                .filter(row -> row.getFolderId() != null)
                .collect(Collectors.groupingBy(InfinityApiRespVO::getFolderId));
        responseVO.removeIf(row -> row.getFolderId() != null);
        // 配置目录下的接口列表
        apiMap.forEach((folderId, children) -> {
            var curr = BeanUtils.toBean(folderMap.get(folderId), InfinityApiRespVO.class);
            children.sort(InfinityApiRespVO::compareTo);
            curr.setChildren(children);
            responseVO.add(curr);
            // 排除已添加的目录
            folderMap.remove(folderId);
        });
        // 追加未添加的目录
        folderMap.forEach((folderId, api) -> responseVO.add(api));
        // 按名称，对目录/接口排序
        responseVO.sort(InfinityApiRespVO::compareTo);
        return responseVO;
    }

    /**
     * 查询该平台下，所有分组
     * <p>
     * 该方法根据给定的查询参数，获取平台下所有的分组信息，并将其转换为API响应格式
     * 使用了存储库过程来获取分组数据，并通过转换器将数据转换为合适的格式
     * 最后，将转换后的分组信息以Map形式返回，便于快速查找
     *
     * @param query 查询参数对象，包含用于过滤分组的参数
     * @return 返回一个Map，键为分组ID，值为包含分组详细信息的InfinityApiRespVO对象
     */
    private Map<Long, InfinityApiRespVO> queryFolderByParam(ApiQueryParamVO query) {
        // 根据查询参数获取分组数据
        List<InfinityFolderDO> folderRespVOS = folderRepoProc.infinityFolderByParam(query);
        // 将分组数据转换为API响应格式
        List<InfinityApiRespVO> folders = InfinityFolderConvert.INSTANCE.toApiFolder(folderRespVOS);
        // 将分组信息收集到Map中，以便通过分组ID快速访问
        return folders.stream().collect(Collectors.toMap(InfinityApiRespVO::getFolderId, Function.identity()));
    }


    /**
     * 根据ID获取InfinityApi的详细信息
     * <p>
     * 此方法首先通过调用infinityApiRepo的findById方法来查找指定ID的InfinityApiDO对象
     * 如果找到对象，则将其转换为InfinityApiDetailVO对象并返回
     * 如果未找到对象，则抛出一个表示数据未找到的BusinessException异常
     *
     * @param id InfinityApi的唯一标识符
     * @return 返回InfinityApi的详细信息视图对象
     * @throws BusinessException 当找不到对应ID的数据时抛出异常
     */
    @Override
    public InfinityApiDetailVO infinityApiById(Long id) {
        Optional<InfinityApiDO> infinityApiDO = infinityApiRepo.findById(id);
        if (infinityApiDO.isEmpty()) {
            throw new BusinessException("Not Found Data");
        }
        return InfinityApiConvert.INSTANCE.doToVO(infinityApiDO.get());
    }

    /**
     * 根据API代码获取Infinity API详细信息
     * <p>
     * 此方法首先通过API代码从数据源获取Infinity API的合并视图（InfinityApiMergeVO），
     * 如果找不到相关数据，则抛出业务异常表示数据未找到
     * 如果数据存在，则将其转换为详细视图（InfinityApiDetailVO）并返回
     *
     * @param apiCode API代码，用于唯一标识一个API
     * @return InfinityApiDetailVO API的详细信息视图对象
     * @throws BusinessException 当找不到数据时抛出的业务异常
     */
    @Override
    public InfinityApiDetailVO infinityApiByApiCode(String apiCode) {
        // 根据API代码从数据源获取API的合并视图
        InfinityApiMergeVO apiMergeVO = infinityApiRepoProc.findByApiCode(apiCode);
        // 检查获取的结果是否为空，如果为空则抛出异常
        if (ObjectUtil.isNull(apiMergeVO)) {
            throw new BusinessException("Not Found Data");
        }
        // 将获取到的合并视图转换为详细视图并返回
        return InfinityApiConvert.INSTANCE.mergeToDetailVO(apiMergeVO);
    }

    /**
     * 保存API信息
     * <p>
     * 此方法首先检查给定的API代码是否已存在于系统中如果存在，则抛出异常如果不存在，则根据saveParam中的信息
     * 决定是新增还是修改API信息使用@Transactional注解确保操作的原子性，遇到异常时回滚事务
     *
     * @param saveParam 包含要保存的API信息的参数对象
     * @return 返回保存后的API的ID
     * @throws BusinessException 当数据不存在时抛出的自定义异常
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long save(ApiSaveParamVO saveParam) {
        // 检查API代码是否已存在
        boolean exists = infinityApiRepoProc.existsByCode(saveParam.getId(), saveParam.getApiCode());
        Assert.isFalse(exists, "编码已存在");

        // TODO 参数规则检查
        // 新增
        if (saveParam.isNew()) {
            // 将保存参数转换为DO并保存到数据库
            InfinityApiDO infinityApiDO = InfinityApiConvert.INSTANCE.saveParamToDO(saveParam);
            infinityApiDO.setTenantId(currentTenantId());
            infinityApiRepo.save(infinityApiDO);
            return infinityApiDO.getId();
        }
        // 修改
        else {
            // 根据ID查找API信息
            Optional<InfinityApiDO> infinityApiDO = infinityApiRepo.findById(saveParam.getId());
            if (infinityApiDO.isEmpty()) {
                throw new BusinessException("Not Found Data");
            }
            InfinityApiDO infinityApi = infinityApiDO.get();
            // 将保存参数合并到现有的API信息中并保存
            InfinityApiConvert.INSTANCE.saveParamMergeToDO(saveParam, infinityApi);
            infinityApi.setTenantId(currentTenantId());
            infinityApiRepo.save(infinityApi);
            return infinityApi.getId();
        }
    }

    /**
     * 重写更新状态方法
     * 使用事务管理，确保数据一致性
     *
     * @param id     要更新的记录的ID
     * @param status 新的状态值
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(Long id, int status) {
        // 调用存储过程更新状态
        infinityApiRepoProc.updateStatus(id, status);
    }

    /**
     * 删除指定资源
     * <p>
     * 本方法通过调用底层存储接口，根据提供的资源标识列表批量删除资源
     * 它使用事务管理，确保在删除过程中遇到异常时能够回滚更改
     *
     * @param ids 要删除的资源的标识列表这些标识是长整型数字，代表资源在存储中的唯一键
     * @return 返回实际被删除的资源数量这可能与请求删除的数量不同，如果某些资源未找到或无法删除
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public long delete(List<Long> ids) {
        return infinityApiRepoProc.delete(ids);
    }
}
