package com.elitescloud.cloudt.system.service;

import com.elitescloud.boot.common.param.IdNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.SysUserTypeDTO;
import com.elitescloud.cloudt.system.dto.req.UserQueryDTO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserListQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserListRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserQueryRespVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;

import java.util.List;
import java.util.Set;

/**
 * 用户相关通用接口.
 *
 * @author Kaiser（wang shao）
 * 2022/9/28
 */
public interface UserQueryService {

    /**
     * 根据账号获取用户ID
     *
     * @param username 登录号
     * @return 用户ID
     */
    ApiResult<Long> getIdByUsername(String username);

    /**
     * 判断账号是否存在
     *
     * @param username 登录号
     * @return 是否存在
     */
    ApiResult<Boolean> existsUsername(String username);

    /**
     * 判断手机号是否存在
     *
     * @param mobile 手机号
     * @return 是否存在
     */
    ApiResult<Boolean> existsMobile(String mobile);

    /**
     * 判断邮箱是否存在
     *
     * @param email 邮箱
     * @return 是否存在
     */
    ApiResult<Boolean> existsEmail(String email);

    /**
     * 判断外部标识是否存在
     *
     * @param outerKey 外部标识
     * @return 是否存在
     */
    ApiResult<Boolean> existsOuterKey(String outerKey);

    /**
     * 根据ID获取用户
     *
     * @param id 用户ID
     * @return 用户信息
     */
    ApiResult<SysUserBasicDTO> getBasicById(Long id);

    /**
     * 根据登录号获取
     *
     * @param username 登录号
     * @return 账号信息
     */
    ApiResult<SysUserBasicDTO> getBasicByUsername(String username);

    /**
     * 根据用户ID获取用户
     *
     * @param id 用户ID
     * @return 用户信息
     */
    ApiResult<List<SysUserBasicDTO>> getBasicById(List<Long> id);

    /**
     * 查询用户
     *
     * @param queryParam 查询参数
     * @return 用户列表
     */
    ApiResult<List<SysUserBasicDTO>> queryUser(UserQueryDTO queryParam);

    /**
     * 用户查询
     *
     * @param queryVO 查询参数
     * @return 用户信息列表
     */
    ApiResult<PagingVO<UserQueryRespVO>> pageQuery(UserQueryVO queryVO);

    /**
     * 查询用户列表
     *
     * @param queryVO 查询参数
     * @return 用户列表
     */
    ApiResult<List<UserListRespVO>> listUser(UserListQueryVO queryVO);

    /**
     * 根据账号ID查询姓名
     *
     * @param ids 账号ID
     * @return 姓名
     */
    ApiResult<List<IdNameParam>> queryUserName(Set<Long> ids);

    /**
     * 获取账号的用户类型
     *
     * @param id 账号ID
     * @return 用户类型
     */
    ApiResult<List<SysUserTypeDTO>> getUserTypes(Long id);

    /**
     * 根据用户账号查询用户信息
     * <p>
     * 主要用于登录
     *
     * @param username 用户账号
     * @return 用户信息
     */
    SysUserDTO getUserByUsername(String username);

    /**
     * 根据手机号查询用户信息
     * <p>
     * 主要用于登录
     *
     * @param mobile 手机号
     * @return 用户信息
     */
    SysUserDTO getUserByMobile(String mobile);

    /**
     * 根据邮箱查询用户信息
     * <p>
     * 主要用于登录
     *
     * @param email 邮箱
     * @return 用户信息
     */
    SysUserDTO getUserByEmail(String email);

    /**
     * 根据用户ID查询用户信息
     * <p>
     * 主要用于登录
     *
     * @param id 用户ID
     * @return 用户信息
     */
    SysUserDTO getUserById(Long id);

    /**
     * 根据账号查询用户信息
     * <p>
     * 主要用于登录
     *
     * @param account 账号
     * @return 用户信息
     */
    SysUserDTO getUserByAccount(String account);

    /**
     * 根据OpenId查询用户信息
     * <p>
     * 主要用于登录
     *
     * @param wechatOpenid 微信openid
     * @return 用户信息
     */
    SysUserDTO getUserByWechatOpenid(String wechatOpenid);

    /**
     * 根据OpenId查询用户信息
     * <p>
     * 主要用于登录
     *
     * @param appId  应用ID
     * @param openId 登录openid
     * @return 用户信息
     */
    SysUserDTO getUserByOpenId(String appId, String openId);
}
