package com.elitescloud.cloudt.system.service.devops.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.core.support.common.CacheableServiceClient;
import com.elitescloud.boot.core.support.common.param.CacheMethodInfo;
import com.elitescloud.boot.core.support.common.param.CacheServiceInfo;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.openfeign.common.DynamicClientHelper;
import com.elitescloud.boot.provider.PlatformAppProvider;
import com.elitescloud.boot.wrapper.Isolatable;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.convert.CacheServiceConvert;
import com.elitescloud.cloudt.system.model.vo.resp.devops.CacheMethodRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.devops.CacheServiceRespVO;
import com.elitescloud.cloudt.system.service.devops.CacheMngService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/9/12
 */
@Component
public class CacheMngServiceImpl implements CacheMngService {
    private static final Logger logger = LoggerFactory.getLogger(CacheMngServiceImpl.class);

    @Autowired
    private PlatformAppProvider appProvider;
    @Autowired
    private Isolatable isolatable;

    @Override
    public ApiResult<List<CacheServiceRespVO>> queryCacheServices(String serviceId) {
        var client = this.buildServiceClient(serviceId, null);

        // 获取缓存服务
        List<CacheServiceInfo> cacheServiceInfos = null;
        try {
            cacheServiceInfos = client.cacheServices(false).computeData();
        } catch (Exception e) {
            logger.error("获取缓存服务异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
        if (CollUtil.isEmpty(cacheServiceInfos)) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 转换格式
        var appList = appProvider.all();
        Map<String, String> appNameMap = CollUtil.isEmpty(appList) ? Collections.emptyMap() :
                appList.stream().collect(Collectors.toMap(CodeNameParam::getCode, CodeNameParam::getName, (t1, t2) -> t1));
        var respVoList = cacheServiceInfos.stream()
                .map(t -> {
                    var respVO = CacheServiceConvert.INSTANCE.convert2RespVO(t);
                    respVO.setAppName(appNameMap.get(t.getAppCode()));
                    return respVO;
                }).collect(Collectors.toList());
        return ApiResult.ok(respVoList);
    }

    @Override
    public ApiResult<List<CacheMethodRespVO>> queryCacheMethods(String serviceId, String cacheServiceKey) {
        var client = this.buildServiceClient(serviceId, null);

        // 获取缓存方法
        Assert.notBlank(cacheServiceKey, "缓存服务标识为空");
        List<CacheMethodInfo> methodInfos = null;
        try {
            methodInfos = client.cacheMethods(cacheServiceKey).computeData();
        } catch (Exception e) {
            logger.error("获取缓存方法异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
        if (CollUtil.isEmpty(methodInfos)) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 转换格式
        var respVoList = methodInfos.stream()
                .map(CacheServiceConvert.INSTANCE::convert2RespVO).collect(Collectors.toList());
        return ApiResult.ok(respVoList);
    }

    @Override
    public ApiResult<Object> queryCacheDataByCacheMethod(String serviceId, String cacheServiceKey, String method, Object[] params, String tenantCode) {
        var client = this.buildServiceClient(serviceId, null);

        Assert.notBlank(cacheServiceKey, "缓存服务标识为空");
        Assert.notBlank(method, "缓存方法为空");

        if (params == null) {
            params = new Object[0];
        }
        try {
            if (StringUtils.hasText(tenantCode)) {
                Object[] finalParams = params;
                return isolatable.apply(() -> client.queryCacheDataByCacheMethod(cacheServiceKey, method, finalParams), tenantCode);
            }
            return client.queryCacheDataByCacheMethod(cacheServiceKey, method, params);
        } catch (Exception e) {
            logger.error("查询缓存数据异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<List<String>> queryCacheItems(String serviceId, String cacheServiceKey, String tenantCode) {
        var client = this.buildServiceClient(serviceId, null);

        Assert.notBlank(cacheServiceKey, "缓存服务标识为空");
        try {
            if (StringUtils.hasText(tenantCode)) {
                return isolatable.apply(() -> client.cacheItems(cacheServiceKey), tenantCode);
            }
            return client.cacheItems(cacheServiceKey);
        } catch (Exception e) {
            logger.error("获取缓存项异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<Object> queryCacheItemData(String serviceId, String cacheServiceKey, String cacheItem, String tenantCode) {
        var client = this.buildServiceClient(serviceId, null);

        Assert.notBlank(cacheServiceKey, "缓存服务标识为空");
        Assert.notBlank(cacheItem, "缓存项为空");
        try {
            if (StringUtils.hasText(tenantCode)) {
                return isolatable.apply(() -> client.cacheItemValue(cacheServiceKey, cacheItem), tenantCode);
            }
            return client.cacheItemValue(cacheServiceKey, cacheItem);
        } catch (Exception e) {
            logger.error("获取缓存数据异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<Map<String, Object>> queryAllCacheItemData(@NotBlank String serviceId, @NotBlank String cacheServiceKey, String tenantCode) {
        var client = this.buildServiceClient(serviceId, null);

        Assert.notBlank(cacheServiceKey, "缓存服务标识为空");
        try {
            if (StringUtils.hasText(tenantCode)) {
                return isolatable.apply(() -> client.cacheItemValues(cacheServiceKey), tenantCode);
            }
            return client.cacheItemValues(cacheServiceKey);
        } catch (Exception e) {
            logger.error("获取缓存数据异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<Boolean> reload(String serviceId, String cacheServiceKey) {
        var client = this.buildServiceClient(serviceId, null);

        Assert.notBlank(cacheServiceKey, "缓存服务标识为空");
        try {
            return client.reload(cacheServiceKey);
        } catch (Exception e) {
            logger.error("获取缓存数据异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    @Override
    public ApiResult<Boolean> clear(String serviceId, String cacheServiceKey) {
        var client = this.buildServiceClient(serviceId, null);

        try {
            return client.clearAll(cacheServiceKey);
        } catch (Exception e) {
            logger.error("获取缓存数据异常：", e);
            return ApiResult.fail("获取失败，服务不支持或架构版本低");
        }
    }

    private CacheableServiceClient buildServiceClient(@NotBlank String serviceId, String url) {
        Assert.notBlank(serviceId, "服务ID为空");
        try {
            if (CharSequenceUtil.isBlank(url)) {
                return DynamicClientHelper.getClient(serviceId, CacheableServiceClient.class, CacheableServiceClient.URI);
            }
            return DynamicClientHelper.buildClient(serviceId, CacheableServiceClient.class, CacheableServiceClient.URI, url);
        } catch (Exception e) {
            throw new BusinessException("构建客户端失败", e);
        }
    }
}
