package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformMenusDO;
import com.elitescloud.cloudt.system.service.impl.SystemDataServiceImpl;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAppDO;
import com.elitescloud.cloudt.system.service.repo.AppRepoProc;
import com.elitescloud.cloudt.system.service.repo.MenuRepoProc;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 菜单与API接口绑定.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
@Slf4j
public class PlatformMenusApiInitProvider extends AbstractBasicDataInitProvider {

    private Map<String, SysPlatformAppDO> appMap;
    private Map<String, SysPlatformMenusDO> menuMap;

    @Override
    public String typeName() {
        return "菜单与API接口绑定";
    }

    @Override
    public String tableName() {
        return "sys_platform_menus_api";
    }

    @Override
    public List<String> fields() {
        return List.of("menus_id", "api_id", "app_code", "menus_code", "api_code", "std_version", "create_time", "modify_time");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("app_code", "应用编码");
        titles.put("menus_code", "菜单编码");
        titles.put("api_code", "接口编码");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");

        return titles;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.desc("create_time"));
    }

    @Override
    public String fieldAppCode() {
        return "app_code";
    }

    @Override
    public void prepareForImport(SystemDataServiceImpl.ImportedData importData, Long tenantId) {
        appMap = SpringContextHolder.getBean(AppRepoProc.class).all()
                .stream()
                .collect(Collectors.toMap(SysPlatformAppDO::getAppCode, t -> t, (t1, t2) -> t1));
        menuMap = SpringContextHolder.getBean(MenuRepoProc.class).all()
                .stream()
                .collect(Collectors.toMap(SysPlatformMenusDO::getMenusCode, t -> t, (t1, t2) -> t1));
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        if (CollUtil.isEmpty(lastStepData)) {
            return UpdateType.IGNORE;
        }

        // 查询API编码
        var apiCode = this.getStringValue(data, "api_code", null);
        if (CharSequenceUtil.isBlank(apiCode)) {
            var originalApiId = getLongValue(data, "api_id", null);
            Assert.notNull(originalApiId, "缺少api_id");
            var originalApiMap = lastStepData.parallelStream()
                    .filter(t -> Objects.equals(originalApiId, t.get("original_id")))
                    .findFirst()
                    .orElse(Collections.emptyMap());
            apiCode = getStringValue(originalApiMap, "api_code", null);
            if (CharSequenceUtil.isBlank(apiCode)) {
                log.info("未查询到API：{}", originalApiId);
                return UpdateType.IGNORE;
            }
        }

        var appCode = this.getStringValue(data, "app_code", null);
        Assert.notBlank(appCode, "应用编码为空");
        var menusCode = this.getStringValue(data, "menus_code", null);
        Assert.notBlank(menusCode, "菜单编码为空");

        if (currentData.isEmpty()) {
            return UpdateType.ADD;
        }

        String finalApiCode = apiCode;
        var id = currentData.stream()
                .filter(t -> appCode.equals(t.get("app_code")) && menusCode.equals(t.get("menus_code")) && finalApiCode.equals(t.get("api_code")))
                .findFirst()
                .map(t -> t.get("id"))
                .orElse(null);
        if (id != null) {
            return UpdateType.IGNORE;
        }

        // 获取api的id
        var apiId = lastStepData.parallelStream()
                .filter(t -> finalApiCode.equals(getStringValue(t, "api_code", ""))).findFirst()
                .map(t -> getLongValue(t, "id", null))
                .orElse(null);
        if (apiId == null) {
            return UpdateType.IGNORE;
        }

        if (CollUtil.isEmpty(appMap)) {
            log.info("应用为空，忽略");
            return UpdateType.IGNORE;
        }

        if (CollUtil.isEmpty(menuMap)) {
            log.info("菜单为空，忽略");
            return UpdateType.IGNORE;
        }
        var menu = menuMap.get(menusCode);
        if (menu == null) {
            log.info("菜单不存在，忽略");
            return UpdateType.IGNORE;
        }

        data.put("id", null);
        data.put("api_id", apiId);
        data.put("menus_id", menu.getId());

        return UpdateType.ADD;
    }
}
