package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.DataRelationConvert;
import com.elitescloud.cloudt.system.model.vo.query.datarelation.DataRelationInstEditQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.datarelation.DataRelationConfigRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.datarelation.DataRelationInstEditRespVO;
import com.elitescloud.cloudt.system.model.vo.save.datarelation.DataRelationInstSaveVO;
import com.elitescloud.cloudt.system.service.DataRelationInstMngService;
import com.elitescloud.cloudt.system.service.manager.DataRelationManager;
import com.elitescloud.cloudt.system.service.manager.DataSelectorManager;
import com.elitescloud.cloudt.system.service.model.entity.SysDataRelationInstanceDO;
import com.elitescloud.cloudt.system.service.repo.DataRelationCategoryRepoProc;
import com.elitescloud.cloudt.system.service.repo.DataRelationInstRepoProc;
import com.elitescloud.cloudt.system.service.repo.DataRelationRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/15
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class DataRelationInstMngServiceImpl implements DataRelationInstMngService {
    @Autowired
    private DataRelationRepoProc repoProc;
    @Autowired
    private DataRelationCategoryRepoProc categoryRepoProc;
    @Autowired
    private DataRelationInstRepoProc instRepoProc;

    @Autowired
    private DataRelationManager manager;
    @Autowired
    private DataSelectorManager dataSelectorManager;
    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<DataRelationConfigRespVO> getConfig(String drCode) {
        Assert.notBlank(drCode, "数据关系编码为空");
        var drDO = repoProc.getByCode(drCode);
        if (drDO == null || Boolean.FALSE.equals(drDO.getEnabled())) {
            return ApiResult.fail("数据关系配置不存在或未启用");
        }

        // 转为VO
        var respVO = DataRelationConvert.INSTANCE.do2ConfigRespVO(drDO);
        // 获取分类
        var catList = categoryRepoProc.listByDrCode(drDO.getCode()).stream()
                .filter(t -> Boolean.TRUE.equals(t.getEnabled()))
                .map(DataRelationConvert.INSTANCE::do2ConfigRespVO)
                .collect(Collectors.toList());
        respVO.setCats(catList);
        // 前端配置
        if (StringUtils.hasText(drDO.getDataSelectorCode())) {
            respVO.setDataSelector(dataSelectorManager.getRespVO(drDO.getDataSelectorCode()));
        }

        return ApiResult.ok(respVO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveValues(DataRelationInstSaveVO saveVO) {
        try {
            this.checkForSave(saveVO);
        } catch (Exception e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        List<SysDataRelationInstanceDO> instDoList = CollUtil.isEmpty(saveVO.getCatList()) ? Collections.emptyList() :
                saveVO.getCatList().stream()
                        .flatMap(cat -> {
                            if (CollUtil.isEmpty(cat.getDataList())) {
                                return Stream.empty();
                            }

                            String dataExtJson = CollUtil.isEmpty(cat.getDataExt()) ? null : JSONUtil.toJsonString(cat.getDataExt());
                            return cat.getDataList().stream()
                                    .flatMap(cateData -> {
                                        String refDataExtJson = CollUtil.isEmpty(cateData.getRefDataExt()) ? null : JSONUtil.toJsonString(cateData.getRefDataExt());
                                        AtomicInteger sortNo = new AtomicInteger(0);

                                        return saveVO.getDataKeyList().stream()
                                                .map(dataKey -> {
                                                    SysDataRelationInstanceDO instDO = new SysDataRelationInstanceDO();
                                                    instDO.setDrCode(saveVO.getDrCode());
                                                    instDO.setCatCode(cat.getCatCode());
                                                    instDO.setDataKey(dataKey);
                                                    instDO.setDataAttribute(ObjectUtil.defaultIfNull(cat.getDataAttribute(), ""));
                                                    instDO.setDataExtJson(dataExtJson);

                                                    instDO.setRefDataKey(cateData.getRefDataKey());
                                                    instDO.setRefDataAttribute(ObjectUtil.defaultIfNull(cateData.getRefDataAttribute(), ""));
                                                    instDO.setRefDataExtJson(refDataExtJson);

                                                    instDO.setSortNo(sortNo.getAndIncrement());
                                                    return instDO;
                                                });
                                    });
                        }).collect(Collectors.toList());

        // 先删除之前的
        instRepoProc.deleteForInst(saveVO.getDrCode(), saveVO.getDataKeyList());
        // 批量保存
        if (!instDoList.isEmpty()) {
            instRepoProc.save(instDoList);
        }

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<DataRelationInstEditRespVO> getValues(DataRelationInstEditQueryVO queryVO) {
        Assert.notNull(queryVO, "查询参数为空");
        Assert.notBlank(queryVO.getDataKey(), "数据标识为空");
        Assert.notBlank(queryVO.getDrCode(), "数据关系编码为空");

        // 查询关联的数据
        var queryBO = DataRelationConvert.INSTANCE.queryVo2Bo(queryVO);
        var valueBoList = manager.queryValues(queryBO);

        // 数据格式转换
        var respVO = new DataRelationInstEditRespVO();
        respVO.setDataKey(queryVO.getDataKey());
        respVO.setDrCode(queryVO.getDrCode());
        respVO.setCatList(Collections.emptyList());
        if (valueBoList.isEmpty()) {
            return ApiResult.ok(respVO);
        }

        Map<String, DataRelationInstEditRespVO.CatData> catDataMap = new LinkedHashMap<>();
        DataRelationInstEditRespVO.InstData instData = null;
        for (var valueBO : valueBoList) {
            var dataList = catDataMap.computeIfAbsent(valueBO.getCatCode() + "#::#" + ObjUtil.defaultIfNull(valueBO.getDataAttribute(), "__undefined"), catCode -> {
                DataRelationInstEditRespVO.CatData catData = new DataRelationInstEditRespVO.CatData();
                catData.setCatCode(valueBO.getCatCode());
                catData.setDataAttribute(valueBO.getDataAttribute());
                catData.setDataExt(valueBO.getDataExt());
                catData.setDataList(new ArrayList<>(valueBoList.size()));

                return catData;
            });

            instData = new DataRelationInstEditRespVO.InstData();
            instData.setRefDataKey(valueBO.getRefDataKey());
            instData.setRefDataAttribute(valueBO.getRefDataAttribute());
            instData.setRefDataExt(valueBO.getRefDataExt());

            dataList.getDataList().add(instData);
        }

        respVO.setCatList(new ArrayList<>(catDataMap.values()));
        return ApiResult.ok(respVO);
    }

    private void checkForSave(DataRelationInstSaveVO saveVO) {
        List<String> dataKeyList = CollUtil.isEmpty(saveVO.getDataKeyList()) ? Collections.emptyList() :
                saveVO.getDataKeyList().stream().filter(StringUtils::hasText).distinct().collect(Collectors.toList());
        Assert.notEmpty(dataKeyList, "业务数据标识为空");
        Assert.notBlank(saveVO.getDrCode(), "数据关系编码为空");

        saveVO.setDataKeyList(dataKeyList);

        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            boolean exists = repoProc.existsCode(saveVO.getDrCode(), null);
            if (!exists) {
                throw new BusinessException("数据关系不存在");
            }

            var catCodes = categoryRepoProc.getCodesByDrCode(saveVO.getDrCode());
            if (CollUtil.isNotEmpty(saveVO.getCatList())) {
                for (var catData : saveVO.getCatList()) {
                    Assert.notBlank(catData.getCatCode(), "分类编码不能为空");
                    Assert.isTrue(catCodes.contains(catData.getCatCode()), "分类" + catData.getCatCode() + "不存在");

                    if (CollUtil.isNotEmpty(catData.getDataList())) {
                        for (var instData : catData.getDataList()) {
                            Assert.notBlank(instData.getRefDataKey(), "关联数据的标识为空");
                        }
                    }
                }
            }

            return null;
        });
    }
}
