package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.spring.SpringUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.model.vo.resp.extend.FormColumnDefinition;
import com.elitescloud.cloudt.system.model.vo.resp.extend.HttpParam;
import com.elitescloud.cloudt.system.model.vo.resp.extend.InfinityLogVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.InfinityResult;
import com.elitescloud.cloudt.system.service.InfinityApiService;
import com.elitescloud.cloudt.system.service.SecurityAuthService;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.MediaType;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @Auther: Mark
 * @Date: 2024/12/16 14:00
 * @Description:
 */

@Slf4j
public class InfinityRestClient {
    
    public InfinityRestClient(WebClient webClient) {
        this.webClient = webClient;
    }

    private WebClient webClient;
    @Lazy
    @Resource
    private InfinityApiService infinityApiService;
    @Lazy
    @Resource
    private SecurityAuthService securityAuthService;

    public static Map<String, SecurityAuthService> consumerMap = new HashMap<>();

    @PostConstruct
    public void initHandlerMap() {
        Map<String, SecurityAuthService> consumers = SpringUtil.getBeansOfType(SecurityAuthService.class);
        consumers.forEach((k, v) -> {
            consumerMap.put(v.getAuthType(), v);
        });
        log.info("认证方式处理器:{}", JSONUtil.toJsonStr(consumerMap));
    }

    public InfinityResult<String> exchange(String apiCode) {
        return exchange(apiCode, String.class, null);
    }

    /**
     * 无参数，返回指定类型
     */
    public <R> InfinityResult<R> exchange(String apiCode, Class<R> responseType) {
        return exchange(apiCode, responseType, null);
    }

    /**
     * 自定义参数，返回字符传
     */
    public InfinityResult<String> exchange(String apiCode, Function<HttpParam.HttpParamBuilder, HttpParam> func) {
        return exchange(apiCode, String.class, func);
    }

    /**
     * 自定义参数，返回指定类型
     *
     * @since 0.3.1-SNAPSHOT 重置日志的报文结构
     */
    @SneakyThrows
    public <R> InfinityResult<R> exchange(String apiCode, Class<R> responseType, Function<HttpParam.HttpParamBuilder,
            HttpParam> fn) {
        InfinityLogVO logVO = new InfinityLogVO();
        var builder = HttpParam.builder().apiConfig(infinityApiService.infinityApiByApiCode(apiCode));
        HttpParam param = fn != null ? fn.apply(builder) : builder.build();
        // 1. 接口地址
        UriComponentsBuilder uriBuilder = UriComponentsBuilder.fromHttpUrl(param.getUrl());
        logVO.setServerUrl(param.getApiConfig().getServerUrl());
        logVO.setApiUrl(param.getApiConfig().getApiUrl());
        logVO.setUri(param.getUrl());
        // ====>>>>> TODO 配置项：是否强制按照接口规范，进行参数转换
        // 2.1 queryParam固定参数
        List<FormColumnDefinition> queryConfig = param.getApiConfig().getQueryParam();
        if (CollUtil.isNotEmpty(queryConfig)) {
            queryConfig.stream()
                    .filter(def -> StrUtil.isNotBlank(def.getFixedValue()))
                    .forEach(def -> {
                        uriBuilder.queryParam(def.getColumnKey(), def.getFixedValue());
                        logVO.addQueryParam(def.getColumnKey(), def.getFixedValue());
                    });
        }
        // 2.2 queryParam动态参数
        if (MapUtil.isNotEmpty(param.getQueryParam())) {
            uriBuilder.queryParams(param.getQueryParam());
            logVO.addQueryParams(param.getQueryParam());
        }
        // pathParam参数固定值逻辑
        Map<String, Object> pathVariables = extractPathVariables(param.getUrl());
        List<FormColumnDefinition> pathParam = param.getApiConfig().getPathParam();
        if (CollUtil.isNotEmpty(pathParam)) {
            pathParam.stream()
                    .filter(def -> StrUtil.isNotBlank(def.getFixedValue()))
                    .forEach(
                            def -> {
                                if (pathVariables.containsKey(def.getColumnKey())) {
                                    pathVariables.put(def.getColumnKey(), def.getFixedValue());
                                    logVO.addQueryParam(def.getColumnKey(), def.getFixedValue());
                                }
                            }
                    );
        }
        //pathParam动态参数
        /*if (MapUtil.isNotEmpty(param.getPathParam())) {
            param.getPathParam().forEach((key, value) -> {
                if (pathVariables.containsKey(key)) {
                    pathVariables.put(key, value);
                }
                logVO.addQueryParam(key, value);
            });
        }*/
        String finalUri = uriBuilder.toUriString();
        if (MapUtil.isEmpty(param.getPathParam())) {
            finalUri = uriBuilder.buildAndExpand(pathVariables).toUriString();
        }
        // 3. 请求方式、pathParam
        WebClient.RequestBodyUriSpec method = webClient.method(param.getRequestMethod());
        WebClient.RequestBodySpec uri = MapUtil.isEmpty(param.getPathParam()) ?
                method.uri(new URI(finalUri))
                // 带路径变量的，不要做`encode`；会把`{}`进行URL编码，导致变量替换失败；
                : method.uri(uriBuilder.build(false).toUriString(), param.getPathParam());
        uri.acceptCharset(StandardCharsets.UTF_8);
        logVO.setRequestMethod(param.getRequestMethod());

        // 4. 认证方式(authMethod)
        SecurityAuthService securityAuthService = consumerMap.get(param.getAuthMethod());
        if (securityAuthService == null) {
            throw new BusinessException("未找到认证方式绑定的认证处理器");
        }
        log.info("命中的认证方式处理器编码:{}", securityAuthService.getAuthType());
        securityAuthService.authentication(uri, param);
        //UnicomClient.supply(() -> securityAuthService.authentication(uri, param), param.getAuthMethod());

        // 5. headerParam
        uri.headers(headers -> {
            // 5.1 固定参数
            List<FormColumnDefinition> headerConfig = param.getApiConfig().getHeaderParam();
            if (CollUtil.isNotEmpty(headerConfig)) {
                headerConfig.stream()
                        .filter(def -> StrUtil.isNotBlank(def.getFixedValue()))
                        .forEach(def -> headers.set(def.getColumnKey(), def.getFixedValue()));
            }
            // 5.2 动态参数
            MultiValueMap<String, String> headerParam = param.getHeaderParam();
            if (MapUtil.isNotEmpty(headerParam)) {
                for (String key : headerParam.keySet()) {
                    headers.set(key, headerParam.getFirst(key));
                }
            }
            logVO.setHeaderParam(headers);
        });
        // 6. 根据`mediaType`, 读取`bodyParam`进行提交
        if (param.needBodyParam()) {
            uri.contentType(MediaType.parseMediaType(param.getMediaType()));
            if (param.formData()) {
                uri.body(param.getBodyFormData());
                logVO.setBodyParam(param.getBodyFormData());
            } else if (param.objectData()) {
                uri.bodyValue(param.getBodyParam());
                logVO.setBodyParam(param.getBodyParam());
            }
        }
        return InfinityResult.of(responseType, param, logVO, uri);
    }

    // 提取占位符参数的方法
    private static Map<String, Object> extractPathVariables(String url) {
        Map<String, Object> variables = new HashMap<>();
        // 正则表达式匹配占位符
        Pattern pattern = Pattern.compile("\\{(\\w+)}");
        Matcher matcher = pattern.matcher(url);

        while (matcher.find()) {
            // 获取占位符名称
            String variableName = matcher.group(1);
            // 将占位符名称添加到 Map 中，初始值为 null
            variables.put(variableName, null);
        }
        return variables;
    }
}
