package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.constant.MimeTypeConstant;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.excel.common.param.ExportColumnParam;
import com.elitescloud.boot.excel.util.ExcelExportUtil;
import com.elitescloud.boot.excel.util.ExcelImportUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.PlatformAppProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.cacheable.SysCacheMenuRpcService;
import com.elitescloud.cloudt.system.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.system.constant.PlatformMenusNodeEnum;
import com.elitescloud.cloudt.system.convert.TenantMenuTreeConvert;
import com.elitescloud.cloudt.system.model.bo.MenuBO;
import com.elitescloud.cloudt.system.model.bo.TenantMenuBO;
import com.elitescloud.cloudt.system.model.vo.resp.menu.*;
import com.elitescloud.cloudt.system.model.vo.save.menu.CustomMenuBoundSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.menu.CustomMenuGroupSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.menu.MenuTreeSaveVO;
import com.elitescloud.cloudt.system.provider.dto.save.SysMenuSaveDTO;
import com.elitescloud.cloudt.system.service.MenuMngService;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.manager.PermissionQueryManager;
import com.elitescloud.cloudt.system.service.model.bo.AppBO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantMenuDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantMenuTreeDO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.google.common.base.Functions;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ContentDisposition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/13
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class MenuMngServiceImpl extends BaseServiceImpl implements MenuMngService {
    private static final TenantMenuTreeConvert CONVERT = TenantMenuTreeConvert.INSTANCE;

    @Autowired
    private TenantMenuRepo tenantMenuRepo;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private TenantMenuTreeRepo tenantMenuTreeRepo;
    @Autowired
    private TenantMenuTreeRepoProc tenantMenuTreeRepoProc;
    @Autowired
    private MenuRepoProc menusRepoProc;
    @Autowired
    private TenantAppRepoProc tenantAppRepoProc;

    @Autowired
    private PermissionQueryManager permissionQueryManager;

    @Autowired
    private PlatformAppProvider appProvider;

    @Autowired
    private SysCacheMenuRpcService cacheMenuRpcService;

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<Long> addMenu(SysMenuSaveDTO saveDTO) {
        var menuDO = CONVERT.dto2do(saveDTO);
        Assert.hasText(menuDO.getMenusCode(), "菜单编码为空");
        var exists = menusRepoProc.existsMenuCode(saveDTO.getMenusCode());
        Assert.isTrue(!exists, "菜单编码已存在");

        if (StringUtils.hasText(saveDTO.getMenusParentCode())) {
            exists = menusRepoProc.existsMenuCode(saveDTO.getMenusParentCode());
            Assert.isTrue(exists, "上级菜单不存在");
            menuDO.setMenusAppCode(menusRepoProc.getAppCodeByMenuCode(saveDTO.getMenusParentCode()));
        } else {
            Assert.hasText(menuDO.getMenusAppCode(), "未知菜单所属应用");
        }

        menuDO.setMenusType(ObjUtil.defaultIfNull(saveDTO.getMenusTypeEnum(), PlatformAppMenusTypeEnum.MENUS_TYPE_BUS).name());
        menuDO.setNodeType(ObjUtil.defaultIfNull(saveDTO.getNodeTypeEnum(), PlatformMenusNodeEnum.MENUS).name());
        menuDO.setMenusOrder(0);
        menuDO.setMenusState(true);
        menuDO.setDisplay(true);
        menuDO.setOuterLink(ObjUtil.defaultIfNull(saveDTO.getOuterLink(), false));
        menuDO.setOuterLinkType(saveDTO.getOuterLinkTypeEnum());
        if (menuDO.getOuterLink()) {
            Assert.notNull(menuDO.getOuterLinkType(), "未知外部链接类型");
            Assert.hasText(menuDO.getMenusRoute(), "外部链接时路由不能为空");
        }

        menusRepoProc.save(menuDO);

        // 清空缓存
        clearCache();

        return ApiResult.ok(menuDO.getId());
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<String> removeMenu(String menuCode) {
        Assert.hasText(menuCode, "菜单编码为空");

        menusRepoProc.deleteByMenuCode(menuCode);

        // 清空缓存
        clearCache();
        return ApiResult.ok(menuCode);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveCustomMenuGroup(CustomMenuGroupSaveVO saveVO) {
        SysTenantMenuTreeDO menuTreeDO = null;
        try {
            menuTreeDO = this.convert(saveVO);
        } catch (Exception e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        tenantMenuTreeRepoProc.save(menuTreeDO);

        // 清空缓存
        clearCache();
        return ApiResult.ok(menuTreeDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateCustomEnabled(Long id, Boolean enabled) {
        Assert.notNull(id, "ID为空");
        Assert.notNull(enabled, "启用状态为空");

        tenantMenuTreeRepoProc.updateEnabled(id, enabled);

        // 清空缓存
        clearCache();
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> removeAllCustom() {
        tenantMenuTreeRepoProc.deleteAll();

        // 清空缓存
        clearCache();
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> removeCustom(List<Long> ids, Boolean cascade) {
        // 判断是否存在下级
        Assert.notEmpty(ids, "ID为空");

        // 清空缓存
        clearCache();

        if (cascade) {
            // 级联删除
            this.deleteCascade(ids);
            return ApiResult.ok(ids);
        }

        // 判断是否存在下级
        var existsNames = tenantMenuTreeRepoProc.existsChildren(ids);
        if (existsNames.isEmpty()) {
            tenantMenuTreeRepoProc.delete(ids);
            return ApiResult.ok(ids);
        }

        return ApiResult.fail("[" + String.join(",", existsNames) + "]存在下级节点，请先删除下级");
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> removeCustomChildren(Long id, Boolean withSelf) {
        Assert.notNull(id, "ID为空");

        if (Boolean.TRUE.equals(withSelf)) {
            this.deleteCascade(List.of(id));
        } else {
            var childrenIds = tenantMenuTreeRepoProc.listChildrenIds(Set.of(id));
            this.deleteCascade(childrenIds);
        }

        // 清空缓存
        clearCache();

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> moveCustom(Long id, Long newParentId, Integer sortNo) {
        Assert.notNull(id, "ID为空");

        // 菜单的话必须在分组下
        var nodeType = tenantMenuTreeRepoProc.getNodeType(id);
        if (!StringUtils.hasText(nodeType)) {
            return ApiResult.fail("修改的数据不存在");
        }
        if (MenuTreeNodeType.MENU.getValue().equals(nodeType) && newParentId == null) {
            return ApiResult.fail("菜单必须在分组下");
        }

        // 不能移动到菜单下面
        if (newParentId != null) {
            var parentNodeType = tenantMenuTreeRepoProc.getNodeType(newParentId);
            if (!StringUtils.hasText(parentNodeType)) {
                return ApiResult.fail("分组数据不存在");
            }
            if (MenuTreeNodeType.MENU.getValue().equals(parentNodeType)) {
                return ApiResult.fail("不能移动到菜单下面");
            }
        }

        // 更新上级
        String parentCode = newParentId == null ? null : tenantMenuTreeRepoProc.getMenuCode(newParentId);
        tenantMenuTreeRepoProc.updateParent(id, parentCode, ObjUtil.defaultIfNull(sortNo, 0));

        // 清空缓存
        clearCache();

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<CustomMenuEditRespVO> getCustomEdit(Long id) {
        Assert.notNull(id, "ID为空");

        var menuGroupDO = tenantMenuTreeRepoProc.get(id);
        if (menuGroupDO == null) {
            return ApiResult.fail("数据不存在");
        }

        var respVO = TenantMenuTreeConvert.INSTANCE.do2EditRespVO(menuGroupDO);
        if (StringUtils.hasText(respVO.getParentMenuCode())) {
            respVO.setParentMenuName(tenantMenuTreeRepoProc.getMenuNameByMenuCode(respVO.getParentMenuCode()));
        }

        // 自定义名称
        if (MenuTreeNodeType.MENU.getValue().equals(menuGroupDO.getNodeType())) {
            respVO.setCustomName(menuGroupDO.getMenuName());
            tenantDataIsolateProvider.byDefaultDirectly(() -> {
                respVO.setMenuName(menusRepoProc.getMenuNameByMenuCode(menuGroupDO.getMenuCode()));
                return null;
            });
            if (CharSequenceUtil.isBlank(respVO.getMenuName())) {
                // 菜单不存在了
                return ApiResult.fail("菜单已不存在");
            }
        }
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<CustomMenuNodeDetailRespVO> getCustomNodeDetail(Long id) {
        Assert.notNull(id, "ID为空");

        var menuGroupDO = tenantMenuTreeRepoProc.get(id);
        if (menuGroupDO == null) {
            return ApiResult.fail("数据不存在");
        }

        var respVO = TenantMenuTreeConvert.INSTANCE.do2NodeDetailRespVO(menuGroupDO);
        if (StringUtils.hasText(respVO.getParentMenuCode())) {
            respVO.setParentMenuName(tenantMenuTreeRepoProc.getMenuNameByMenuCode(respVO.getParentMenuCode()));
        }

        // 自定义名称和下级
        if (MenuTreeNodeType.MENU.getValue().equals(menuGroupDO.getNodeType())) {
            respVO.setCustomName(menuGroupDO.getMenuName());
            tenantDataIsolateProvider.byDefaultDirectly(() -> {
                respVO.setMenuName(menusRepoProc.getMenuNameByMenuCode(menuGroupDO.getMenuCode()));
                return null;
            });
            if (CharSequenceUtil.isBlank(respVO.getMenuName())) {
                // 菜单不存在了
                return ApiResult.fail("菜单已不存在");
            }
            respVO.setChildren(Collections.emptyList());
            respVO.setChildrenNum(0L);
        } else {
            // 查询下级节点信息
            var children = tenantMenuTreeRepoProc.queryByParentMenuCode(menuGroupDO.getMenuCode());
            // 获取原名称
            Set<String> menuCodes = children.isEmpty() ? Collections.emptySet() : children.stream()
                    .filter(t -> MenuTreeNodeType.MENU.getValue().equals(t.getNodeType()))
                    .map(CustomMenuNodeDetailRespVO::getMenuCode)
                    .collect(Collectors.toSet());
            if (!menuCodes.isEmpty()) {
                Map<String, String> menuCodeNameMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuNameByMenuCode(menuCodes));
                children = children.stream()
                        .peek(t -> {
                            t.setCustomName(t.getMenuName());
                            t.setMenuName(menuCodeNameMap.getOrDefault(t.getMenuCode(), t.getMenuName()));
                            t.setParentMenuCode(respVO.getMenuCode());
                            t.setParentMenuName(respVO.getMenuName());
                        }).collect(Collectors.toList());
            } else {
                children = children.stream()
                        .peek(t -> {
                            t.setParentMenuCode(respVO.getMenuCode());
                            t.setParentMenuName(respVO.getMenuName());
                        }).collect(Collectors.toList());
            }

            respVO.setChildren(children);
            respVO.setChildrenNum((long) children.size());
        }
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> listGroup() {
        var dataList = tenantMenuTreeRepoProc.listGroup();
        return ApiResult.ok(dataList);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> saveBoundMenus(CustomMenuBoundSaveVO saveVO) {
        Assert.hasText(saveVO.getGroupCode(), "分组编码为空");
        var menuGroup = tenantMenuTreeRepoProc.getByMenuCode(saveVO.getGroupCode());
        if (menuGroup == null) {
            return ApiResult.fail("分组不存在");
        }

        boolean all = ObjectUtil.defaultIfNull(saveVO.getAll(), true);
        List<SysTenantMenuTreeDO> menuTreeDoList = null;
        if (CollUtil.isEmpty(saveVO.getBoundMenus())) {
            menuTreeDoList = Collections.emptyList();
        } else {
            Set<String> existsCodes = new HashSet<>(32);
            if (!all) {
                // 新增时，忽略已存在的
                var boundCodes = tenantMenuTreeRepoProc.getBoundMenuCodes(menuGroup.getMenuCode());
                existsCodes.addAll(boundCodes);
            }

            menuTreeDoList = new ArrayList<>(saveVO.getBoundMenus().size());
            int i = 0;
            for (CustomMenuBoundSaveVO.Menu menu : saveVO.getBoundMenus()) {
                if (existsCodes.contains(menu.getMenuCode())) {
                    continue;
                }
                existsCodes.add(menu.getMenuCode());
                SysTenantMenuTreeDO menuTreeDO = new SysTenantMenuTreeDO();
                menuTreeDO.setMenuCode(menu.getMenuCode());
                menuTreeDO.setMenuName(menu.getCustomName());
                menuTreeDO.setNodeType(MenuTreeNodeType.MENU.getValue());
                menuTreeDO.setParentMenuCode(menuGroup.getMenuCode());
                menuTreeDO.setCustom(true);
                menuTreeDO.setSortNo(i++);
                menuTreeDO.setEnabled(true);

                menuTreeDoList.add(menuTreeDO);
            }
        }

        if (all) {
            tenantMenuTreeRepoProc.deleteBoundMenuCodes(menuGroup.getMenuCode());
        }
        if (!menuTreeDoList.isEmpty()) {
            tenantMenuTreeRepoProc.save(menuTreeDoList);
        }

        return ApiResult.ok(saveVO.getGroupCode());
    }

    @Override
    public ApiResult<List<CustomMenuBoundRespVO>> getBoundMenus(String groupCode) {
        var customMenuList = tenantMenuTreeRepoProc.getBoundMenu(groupCode);
        if (customMenuList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        var menuCodes = customMenuList.stream().map(CodeNameParam::getCode).filter(StringUtils::hasText).collect(Collectors.toSet());
        var menuMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuCodeAndNames(menuCodes))
                .stream().collect(Collectors.toMap(IdCodeNameParam::getCode, Functions.identity(), (t1, t2) -> t1));
        var respVoList = customMenuList.stream()
                .filter(t -> menuMap.containsKey(t.getCode()))
                .map(t -> {
                    var menu = menuMap.get(t.getCode());
                    CustomMenuBoundRespVO respVO = new CustomMenuBoundRespVO();
                    respVO.setId(menu.getId());
                    respVO.setCode(menu.getCode());
                    respVO.setName(menu.getName());
                    respVO.setCustomName(t.getName());

                    return respVO;
                }).collect(Collectors.toList());
        return ApiResult.ok(respVoList);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveTree(List<MenuTreeSaveVO> saveVOList) {
        // 组织菜单树
        var tenantId = currentTenantId();
        var menuTreeDoList = convertAndCheck(tenantId, saveVOList);

        // 先清除原有数据
        tenantMenuTreeRepoProc.deleteByTenantId();

        // 再保存现有数据
        tenantMenuTreeRepo.saveAll(menuTreeDoList);

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateEnabledCustom() {
        // 获取现有配置
        var tenantId = currentTenantId();
        var enabledCurrent = tenantMenuRepoProc.getEnabledByTenant(tenantId);

        if (enabledCurrent == null) {
            // 如果没有，则新建
            var tenantMenuDO = new SysTenantMenuDO();
            tenantMenuDO.setSysTenantId(tenantId);
            tenantMenuDO.setEnabled(true);
            tenantMenuRepo.save(tenantMenuDO);
            return ApiResult.ok(true);
        }

        // 有则更新
        boolean enabled = !enabledCurrent;
        tenantMenuRepoProc.updateEnabledByTenant(tenantId, enabled);
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateCustomName(Long id, String customName) {
        Assert.notNull(id, "菜单ID为空");

        tenantMenuTreeRepoProc.updateMenuName(id, customName);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<List<CustomMenuTreeRespVO>> getCustomMenuTree(Boolean tree) {
        // 查询数据列表
        List<CustomMenuTreeRespVO> respVoList = this.queryCustomMenuTree();
        if (respVoList.isEmpty()) {
            return ApiResult.ok(respVoList);
        }

        // 转为树形
        TreeDataUtil<CustomMenuTreeRespVO> treeDataUtil = new TreeDataUtil<>(respVoList,
                CustomMenuTreeRespVO::getId, CustomMenuTreeRespVO::getParentId, CustomMenuTreeRespVO::setChildren,
                Comparator.comparingInt(CustomMenuTreeRespVO::getSortNo));
        respVoList = ((List<CustomMenuTreeRespVO>) treeDataUtil.getRoots()).stream()
                .filter(t -> MenuTreeNodeType.MENU_GROUP.getValue().equals(t.getNodeType()))
                .collect(Collectors.toList());
        // 设置深度
        this.setDepth(respVoList, CustomMenuTreeRespVO::getChildren, 0, CustomMenuTreeRespVO::setDepth);
        if (tree == null || tree) {
            return ApiResult.ok(respVoList);
        }

        respVoList = CollectionUtil.expandTree(respVoList, CustomMenuTreeRespVO::getChildren);
        return ApiResult.ok(respVoList);
    }

    @Override
    public ApiResult<List<MenuTreeRespVO>> getTreeDefault(Boolean tree) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        // 先获取应用
        Set<String> appCodes = currentUser.getTenant() == null ? null :
                tenantDataIsolateProvider.byDefaultDirectly(() -> tenantAppRepoProc.getAppCode(currentUser.getTenantId()));
        if (!currentUser.isOperation() && CollectionUtils.isEmpty(appCodes)) {
            // 租户尚未分配应用
            return ApiResult.ok(Collections.emptyList());
        }
        List<MenuTreeRespVO> respVOList = convertMenuTreeRespVO(appCodes);
        if (respVOList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 获取菜单
        List<MenuTreeRespVO> finalRespVOList = respVOList;
        tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.queryMenu(appCodes, true, true, true, true)).stream()
                .filter(t -> {
                    // 过滤业务菜单
                    if (!PlatformAppMenusTypeEnum.MENUS_TYPE_BUS.name().equals(t.getMenusType())) {
                        return false;
                    }

                    return true;
                })
                .forEach(t -> {
                    MenuTreeRespVO respVO = new MenuTreeRespVO();
                    respVO.setMenuCode(t.getMenusCode());
                    respVO.setMenuName(t.getMenusName());
                    respVO.setMenusType(t.getMenusType());
                    respVO.setNodeType(MenuTreeNodeType.MENU.getValue());
                    respVO.setParentMenuCode(t.getMenusParentCode());
                    if (CharSequenceUtil.isBlank(t.getMenusParentCode())) {
                        // 顶级的设置应用编码
                        respVO.setParentMenuCode(t.getMenusAppCode());
                    }
                    respVO.setMenusIcon(t.getMenusIcon());
                    respVO.setHidden(t.getDisplay() != null && !t.getDisplay());
                    respVO.setEnabled(t.getMenusState() == null || t.getMenusState());
                    respVO.setSortNo(ObjectUtil.defaultIfNull(t.getMenusOrder(), 1));
                    respVO.setHasChildren(false);
                    respVO.setChildren(new ArrayList<>(128));

                    finalRespVOList.add(respVO);
                });

        // 转树形
        if (BooleanUtil.isFalse(tree)) {
            return ApiResult.ok(respVOList);
        }
        TreeDataUtil<MenuTreeRespVO> treeDataUtil = new TreeDataUtil<>(respVOList, MenuTreeRespVO::getMenuCode,
                MenuTreeRespVO::getParentMenuCode, MenuTreeRespVO::setChildren, Comparator.comparingInt(MenuTreeRespVO::getSortNo));
        respVOList = ((List<MenuTreeRespVO>) treeDataUtil.getRoots()).stream()
                .filter(t -> MenuTreeNodeType.APP.getValue().equals(t.getNodeType()))
                .collect(Collectors.toList());

        return ApiResult.ok(respVOList);
    }

    @Override
    public ApiResult<List<MenuTreeCustomRespVO>> getTreeCustom(Boolean tree) {
        var currentTenant = tenantClientProvider.getCurrentTenant();
        // 查询租户的菜单
        var menuTreeDOList = tenantMenuTreeRepoProc.queryByTenantId();
        if (menuTreeDOList.isEmpty()) {
            // 尚未设置
            return ApiResult.ok(Collections.emptyList());
        }

        // 查询默认菜单
        var menuDOMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.queryMenu(null, true, true, true, true)).stream()
                .filter(t -> {
                    // 过滤业务菜单
                    if (!PlatformAppMenusTypeEnum.MENUS_TYPE_BUS.name().equals(t.getMenusType())) {
                        return false;
                    }

                    return true;
                }).collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));
        var allApps = appProvider.all().stream().collect(Collectors.toMap(CodeNameParam::getCode, t -> t, (t1, t2) -> t1));

        // 组织vo
        List<MenuTreeCustomRespVO> menuVOList = new ArrayList<>(menuTreeDOList.size());
        MenuTreeCustomRespVO menuVO = null;
        for (SysTenantMenuTreeDO menuTreeDO : menuTreeDOList) {
            menuVO = new MenuTreeCustomRespVO();
            menuVO.setMenuCode(menuTreeDO.getMenuCode());
            menuVO.setCustom(menuTreeDO.getCustom());
            menuVO.setMenuName(menuTreeDO.getMenuName());

            // 设置节点原始信息
            var nodeType = MenuTreeNodeType.valueOf(menuTreeDO.getNodeType());
            if (nodeType == null) {
                log.error("未知菜单节点类型：" + menuTreeDO.getNodeType());
                continue;
            }
            if (MenuTreeNodeType.MENU == nodeType) {
                MenuBO menuBO = menuDOMap.get(menuTreeDO.getMenuCode());
                if (menuBO == null) {
                    // 是否是自定义菜单
                    if (!menuTreeDO.getCustom()) {
                        log.info("菜单不存在：" + menuTreeDO.getMenuCode());
                        continue;
                    }
                    menuVO.setHidden(false);
                    menuVO.setEnabled(true);
                } else {
                    if (CharSequenceUtil.isBlank(menuVO.getMenuName())) {
                        menuVO.setMenuName(menuBO.getMenusName());
                    }
                    menuVO.setOriginalMenuName(menuBO.getMenusName());
                    menuVO.setOriginalMenusIcon(menuBO.getMenusIcon());

                    menuVO.setHidden(menuBO.getDisplay() != null && !menuBO.getDisplay());
                    menuVO.setEnabled(menuBO.getMenusState() == null || menuBO.getMenusState());
                }
            } else if (MenuTreeNodeType.APP == nodeType) {
                var app = allApps.get(menuTreeDO.getMenuCode());
                if (app == null) {
                    log.info("租户应用已不存在：{}", menuTreeDO.getMenuCode());
                    continue;
                }
                if (CharSequenceUtil.isBlank(menuVO.getMenuName())) {
                    menuVO.setMenuName(app.getName());
                }
                menuVO.setOriginalMenuName(app.getName());
                menuVO.setOriginalMenusIcon("");
                menuVO.setHidden(false);
                menuVO.setEnabled(true);
            } else {
                log.error("暂不支持的节点类型：{}", nodeType.getValue());
                continue;
            }
            menuVO.setNodeType(menuTreeDO.getNodeType());
            menuVO.setParentMenuCode(menuTreeDO.getParentMenuCode());
            menuVO.setMenusIcon(menuTreeDO.getIcon());
            menuVO.setSortNo(menuTreeDO.getSortNo());
            menuVO.setHasChildren(false);
            menuVO.setChildren(new ArrayList<>(128));

            menuVOList.add(menuVO);
        }

        // 树形结构
        if (BooleanUtil.isFalse(tree)) {
            return ApiResult.ok(menuVOList);
        }
        TreeDataUtil<MenuTreeCustomRespVO> treeDataUtil = new TreeDataUtil<>(menuVOList, MenuTreeCustomRespVO::getMenuCode,
                MenuTreeCustomRespVO::getParentMenuCode, MenuTreeCustomRespVO::setChildren, Comparator.comparing(MenuTreeCustomRespVO::getSortNo, Comparator.nullsLast(Integer::compareTo)));
        return ApiResult.ok((List<MenuTreeCustomRespVO>) treeDataUtil.getRoots());
    }

    @Override
    public ApiResult<Boolean> getCustomEnabled() {
        var tenantId = currentTenantId();
        var enabled = tenantMenuRepoProc.getEnabledByTenant(tenantId);
        return ApiResult.ok(ObjectUtil.defaultIfNull(enabled, false));
    }

    @Override
    public ResponseEntity<StreamingResponseBody> export() {
        var menuList = tenantMenuTreeRepoProc.queryMenuBos();
        if (menuList.isEmpty()) {
            // 不存在
            var notFound = "未配置自定义菜单".getBytes(StandardCharsets.UTF_8);
            return ResponseEntity.ok()
                    .contentType(MediaType.APPLICATION_OCTET_STREAM)
                    .header(HttpHeaders.CONTENT_DISPOSITION, ContentDisposition.builder("attachment")
                            .filename("未配置自定义菜单.xlsx", StandardCharsets.UTF_8)
                            .build().toString())
                    .contentLength(notFound.length)
                    .body(outputStream -> {
                        try {
                            StreamUtils.copy(notFound, outputStream);
                        } catch (Exception e) {
                            log.error("下载资源文件异常：", e);
                        }
                    });
        }

        var bytes = export2Bytes(menuList);
        // 下载文件
        var contentDisposition = ContentDisposition.builder("attachment")
                .filename("自定义菜单.xlsx", StandardCharsets.UTF_8)
                .build();
        return ResponseEntity.ok()
                .contentType(MimeTypeConstant.parseMediaType(MimeTypeConstant.XLSX))
                .header(HttpHeaders.CONTENT_DISPOSITION, contentDisposition.toString())
                .contentLength(bytes.length)
                .body(outputStream -> {
                    try {
                        StreamUtils.copy(bytes, outputStream);
                    } catch (Exception e) {
                        log.error("导出自定义菜单异常：", e);
                    }
                });
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> importByFile(MultipartFile file, Boolean increment) {
        if (file == null || file.isEmpty()) {
            return ApiResult.fail("导入文件为空");
        }

        // 解析导入文件
        List<TenantMenuBO> dataList = null;
        try {
            dataList = this.analyzeImportFile(file);
        } catch (Exception e) {
            log.error("解析导入数据异常", e);
            return ApiResult.fail("解析导入数据异常");
        }

        // 保存导入数据
        this.saveImportData(dataList, ObjUtil.defaultIfNull(increment, false));

        // 清空缓存
        clearCache();

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> importByPlatformMenu() {
        var appMap = permissionQueryManager.tenantApps(null);
        if (appMap != null && appMap.isEmpty()) {
            return ApiResult.fail("未找到有效菜单");
        }
        // 查询原始数据
        var dataList = queryPlatformMenuToImport(appMap);
        if (dataList.isEmpty()) {
            return ApiResult.fail("菜单数据为空");
        }

        // 保存数据
        this.saveImportData(dataList, false);

        // 清空缓存
        clearCache();

        return ApiResult.ok(true);
    }
    
    private void clearCache() {
        cacheMenuRpcService.clearCache();
    }

    private void deleteCascade(List<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            return;
        }

        // 查找下级，先删除下级
        var childrenIds = tenantMenuTreeRepoProc.listChildrenIds(ids);
        this.deleteCascade(childrenIds);

        // 删除自身
        tenantMenuTreeRepoProc.delete(ids);
    }

    private List<TenantMenuBO> queryPlatformMenuToImport(Map<String, AppBO> appMap) {
        // 查询原始菜单列表
        var menuList = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.queryMenu(appMap == null ? null : appMap.keySet(), false, true, true, true))
                .stream()
                .filter(t -> StringUtils.hasText(t.getMenusAppCode()))
                .map(t -> {
                    TenantMenuBO tenantMenuBO = new TenantMenuBO();
                    tenantMenuBO.setMenuCode(t.getMenusCode());
                    tenantMenuBO.setMenuName(t.getMenusName());

                    var nodeType = StringUtils.hasText(t.getMenusRoute()) ? MenuTreeNodeType.MENU.getValue() : MenuTreeNodeType.MENU_GROUP.getValue();
                    tenantMenuBO.setNodeType(nodeType);

                    tenantMenuBO.setParentMenuCode(t.getMenusParentCode());
                    tenantMenuBO.setIcon(t.getMenusIcon());
                    tenantMenuBO.setSortNo(t.getMenusOrder());
                    tenantMenuBO.setEnabled(true);
                    tenantMenuBO.setAppCode(t.getMenusAppCode());

                    return tenantMenuBO;
                }).collect(Collectors.toList());
        if (menuList.isEmpty()) {
            return Collections.emptyList();
        }

        // 转换格式
        return this.normalizePlatformMenuToImport(menuList, appMap);
    }

    private List<TenantMenuBO> normalizePlatformMenuToImport(List<TenantMenuBO> menuList, Map<String, AppBO> appMap) {
        // 对菜单进行分组
        Set<String> menuGroupCodes = new HashSet<>(64);
        Set<String> defaultMenuGroupCodes = new HashSet<>(64);
        Map<String, String> menuParentCodeMap = new HashMap<>(menuList.size());
        for (TenantMenuBO tenantMenuBO : menuList) {
            if (StringUtils.hasText(tenantMenuBO.getParentMenuCode())) {
                menuParentCodeMap.put(tenantMenuBO.getMenuCode(), tenantMenuBO.getParentMenuCode());
                continue;
            }
            menuGroupCodes.add(tenantMenuBO.getMenuCode());
        }
        List<TenantMenuBO> resultList = new ArrayList<>();
        for (var menuBO : menuList) {
            // 原顶级菜单分组，将作为自定义二级分组
            if (menuGroupCodes.contains(menuBO.getMenuCode())) {
                if (MenuTreeNodeType.MENU.getValue().equals(menuBO.getNodeType())) {
                    // 如果是菜单，则为其添加个默认分组
                    String parentMenuCode = "__" + menuBO.getAppCode() + "_";
                    menuBO.setParentMenuCode(parentMenuCode);
                    resultList.add(menuBO);

                    // 添加默认分组
                    if (!defaultMenuGroupCodes.contains(parentMenuCode)) {
                        TenantMenuBO tenantMenuBO = new TenantMenuBO();
                        tenantMenuBO.setMenuCode(parentMenuCode);
                        tenantMenuBO.setMenuName("默认分组");

                        tenantMenuBO.setNodeType(MenuTreeNodeType.MENU_GROUP.getValue());

                        tenantMenuBO.setParentMenuCode(menuBO.getAppCode());
                        tenantMenuBO.setSortNo(0);
                        tenantMenuBO.setEnabled(true);

                        defaultMenuGroupCodes.add(parentMenuCode);
                        resultList.add(tenantMenuBO);
                    }
                    continue;
                }
                menuBO.setParentMenuCode(menuBO.getAppCode());
                resultList.add(menuBO);
                continue;
            }

            // 其它的都将归纳到二级分组下
            String tempParentCode = menuBO.getParentMenuCode();
            String tempMenuCode = menuBO.getMenuCode();
            while (!menuGroupCodes.contains(tempParentCode)) {
                tempParentCode = menuParentCodeMap.get(tempMenuCode);
                tempMenuCode = tempParentCode;
                if (tempParentCode == null) {
                    break;
                }
            }
            if (StringUtils.hasText(tempParentCode)) {
                menuBO.setParentMenuCode(tempParentCode);
                resultList.add(menuBO);
            }
        }

        // 添加所属应用作为一级分组
        for (AppBO app : appMap.values()) {
            TenantMenuBO tenantMenuBO = new TenantMenuBO();
            tenantMenuBO.setMenuCode(app.getAppCode());
            tenantMenuBO.setMenuName(app.getAppName());

            tenantMenuBO.setNodeType(MenuTreeNodeType.MENU_GROUP.getValue());

            tenantMenuBO.setParentMenuCode(null);
            tenantMenuBO.setSortNo(ObjectUtil.defaultIfNull(app.getAppOrder(), 0));
            tenantMenuBO.setEnabled(true);

            resultList.add(tenantMenuBO);
        }
        return resultList;
    }

    private void saveImportData(List<TenantMenuBO> dataList, boolean increment) {
        if (CollUtil.isEmpty(dataList)) {
            throw new BusinessException("导入数据为空");
        }

        List<String> existsCodes = increment ? tenantMenuTreeRepoProc.allMenuCodes() : Collections.emptyList();
        var nodeTypesAll = Set.of(MenuTreeNodeType.MENU_GROUP.getValue(), MenuTreeNodeType.MENU.getValue());

        var tenantMenuTreeDoList = dataList.stream()
                .filter(t -> {
                    if (CharSequenceUtil.isBlank(t.getMenuCode())) {
                        return false;
                    }
                    if (increment) {
                        return !existsCodes.contains(t.getMenuCode());
                    }
                    return true;
                })
                .map(t -> {
                    SysTenantMenuTreeDO tenantMenuTreeDO = new SysTenantMenuTreeDO();
                    tenantMenuTreeDO.setMenuCode(t.getMenuCode());
                    tenantMenuTreeDO.setMenuName(t.getMenuName());

                    tenantMenuTreeDO.setNodeType(t.getNodeType());
                    Assert.hasText(t.getNodeType(), "存在节点类型为空");
                    Assert.isTrue(nodeTypesAll.contains(t.getNodeType()), () -> "节点类型只能是：" + String.join("、", nodeTypesAll));

                    tenantMenuTreeDO.setParentMenuCode(StringUtils.hasText(t.getParentMenuCode()) ? t.getParentMenuCode() : null);
                    tenantMenuTreeDO.setIcon(t.getIcon());
                    tenantMenuTreeDO.setCustom(true);
                    tenantMenuTreeDO.setSortNo(ObjectUtil.defaultIfNull(t.getSortNo(), 0));
                    tenantMenuTreeDO.setEnabled(true);

                    return tenantMenuTreeDO;
                }).collect(Collectors.toList());
        if (!increment) {
            // 先清空
            tenantMenuTreeRepoProc.delete();
        }
        if (!tenantMenuTreeDoList.isEmpty()) {
            tenantMenuTreeRepoProc.save(tenantMenuTreeDoList);
        }

        // 获取现有配置
        var tenantId = currentTenantId();
        var enabledCurrent = tenantMenuRepoProc.getByTenantId(tenantId);
        if (enabledCurrent == null) {
            // 如果没有，则新建
            var tenantMenuDO = new SysTenantMenuDO();
            tenantMenuDO.setSysTenantId(tenantId);
            tenantMenuDO.setEnabled(true);
            tenantMenuRepo.save(tenantMenuDO);
        }
    }

    @SuppressWarnings("unchecked")
    private List<TenantMenuBO> analyzeImportFile(MultipartFile file) throws Exception {
        return (List<TenantMenuBO>) ExcelImportUtil.instance(file.getInputStream())
                .headRow(2)
                .dataType(TenantMenuBO.class, 2)
                .readAllSync();
    }

    private byte[] export2Bytes(List<TenantMenuBO> menuList) {
        List<ExportColumnParam> columnList = new ArrayList<>();
        columnList.add(new ExportColumnParam("menuCode", "菜单编码"));
        columnList.add(new ExportColumnParam("menuName", "菜单名称"));
        columnList.add(new ExportColumnParam("nodeType", "节点类型"));
        columnList.add(new ExportColumnParam("parentMenuCode", "上级菜单编码"));
        columnList.add(new ExportColumnParam("icon", "图标"));
        columnList.add(new ExportColumnParam("sortNo", "排序"));

        menuList = menuList.stream()
                .sorted(Comparator.comparing(TenantMenuBO::getParentMenuCode, Comparator.nullsFirst(String::compareTo))
                        .thenComparing(TenantMenuBO::getSortNo, Comparator.nullsLast(Integer::compareTo)))
                .collect(Collectors.toList());
        var bos = new ByteArrayOutputStream();
        try {
            ExcelExportUtil.instance(bos)
                    .fields(columnList, true)
                    .write("sys_tenant_menu_tree", menuList);
        } catch (IOException e) {
            throw new BusinessException("导出异常", e);
        }
        return bos.toByteArray();
    }

    private <T> void setDepth(List<T> dataList, Function<T, List<T>> getChildren, int parentDepth, BiConsumer<T, Integer> setDepth) {
        if (CollectionUtils.isEmpty(dataList)) {
            return;
        }

        for (T data : dataList) {
            setDepth.accept(data, parentDepth + 1);

            this.setDepth(getChildren.apply(data), getChildren, parentDepth + 1, setDepth);
        }
    }

    private List<CustomMenuTreeRespVO> queryCustomMenuTree() {
        // 先查询自定义菜单
        var tenantMenuList = tenantMenuTreeRepoProc.all().stream()
                .map(t -> {
                    CustomMenuTreeRespVO respVO = new CustomMenuTreeRespVO();
                    respVO.setNodeType(t.getNodeType());
                    respVO.setIcon(t.getIcon());
                    respVO.setId(t.getId());
                    respVO.setCode(t.getMenuCode());
                    respVO.setName(t.getMenuName());
                    respVO.setSortNo(ObjectUtil.defaultIfNull(t.getSortNo(), 0));
                    respVO.setParentId(null);
                    respVO.setParentCode(t.getParentMenuCode());

                    return respVO;
                }).collect(Collectors.toList());
        if (tenantMenuList.isEmpty()) {
            return tenantMenuList;
        }
        var groupCodeAndIdMap = tenantMenuList.stream()
                .filter(t -> MenuTreeNodeType.MENU_GROUP.getValue().equals(t.getNodeType()))
                .collect(Collectors.toMap(CustomMenuTreeRespVO::getCode, CustomMenuTreeRespVO::getId, (t1, t2) -> t1));

        // 设置菜单信息
        var menuCodeAndNameMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuCodeAndNames())
                .stream().collect(Collectors.toMap(IdCodeNameParam::getCode, IdCodeNameParam::getName));
        for (CustomMenuTreeRespVO respVO : tenantMenuList) {
            if (MenuTreeNodeType.MENU_GROUP.getValue().equals(respVO.getNodeType())) {
                if (StringUtils.hasText(respVO.getParentCode())) {
                    respVO.setParentId(groupCodeAndIdMap.get(respVO.getParentCode()));
                }
                continue;
            }

            if (MenuTreeNodeType.MENU.getValue().equals(respVO.getNodeType())) {
                respVO.setParentId(groupCodeAndIdMap.get(respVO.getParentCode()));
                if (CharSequenceUtil.isBlank(respVO.getName())) {
                    respVO.setName(menuCodeAndNameMap.get(respVO.getCode()));
                }
            }
        }
        return tenantMenuList;
    }

    private SysTenantMenuTreeDO convert(CustomMenuGroupSaveVO saveVO) {
        SysTenantMenuTreeDO menuTreeDO = null;
        boolean exists = false;
        if (saveVO.getId() == null) {
            menuTreeDO = new SysTenantMenuTreeDO();

            // 判断编码是否存在
            Assert.hasText(saveVO.getMenuCode(), "编码为空");
            exists = tenantMenuTreeRepoProc.existsCustomMenuCode(saveVO.getMenuCode(), saveVO.getId());
            Assert.isTrue(!exists, "编码已存在");
        } else {
            menuTreeDO = tenantMenuTreeRepoProc.get(saveVO.getId());
            Assert.notNull(menuTreeDO, "编辑的数据不存在");

            // 编码不可修改
            Assert.isTrue(menuTreeDO.getMenuCode().equals(saveVO.getMenuCode()), "编码不可修改");
        }
        menuTreeDO = TenantMenuTreeConvert.INSTANCE.customGroup2Do(saveVO, menuTreeDO);

        // 判断分组是否存在
        if (StringUtils.hasText(saveVO.getParentMenuCode())) {
            exists = tenantMenuTreeRepoProc.existsCustomMenuCode(saveVO.getParentMenuCode(), null);
            Assert.isTrue(exists, "分组不存在");
        } else {
            menuTreeDO.setParentMenuCode(null);
        }

        menuTreeDO.setNodeType(MenuTreeNodeType.MENU_GROUP.getValue());
        if (menuTreeDO.getSortNo() == null) {
            menuTreeDO.setSortNo(0);
        }
        if (menuTreeDO.getEnabled() == null) {
            menuTreeDO.setEnabled(true);
        }

        menuTreeDO.setCustom(true);
        return menuTreeDO;
    }

    private List<MenuTreeRespVO> convertMenuTreeRespVO(Set<String> appCodes) {
        // 先获取应用
        var allApps = appProvider.all();
        if (allApps.isEmpty()) {
            throw new BusinessException("未获取都有效的应用列表");
        }

        List<MenuTreeRespVO> respVOList = new ArrayList<>(256);
        if (CollectionUtils.isEmpty(appCodes)) {
            // 所有应用
            MenuTreeRespVO respVO = null;
            int sortNo = 1;
            for (CodeNameParam app : allApps) {
                respVO = new MenuTreeRespVO();
                respVO.setMenuCode(app.getCode());
                respVO.setMenuName(app.getName());
                respVO.setNodeType(MenuTreeNodeType.APP.getValue());
                respVO.setHidden(false);
                respVO.setEnabled(true);
                respVO.setSortNo(sortNo++);
                respVO.setHasChildren(false);
                respVO.setChildren(new ArrayList<>(128));

                respVOList.add(respVO);
            }
            return respVOList;
        }

        MenuTreeRespVO respVO = null;
        int sortNo = 1;
        for (var app : allApps) {
            if (!appCodes.contains(app.getCode())) {
                // 未分配当前应用
                continue;
            }

            respVO = new MenuTreeRespVO();
            respVO.setMenuCode(app.getCode());
            respVO.setMenuName(app.getName());
            respVO.setNodeType(MenuTreeNodeType.APP.getValue());
            respVO.setHidden(false);
            respVO.setEnabled(true);
            respVO.setSortNo(sortNo++);
            respVO.setHasChildren(false);
            respVO.setChildren(new ArrayList<>(128));

            respVOList.add(respVO);
        }

        return respVOList;
    }

    private List<SysTenantMenuTreeDO> convertAndCheck(Long tenantId, List<MenuTreeSaveVO> saveVOList) {
        if (CollectionUtils.isEmpty(saveVOList)) {
            return Collections.emptyList();
        }
        // 转为do列表
        List<SysTenantMenuTreeDO> menuTreeDOList = new ArrayList<>(256);
        convertToList(saveVOList, null, menuTreeDOList);

        var menuCodes = menuTreeDOList.stream().map(SysTenantMenuTreeDO::getMenuCode).collect(Collectors.toSet());
        // 检查参数
        MenuTreeNodeType nodeType = null;
        for (SysTenantMenuTreeDO menuTreeDO : menuTreeDOList) {
            if (StringUtils.hasText(menuTreeDO.getParentMenuCode()) && !menuCodes.contains(menuTreeDO.getParentMenuCode())) {
                // 去掉顶级菜单的parentCode
                menuTreeDO.setParentMenuCode("");
            }

            // 节点类型
            nodeType = MenuTreeNodeType.valueOf(menuTreeDO.getNodeType());
            Assert.notNull(nodeType, "未知节点类型" + menuTreeDO.getNodeType());

            // 是否自定义
            if (menuTreeDO.getCustom() == null) {
                menuTreeDO.setCustom(false);
            }
            if (nodeType == MenuTreeNodeType.APP) {
                menuTreeDO.setCustom(false);
            }
        }

        // 设置排序
        var menuTreeMap = menuTreeDOList.stream().collect(Collectors.groupingBy(SysTenantMenuTreeDO::getParentMenuCode));
        for (List<SysTenantMenuTreeDO> menus : menuTreeMap.values()) {
            int sortNo = 0;
            for (SysTenantMenuTreeDO menu : menus) {
                menu.setSortNo(sortNo++);
            }
        }

        return menuTreeDOList;
    }

    private void convertToList(List<MenuTreeSaveVO> saveVOList, String parentMenuCode, List<SysTenantMenuTreeDO> menuTreeDOList) {
        if (CollectionUtils.isEmpty(saveVOList)) {
            return;
        }
        if (parentMenuCode == null) {
            parentMenuCode = "";
        }

        int sortNo = 1;
        List<SysTenantMenuTreeDO> newList = new ArrayList<>(128);
        SysTenantMenuTreeDO menuTreeDO = null;
        for (MenuTreeSaveVO menuTreeSaveVO : saveVOList) {
            menuTreeDO = CONVERT.saveVo2Do(menuTreeSaveVO);
            if (StringUtils.hasText(parentMenuCode) || menuTreeDO.getParentMenuCode() == null) {
                menuTreeDO.setParentMenuCode(parentMenuCode);
            }
            menuTreeDO.setIcon(menuTreeSaveVO.getMenusIcon());
            menuTreeDO.setSortNo(sortNo);

            sortNo++;
            newList.add(menuTreeDO);

            // 如果有子列表，则遍历下级
            convertToList(menuTreeSaveVO.getChildren(), menuTreeSaveVO.getMenuCode(), menuTreeDOList);
        }
        menuTreeDOList.addAll(newList);
    }
}
