package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.NoticeConvert;
import com.elitescloud.cloudt.system.model.vo.query.extend.NoticePageMngQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.NoticeEditRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.NoticePageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.NoticeSaveVO;
import com.elitescloud.cloudt.system.provider.dto.save.SysNoticeSaveDTO;
import com.elitescloud.cloudt.system.service.NoticeMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysNoticeDO;
import com.elitescloud.cloudt.system.service.model.entity.SysNoticeTxtDO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/4/22
 */
@Service
@Slf4j
public class NoticeMngServiceImpl extends BaseNoticeService implements NoticeMngService {

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(NoticeSaveVO saveVO) {
        var noticeDO = this.upsert(saveVO);
        return ApiResult.ok(noticeDO.getId());
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveSys(NoticeSaveVO saveVO) {
        var noticeDO = this.upsert(saveVO);
        return ApiResult.ok(noticeDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<Long>> delete(Set<Long> ids, Boolean real) {
        if (ids != null) {
            ids = ids.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        }
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("公告ID为空");
        }

        // 删除
        noticeAdapter(ids, t -> {
            if (real == null || Boolean.TRUE.equals(real)) {
                repoProc.delete(t);
                txtRepoProc.deleteByNoticeId(t);
            } else {
                repoProc.updateDeleteFlag(t);
            }
        });

        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<Long>> execPublish(Set<Long> ids) {
        if (ids != null) {
            ids = ids.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        }
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("公告ID为空");
        }

        // 发布
        this.noticeAdapter(ids, t -> {
            repoProc.updatePublish(t);
            repoProc.updatePublish(t, LocalDateTime.now());
        });

        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<Long>> execRevoke(Set<Long> ids) {
        if (ids != null) {
            ids = ids.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        }
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("公告ID为空");
        }

        // 撤回
        this.noticeAdapter(ids, t -> {
            repoProc.updateRevoke(t);
        });

        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> publish(SysNoticeSaveDTO saveDTO) {
        Assert.hasText(saveDTO.getTitle(), "标题为空");

        // 保存公告
        var saveVO = NoticeConvert.INSTANCE.dto2SaveVO(saveDTO);
        Long noticeId = null;
        if (saveVO.getSys() == null || !saveVO.getSys()) {
            // 租户的公告
            saveVO.setSys(false);
            noticeId = this.save(saveVO).computeData();
        } else {
            noticeId = SpringContextHolder.getBean(NoticeMngService.class).saveSys(saveVO).computeData();
        }

        // 发布公告
        this.execPublish(Set.of(noticeId));

        return ApiResult.ok(noticeId);
    }

    @Override
    public ApiResult<NoticeEditRespVO> getEditVO(Long id) {
        if (id == null) {
            return ApiResult.fail("ID为空");
        }

        var noticeVO = get(id);
        if (noticeVO == null) {
            noticeVO = tenantDataIsolateProvider.byDefaultDirectly(() -> get(id));
        }

        return ApiResult.ok(noticeVO);
    }

    @Override
    public ApiResult<PagingVO<NoticePageRespVO>> pageMng(NoticePageMngQueryVO queryVO) {
        var pageData = this.queryByPage(queryVO, false);

        return ApiResult.ok(pageData);
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<PagingVO<NoticePageRespVO>> pageSysMng(NoticePageMngQueryVO queryVO) {
        var pageData = this.queryByPage(queryVO, true);

        return ApiResult.ok(pageData);
    }

    private void noticeAdapter(Collection<Long> ids, Consumer<Collection<Long>> consumer) {
        var normalIds = repoProc.exists(ids);
        if (!normalIds.isEmpty()) {
            consumer.accept(normalIds);

            if (normalIds.size() == ids.size()) {
                return;
            }
        }

        // 系统公告
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            var sysIds = repoProc.exists(ids);
            if (!sysIds.isEmpty()) {
                consumer.accept(sysIds);
            }

            return null;
        });
    }

    private NoticeEditRespVO get(long id) {
        var noticeDO = repoProc.get(id);
        if (noticeDO == null) {
            return null;
        }
        var noticeVO = NoticeConvert.INSTANCE.convert2EditVO(noticeDO);
        noticeVO.setPublished(this.hasPublished(noticeDO));
        super.fillDetailInfo(noticeDO, noticeVO);

        return noticeVO;
    }

    private SysNoticeDO upsert(NoticeSaveVO saveVO) {
        SysNoticeDO noticeDO = this.checkAndConvert(saveVO);

        // 保存公告信息
        repoProc.save(noticeDO);
        // 保存公告内容
        var noticeTxtDO = this.saveTxt(saveVO, noticeDO);
        repoProc.updateTxtId(noticeDO.getId(), noticeTxtDO.getId());
        return noticeDO;
    }

    private PagingVO<NoticePageRespVO> queryByPage(NoticePageMngQueryVO queryVO, boolean sys) {
        var pageData = repoProc.pageMng(queryVO, sys).map(t -> {
            var respVO = NoticeConvert.INSTANCE.convert2PageVO(t);
            respVO.setPublished(this.hasPublished(t));
            return respVO;
        });
        if (pageData.isEmpty()) {
            return pageData;
        }

        super.fillListInfo(pageData.getRecords());

        return pageData;
    }

    private boolean hasPublished(SysNoticeDO noticeDO) {
        if (Boolean.FALSE.equals(noticeDO.getPublished())) {
            return false;
        }

        return noticeDO.getPublishTime() != null && noticeDO.getPublishTime().isBefore(LocalDateTime.now());
    }

    private SysNoticeDO checkAndConvert(NoticeSaveVO saveVO) {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();

        SysNoticeDO noticeDO = null;
        if (saveVO.getId() == null) {
            noticeDO = new SysNoticeDO();
        } else {
            noticeDO = repoProc.get(saveVO.getId());
            Assert.notNull(noticeDO, "公告不存在");

            // 已发布的不可修改
            if (this.hasPublished(noticeDO)) {
                throw new BusinessException("公告已发布，不可修改");
            }
        }

        Assert.hasText(saveVO.getTitle(), "标题为空");
        Assert.hasText(saveVO.getTxt(), "内容为空");

        // 默认值处理
        ObjUtil.ifNull(saveVO.getAuthorId(), currentUser.getUserId(), saveVO::setAuthorId);
        ObjUtil.ifNull(saveVO.getTop(), false, saveVO::setTop);
        ObjUtil.ifNull(saveVO.getSys(), false, saveVO::setSys);

        NoticeConvert.INSTANCE.convert2DO(saveVO, noticeDO);

        noticeDO.setPublished(saveVO.getPublishTime() != null);
        noticeDO.setChangeTime(LocalDateTime.now());

        if (CollUtil.isNotEmpty(saveVO.getFileCodes())) {
            noticeDO.setFileCodesStr(String.join(",", saveVO.getFileCodes()));
        } else {
            noticeDO.setFileCodesStr(null);
        }

        ObjUtil.ifNull(noticeDO.getReadAmount(), 0L, noticeDO::setReadAmount);

        return noticeDO;
    }

    private SysNoticeTxtDO saveTxt(NoticeSaveVO saveVO, SysNoticeDO noticeDO) {
        SysNoticeTxtDO noticeTxtDO = noticeDO.getTxtId() == null ? new SysNoticeTxtDO() : txtRepoProc.get(noticeDO.getTxtId());
        noticeTxtDO.setTxt(saveVO.getTxt());
        noticeTxtDO.setNoticeId(noticeDO.getId());

        txtRepoProc.save(noticeTxtDO);
        return noticeTxtDO;
    }
}
