package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.NoticeConvert;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonPublishedNoticePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonPublishedNoticePageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonPublishedNoticeRespVO;
import com.elitescloud.cloudt.system.service.NoticeQueryService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.concurrent.CompletableFuture;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/4/22
 */
@Slf4j
@Service
public class NoticeQueryServiceImpl extends BaseNoticeService implements NoticeQueryService {

    @Override
    public ApiResult<PagingVO<CommonPublishedNoticePageRespVO>> pageQueryPublished(CommonPublishedNoticePageQueryVO queryVO) {
        var pageData = pageQuery(queryVO);
        return ApiResult.ok(pageData);
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<PagingVO<CommonPublishedNoticePageRespVO>> pageQueryPublishedSys(CommonPublishedNoticePageQueryVO queryVO) {
        var pageData = pageQuery(queryVO);
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<CommonPublishedNoticeRespVO> viewPublishedNotice(Long id) {
        if (id == null) {
            return ApiResult.fail("公告ID为空");
        }

        CommonPublishedNoticeRespVO noticeVO = view(id);
        if (noticeVO == null) {
            noticeVO = tenantDataIsolateProvider.byDefaultDirectly(() -> view(id));
        }

        return ApiResult.ok(noticeVO);
    }

    private PagingVO<CommonPublishedNoticePageRespVO> pageQuery(CommonPublishedNoticePageQueryVO queryVO) {
        var pageData = repoProc.pagePublished(queryVO).map(NoticeConvert.INSTANCE::convert2PublishedPageVO);
        if (pageData.isEmpty()) {
            return pageData;
        }

        super.fillListInfo(pageData.getRecords());

        return pageData;
    }

    private CommonPublishedNoticeRespVO view(long id) {
        var noticeDO = repoProc.get(id);
        if (noticeDO == null || Boolean.FALSE.equals(noticeDO.getPublished()) || noticeDO.getPublishTime().isAfter(LocalDateTime.now())) {
            return null;
        }

        var noticeVO = NoticeConvert.INSTANCE.convert2PublishedVO(noticeDO);
        super.fillDetailInfo(noticeDO, noticeVO);

        // 增加阅读量
        CompletableFuture.runAsync(() -> {
            repoProc.increaseNum(id);
        }, taskExecutor).whenComplete((v, e) -> {
            if (e != null) {
                log.error("更新阅读量异常：{}", id, e);
            }
        });
        return noticeVO;
    }
}
