package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformMenusApiDO;
import com.elitescloud.cloudt.system.common.DprPropagateStrategyEnum;
import com.elitescloud.cloudt.system.model.vo.save.role.*;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformMenusDO;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformMenusApiRepoProc;
import com.elitescloud.cloudt.system.convert.PermissionConverter;
import com.elitescloud.cloudt.system.dto.req.UserRoleSaveDTO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.*;
import com.elitescloud.cloudt.system.service.PermissionMngService;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.manager.PermissionMngManager;
import com.elitescloud.cloudt.system.service.model.bo.*;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAppDO;
import com.elitescloud.cloudt.system.service.repo.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/23/2023
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class PermissionMngServiceImpl extends BaseServiceImpl implements PermissionMngService {

    @Autowired
    private PermissionMngManager permissionMngManager;
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private SysPlatformMenusApiRepoProc menusApiRepoProc;
    @Autowired
    private MenuRepoProc menuRepoProc;
    @Autowired
    private ApiManageRepoProc apiRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;
    @Autowired
    private OrgRepoProc orgRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveUserRoles(Long userId, List<Long> roleIds) {
        // 保存用户角色
        permissionMngManager.saveUserRole(userId, roleIds);

        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> saveUserRoles(List<UserRoleSaveDTO> saveDTOList) {
        if (CollUtil.isEmpty(saveDTOList)) {
            return ApiResult.fail("保存数据为空");
        }

        // 先查询用户和角色信息
        Set<String> usernames = new HashSet<>(64);
        Set<String> roleCodes = new HashSet<>(64);
        for (UserRoleSaveDTO userRoleSaveDTO : saveDTOList) {
            if (userRoleSaveDTO.getUserId() == null) {
                if (CharSequenceUtil.isBlank(userRoleSaveDTO.getUsername())) {
                    return ApiResult.fail("存在用户信息为空的记录");
                }
                usernames.add(userRoleSaveDTO.getUsername());
            }

            if (CollUtil.isEmpty(userRoleSaveDTO.getRoleIds())) {
                if (CollUtil.isNotEmpty(userRoleSaveDTO.getRoleCodes())) {
                    roleCodes.addAll(userRoleSaveDTO.getRoleCodes());
                }
            }
        }
        Map<String, Long> userIdMap;
        if (usernames.isEmpty()) {
            userIdMap = Collections.emptyMap();
        } else {
            userIdMap = tenantDataIsolateProvider.byDefaultDirectly(() -> userRepoProc.getIdAndUsernameByUsername(usernames));
            var notExistsUsernames = usernames.stream().filter(t -> !userIdMap.containsKey(t)).collect(Collectors.joining(","));
            if (CharSequenceUtil.isNotBlank(notExistsUsernames)) {
                return ApiResult.fail("以下用户不存在：" + notExistsUsernames);
            }
        }
        Map<String, Long> roleIdMap;
        if (!roleCodes.isEmpty()) {
            roleIdMap = roleRepoProc.getIdAndCodeByCode(roleCodes);
            var notExistsRoles = roleCodes.stream().filter(t -> !roleIdMap.containsKey(t)).collect(Collectors.joining(","));
            if (CharSequenceUtil.isNotBlank(notExistsRoles)) {
                return ApiResult.fail("以下角色不存在：" + notExistsRoles);
            }
        } else {
            roleIdMap = Collections.emptyMap();
        }

        // 保存用户角色
        List<Long> userIdList = new ArrayList<>();
        for (UserRoleSaveDTO dto : saveDTOList) {
            var userId = ObjectUtil.defaultIfNull(dto.getUserId(), userIdMap.get(dto.getUsername()));
            List<Long> roleIds = null;
            if (CollUtil.isEmpty(dto.getRoleIds()) && CollUtil.isNotEmpty(dto.getRoleCodes())) {
                roleIds = dto.getRoleCodes().stream().map(roleIdMap::get).collect(Collectors.toList());
            } else {
                roleIds = CollUtil.isEmpty(dto.getRoleIds()) ? Collections.emptyList() : new ArrayList<>(dto.getRoleIds());
            }
            this.saveUserRoles(userId, roleIds);
        }
        return ApiResult.ok(userIdList);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveRoleUser(Long roleId, List<Long> userIds, Boolean incremental) {
        if (roleId == null) {
            return ApiResult.fail("角色ID为空");
        }

        if (incremental == null) {
            incremental = false;
        }
        permissionMngManager.saveRoleUser(roleId, userIds, incremental, false);

        return ApiResult.ok(roleId);
    }

    @Override
    public ApiResult<Long> saveRoleEmployee(Long roleId, List<Long> userIds, Boolean incremental) {
        if (roleId == null) {
            return ApiResult.fail("角色ID为空");
        }

        if (incremental == null) {
            incremental = false;
        }
        permissionMngManager.saveRoleUser(roleId, userIds, incremental, true);

        return ApiResult.ok(roleId);
    }

    @Override
    public ApiResult<Boolean> saveUserRoleByOrg(OrgRolePermissionSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getOrgCodes()) || CollUtil.isEmpty(saveVO.getRoleCodes())) {
            return ApiResult.fail("缺少必要参数");
        }

        // 查找账户ID
        var userIds = orgRepoProc.queryEmployeeByOrg(saveVO.getOrgCodes(), Boolean.TRUE.equals(saveVO.getWithChildrenOrg()), true);
        if (userIds.isEmpty()) {
            return ApiResult.ok(true);
        }

        // 保存用户的角色
        permissionMngManager.addUserRole(userIds, saveVO.getRoleCodes());
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> removeRoleUser(Long roleId, Long userId) {
        if (ObjectUtil.hasNull(roleId, userId)) {
            return ApiResult.fail("角色和账号不能为空");
        }

        permissionMngManager.removeRoleUser(roleId, List.of(userId));
        return ApiResult.ok(roleId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> savePermissionMenu(Long roleId, List<RoleAppPermissionSaveVO> saveVOList) {
        // 过滤出菜单和按钮
        List<SysAppPermissionSaveBO> saveBos = null;
        if (CollectionUtils.isEmpty(saveVOList)) {
            saveBos = Collections.emptyList();
        } else {
            saveBos = saveVOList.stream()
                    .map(this::vo2Bo)
                    .collect(Collectors.toList());
        }

        // 保存权限菜单
        permissionMngManager.saveRoleMenu(roleId, saveBos);

        return ApiResult.ok(roleId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> addPermissionMenu(Long roleId, RoleAppPermissionSaveVO saveVO) {
        if (roleId == null) {
            return ApiResult.fail("角色ID为空");
        }
        if (saveVO == null || CharSequenceUtil.isBlank(saveVO.getAppCode()) || CollUtil.isEmpty(saveVO.getPermissionList())) {
            return ApiResult.fail("权限信息为空");
        }

        var saveBO = this.vo2Bo(saveVO);
        permissionMngManager.addRoleMenu(roleId, saveBO);

        return ApiResult.ok(roleId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> removePermissionMenu(Long roleId, RoleAppPermissionSaveVO saveVO) {
        if (roleId == null) {
            return ApiResult.fail("角色ID为空");
        }
        if (saveVO == null || CharSequenceUtil.isBlank(saveVO.getAppCode()) || CollUtil.isEmpty(saveVO.getPermissionList())) {
            return ApiResult.fail("权限信息为空");
        }

        var saveBO = this.vo2Bo(saveVO);
        permissionMngManager.removeRoleMenu(roleId, saveBO);

        return ApiResult.ok(roleId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> savePermissionMenuBatch(RoleAppPermissionBatchSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getRoleIds())) {
            return ApiResult.fail("角色为空");
        }
        if (CharSequenceUtil.isBlank(saveVO.getAppCode()) || CollUtil.isEmpty(saveVO.getPermissionList())) {
            return ApiResult.fail("权限信息为空");
        }

        var saveBO = this.vo2Bo(saveVO);
        permissionMngManager.saveRoleMenuBatch(saveBO);

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveApiDataPermission(SysDprSaveVO saveVO) {
        SysDprSaveBO saveBO = new SysDprSaveBO();
        saveBO.setRoleId(saveVO.getRoleId());
        saveBO.setMenuCode(saveVO.getMenuCode());
        saveBO.setApiCode(saveVO.getApiCode());
        saveBO.setRange(saveVO.getRange());

        // 自定义规则
        saveBO.setCustomRuleList(new ArrayList<>(16));
        if (CollUtil.isNotEmpty(saveVO.getCustomRuleList())) {
            SysDprApiCustomRuleSaveBO bo;
            for (SysDprApiCustomRuleSaveVO vo : saveVO.getCustomRuleList()) {
                bo = PermissionConverter.INSTANCE.convertCustomRuleSaveBO(vo);

                saveBO.getCustomRuleList().add(bo);
            }
        }

        // 字段权限
        saveBO.setFieldList(new ArrayList<>(16));
        if (CollUtil.isNotEmpty(saveVO.getFieldList())) {
            SysDprApiFieldSaveBO bo = null;
            for (SysDprApiFieldSaveVO vo : saveVO.getFieldList()) {
                bo = PermissionConverter.INSTANCE.convertFieldSaveBO(vo);

                saveBO.getFieldList().add(bo);
            }
        }

        // 保存
        permissionMngManager.saveDataPermission(saveBO);
        return ApiResult.ok(saveVO.getRoleId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveDataPermission(DataPermissionSaveVO saveVO) {
        permissionMngManager.saveDataPermission(saveVO);
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveApiDataPermission(SysDprBatchSaveVO saveVO) {
        SysDprBatchSaveBO saveBO = new SysDprBatchSaveBO();
        saveBO.setRoleId(saveVO.getRoleId());
        saveBO.setStrategy(this.convertSaveStrategy(saveVO.getSaveStrategy()));

        // 菜单API信息
        if (CollUtil.isEmpty(saveVO.getMenuApiList())) {
            return ApiResult.fail("请先选择API接口");
        }
        saveBO.setMenuApiList(saveVO.getMenuApiList().stream().map(t -> {
            SysDprBatchSaveBO.MenuApiSaveBO menuApi = new SysDprBatchSaveBO.MenuApiSaveBO();
            menuApi.setMenuCode(t.getMenuCode());
            menuApi.setApiCode(t.getApiCode());

            return menuApi;
        }).collect(Collectors.toList()));

        // 自定义规则
        saveBO.setCustomRuleList(new ArrayList<>(16));
        if (CollUtil.isNotEmpty(saveVO.getCustomRuleList())) {
            SysDprApiCustomRuleSaveBO bo;
            for (SysDprApiCustomRuleSaveVO vo : saveVO.getCustomRuleList()) {
                bo = PermissionConverter.INSTANCE.convertCustomRuleSaveBO(vo);

                saveBO.getCustomRuleList().add(bo);
            }
        }

        permissionMngManager.saveDataPermissionBatch(saveBO);
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveMenuApi(MenuApiSaveVO saveVO) {
        Assert.notNull(saveVO.getApiId(), "请选择接口");
        var apiCode = apiRepoProc.getCode(saveVO.getApiId());
        if (CharSequenceUtil.isBlank(apiCode)) {
            return ApiResult.fail("接口数据异常，接口编码为空");
        }

        // 查询应用信息
        Map<String, Long> appMap = appRepoProc.all().stream().collect(Collectors.toMap(SysPlatformAppDO::getAppCode, SysPlatformAppDO::getId, (t1, t2) -> t1));

        // 查询菜单信息
        Set<Long> menuIds = ObjectUtil.defaultIfNull(saveVO.getMenuIds(), Collections.emptySet());
        Map<Long, SysPlatformMenusDO> menuMap = menuIds.isEmpty() ? Collections.emptyMap() : menuRepoProc.get(menuIds).stream().collect(Collectors.toMap(SysPlatformMenusDO::getId, t -> t, (t1, t2) -> t1));

        // 查询已有配置
        var existsList = menusApiRepoProc.queryListByApi(saveVO.getApiId(), saveVO.getAppCode()).stream().collect(Collectors.toMap(SysPlatformMenusApiDO::getMenusId, t -> t, (t1, t2) -> t1));

        List<SysPlatformMenusApiDO> toAddList = new ArrayList<>();
        for (var entry : menuMap.entrySet()) {
            if (existsList.containsKey(entry.getKey())) {
                // 已存在
                continue;
            }
            SysPlatformMenusApiDO menusApiDO = new SysPlatformMenusApiDO();
            menusApiDO.setAppId(appMap.get(entry.getValue().getMenusAppCode()));
            menusApiDO.setAppCode(entry.getValue().getMenusAppCode());
            menusApiDO.setMenusId(entry.getValue().getId());
            menusApiDO.setMenusCode(entry.getValue().getMenusCode());
            menusApiDO.setApiId(saveVO.getApiId());
            menusApiDO.setApiCode(apiCode);

            toAddList.add(menusApiDO);
        }
        Set<Long> toDelIds = new HashSet<>();
        for (var entry : existsList.entrySet()) {
            if (menuMap.containsKey(entry.getKey())) {
                continue;
            }
            toDelIds.add(entry.getValue().getId());
        }

        // 删除去掉的
        if (!toDelIds.isEmpty()) {
            menusApiRepoProc.delete(toDelIds);
        }
        if (!toAddList.isEmpty()) {
            menusApiRepoProc.save(toAddList);
        }
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> clearPermission(Long roleId, String appCode) {
        if (roleId == null) {
            return ApiResult.fail("请选择角色");
        }

        permissionMngManager.clearRoleMenu(roleId, appCode);
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> copyPermission(RolePermissionCopySaveVO saveVO) {
        if (saveVO.getOriginalRoleId() == null) {
            return ApiResult.fail("请选择原角色");
        }
        if (CollUtil.isEmpty(saveVO.getTargetIds())) {
            return ApiResult.ok(true);
        }

        permissionMngManager.copyRoleMenu(saveVO.getOriginalRoleId(), saveVO.getTargetIds(), saveVO.getAppCode(), ObjectUtil.defaultIfNull(saveVO.getMerge(), true));
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> propagateDataPermission(RolePermissionPropagationSaveVO saveVO) {
        if (saveVO.getRoleId() == null) {
            return ApiResult.fail("请选择角色");
        }
        if (CharSequenceUtil.isBlank(saveVO.getBusinessObjectCode())) {
            return ApiResult.fail("请选择业务对象资源");
        }

        permissionMngManager.propagateDataPermission(saveVO.getRoleId(), saveVO.getBusinessObjectCode(), ObjectUtil.defaultIfNull(saveVO.getStrategy(), DprPropagateStrategyEnum.OVERWRITE));
        return ApiResult.ok(true);
    }

    private SysAppPermissionSaveBO vo2Bo(RoleAppPermissionSaveVO saveVO) {
        var permissionList = convertPermissionVo(saveVO.getPermissionList());
        return new SysAppPermissionSaveBO(saveVO.getAppCode(), permissionList);
    }

    private SysAppPermissionBatchSaveBO vo2Bo(RoleAppPermissionBatchSaveVO saveVO) {
        var permissionList = convertPermissionVo(saveVO.getPermissionList());
        return new SysAppPermissionBatchSaveBO(saveVO.getAppCode(), saveVO.getAdd() != null && saveVO.getAdd(), saveVO.getRoleIds(), permissionList);
    }

    private List<SysPermissionSaveBO> convertPermissionVo(List<RolePermissionSaveVO> permissionList) {
        if (CollUtil.isEmpty(permissionList)) {
            return Collections.emptyList();
        }

        return permissionList.stream()
                .filter(tt -> {
                    var nodeType = MenuTreeNodeType.valueOf(tt.getNodeType());
                    return nodeType == MenuTreeNodeType.APP || nodeType == MenuTreeNodeType.MENU || nodeType == MenuTreeNodeType.ACTION;
                }).map(tt -> new SysPermissionSaveBO(tt.getCode(), tt.getNodeType()))
                .collect(Collectors.toList());
    }

    private SysDprBatchSaveBO.Strategy convertSaveStrategy(String strategyName) {
        if (CharSequenceUtil.isBlank(strategyName)) {
            return SysDprBatchSaveBO.Strategy.APPEND;
        }

        var strategyMap = Arrays.stream(SysDprBatchSaveBO.Strategy.values())
                .collect(Collectors.toMap(SysDprBatchSaveBO.Strategy::name, t -> t, (t1, t2) -> t1));
        return strategyMap.getOrDefault(strategyName, SysDprBatchSaveBO.Strategy.APPEND);
    }
}
