package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import com.elitescloud.cloudt.system.convert.AreaConvert;
import com.elitescloud.cloudt.system.convert.UserConvert;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.model.bo.PermissionResBO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.SubUserDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.SubUserListRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserListRespVO;
import com.elitescloud.cloudt.system.model.vo.save.sys.SubUserBaseSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.sys.SubUserBatchSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.sys.SubUserSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.user.UserSaveVO;
import com.elitescloud.cloudt.system.service.SubUserMngService;
import com.elitescloud.cloudt.system.service.UserMngService;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.common.constant.PermissionOwnerTypeEnum;
import com.elitescloud.cloudt.system.service.common.constant.SubUserPermissionTypeEnum;
import com.elitescloud.cloudt.system.service.manager.PermissionMngManager;
import com.elitescloud.cloudt.system.service.manager.UserMngManager;
import com.elitescloud.cloudt.system.service.model.bo.AreaBO;
import com.elitescloud.cloudt.system.service.model.bo.SysUserSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysSubUserDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.google.common.base.Functions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/27
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class SubUserMngServiceImpl implements SubUserMngService {

    @Autowired
    private SubUserRepoProc repoProc;
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private UserTypeRepoProc userTypeRepoProc;
    @Autowired
    private UserRoleRepoProc userRoleRepoProc;
    @Autowired
    private PermissionResRepoProc permissionResRepoProc;

    @Autowired
    private PermissionMngManager permissionMngManager;
    @Autowired
    private UserMngManager userMngManager;
    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;
    @Autowired
    private TenantClientProvider tenantClientProvider;
    @Autowired
    private UserMngService userMngService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(SubUserSaveVO saveVO) {
        SysSubUserDO subUserDO = null;
        try {
            subUserDO = checkForSave(saveVO);
        } catch (Exception e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // 保存账号信息
        var tenant = tenantClientProvider.getSessionTenant();
        SysUserDO userDO = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            SysUserSaveBO userSaveBO = null;
            try {
                userSaveBO = this.convert2UserSaveBO(saveVO.getUserInfo(), tenant);
            } catch (Exception e) {
                throw new BusinessException("保存账号失败，" + e.getMessage());
            }
            return userMngManager.upsert(userSaveBO);
        });
        subUserDO.setSubUserId(userDO.getId());

        repoProc.save(subUserDO);

        // 保存权限信息
        this.saveUserPermission(Set.of(userDO.getId()), saveVO);

        return ApiResult.ok(subUserDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> saveBatch(SubUserBatchSaveVO saveVO) {
        // 检查账号信息
        var userIds = saveVO.getUserIds();
        Assert.notEmpty(userIds, "账号为空");
        var existsUserIds = repoProc.filterExistsSubUser(userIds);
        if (userIds.size() == existsUserIds.size()) {
            return ApiResult.fail("子账号已存在");
        }
        this.checkBaseSubUserSaveVO(saveVO);

        // 转为子账号
        List<SysSubUserDO> subUserList = new ArrayList<>();
        SysSubUserDO subUser = null;
        for (Long userId : userIds) {
            subUser = new SysSubUserDO();
            subUser.setSubUserId(userId);
            subUser.setType(saveVO.getBelongType().getValue());
            subUser.setTypeId(saveVO.getBelongerId());
            subUser.setPermissionType(saveVO.getGrantType().name());

            subUserList.add(subUser);
        }
        repoProc.save(subUserList);

        // 保存权限
        this.saveUserPermission(userIds, saveVO);

        var idList = subUserList.stream().map(SysSubUserDO::getId).collect(Collectors.toList());
        return ApiResult.ok(idList);
    }

    @Override
    public ApiResult<List<SubUserListRespVO>> querySubUserList(BelongType belongType, String belongerId, String roleCode) {
        var belonger = this.normalize(belongType, belongerId);
        var subUserListRespVOList = repoProc.querySubUsers(belonger, roleCode);
        if (subUserListRespVOList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 查询账号信息
        var userIds = subUserListRespVOList.stream().map(SubUserListRespVO::getSubUserId).collect(Collectors.toSet());
        var userMap = tenantDataIsolateProvider.byDefaultDirectly(() -> userRepoProc.queryUserList(userIds)).stream()
                .collect(Collectors.toMap(UserListRespVO::getId, Functions.identity(), (t1, t2) -> t1));

        subUserListRespVOList = subUserListRespVOList.stream()
                .filter(t -> userMap.containsKey(t.getSubUserId()))
                .peek(t -> {
                    var user = userMap.get(t.getSubUserId());
                    t.setUsername(user.getUsername());
                    t.setLastName(user.getLastName());
                    t.setFirstName(user.getFirstName());
                    t.setMobile(user.getMobile());
                    t.setEmail(user.getEmail());
                }).collect(Collectors.toList());
        return ApiResult.ok(subUserListRespVOList);
    }

    @Override
    public ApiResult<SubUserDetailRespVO> get(Long id) {
        Assert.notNull(id, "记录ID为空");

        var subUserDO = repoProc.get(id);
        if (subUserDO == null) {
            return ApiResult.fail("子账号记录不存在");
        }
        var userInfo = userMngService.get(subUserDO.getSubUserId()).getData();
        if (userInfo == null) {
            return ApiResult.fail("登录账号已不存在");
        }

        SubUserDetailRespVO respVO = new SubUserDetailRespVO();
        respVO.setUser(userInfo);
        var permissionType = SubUserPermissionTypeEnum.valueOf(subUserDO.getPermissionType());
        var belonger = new BelongType.Belonger(BelongType.valueOf(subUserDO.getType()), subUserDO.getTypeId());
        switch (permissionType) {
            case GRANT:
                var resIds = permissionResRepoProc.listPermissionOfOwner(belonger, PermissionOwnerTypeEnum.USER, subUserDO.getSubUserId().toString()).stream()
                        .map(PermissionResBO.Res::getResId)
                        .collect(Collectors.toSet());
                respVO.setMenuCodes(resIds);
                break;
            case GRANT_BY_ROLE:
                var roleList = userRoleRepoProc.getRolesOfUser(subUserDO.getSubUserId(), belonger);
                respVO.setRoleList(roleList);
                break;
            case EXTENDS_ALL:
                break;
            default:
                return ApiResult.fail("暂不支持的授权类型");
        }

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> getRolesOfSubUser(Long id) {
        Assert.notNull(id, "记录ID为空");

        var subUserDO = repoProc.get(id);
        if (subUserDO == null) {
            return ApiResult.fail("子账号记录不存在");
        }
        var permissionType = SubUserPermissionTypeEnum.valueOf(subUserDO.getPermissionType());
        if (permissionType != SubUserPermissionTypeEnum.GRANT_BY_ROLE) {
            return ApiResult.ok(Collections.emptyList());
        }

        var belonger = new BelongType.Belonger(BelongType.valueOf(subUserDO.getType()), subUserDO.getTypeId());
        var roleList = userRoleRepoProc.getRolesOfUser(subUserDO.getSubUserId(), belonger);
        return ApiResult.ok(roleList);
    }

    @Override
    public ApiResult<Set<String>> getMenuCodesOfSubUser(Long id) {
        Assert.notNull(id, "记录ID为空");

        var subUserDO = repoProc.get(id);
        if (subUserDO == null) {
            return ApiResult.fail("子账号记录不存在");
        }
        var permissionType = SubUserPermissionTypeEnum.valueOf(subUserDO.getPermissionType());
        if (permissionType != SubUserPermissionTypeEnum.GRANT) {
            return ApiResult.ok(Collections.emptySet());
        }

        var belonger = new BelongType.Belonger(BelongType.valueOf(subUserDO.getType()), subUserDO.getTypeId());
        var menuCodes = permissionResRepoProc.listPermissionOfOwner(belonger, PermissionOwnerTypeEnum.USER, subUserDO.getSubUserId().toString()).stream()
                .map(PermissionResBO.Res::getResId)
                .collect(Collectors.toSet());
        return ApiResult.ok(menuCodes);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<Long>> del(Set<Long> ids, Boolean delUser) {
        Assert.notEmpty(ids, "记录ID为空");

        var subUserDOList = repoProc.get(ids);
        if (subUserDOList.isEmpty()) {
            return ApiResult.ok(ids);
        }

        // 删除权限资源
        for (SysSubUserDO subUserDO : subUserDOList) {
            var permissionType = SubUserPermissionTypeEnum.valueOf(subUserDO.getPermissionType());
            var belonger = new BelongType.Belonger(new BelongType(subUserDO.getType()), subUserDO.getTypeId());
            switch (permissionType) {
                case GRANT:
                    permissionResRepoProc.delete(belonger, PermissionOwnerTypeEnum.USER, subUserDO.getSubUserId().toString());
                    break;
                case GRANT_BY_ROLE:
                    userRoleRepoProc.deleteRolesOfUser(subUserDO.getSubUserId(), belonger);
                    break;
                case EXTENDS_ALL:
                    break;
                default:
                    return ApiResult.fail("暂不支持的授权类型");
            }
        }

        if (delUser == null || !delUser) {
            // 不需要删除账号
            repoProc.delete(ids);
            return ApiResult.ok(ids);
        }

        // 删除账号信息
        var userIds = repoProc.querySubUserIds(ids);
        if (userIds.isEmpty()) {
            return ApiResult.ok(ids);
        }
        for (Long userId : userIds) {
            userMngManager.delete(userId);
        }

        repoProc.delete(ids);
        return ApiResult.ok(ids);
    }

    private void saveUserPermission(Set<Long> userIds, SubUserBaseSaveVO saveVO) {
        var grantType = saveVO.getGrantType();
        switch (grantType) {
            case EXTENDS_ALL:
                // 继承所有，无需处理
                return;
            case GRANT_BY_ROLE:
                if (CollUtil.isEmpty(saveVO.getRoleCodes())) {
                    return;
                }
                permissionMngManager.addUserRole(userIds, saveVO.getRoleCodes());
                break;
            case GRANT:
                List<PermissionResBO.Res> permissionResList = CollUtil.isEmpty(saveVO.getMenuCodes()) ? Collections.emptyList() :
                        saveVO.getMenuCodes().stream().filter(StringUtils::hasText).map(t -> new PermissionResBO.Res(MenuTreeNodeType.MENU, t)).collect(Collectors.toList());
                var permissionList = userIds.stream().filter(ObjUtil::isNotNull).map(t -> {
                    PermissionResBO permissionResBO = new PermissionResBO();
                    permissionResBO.setOwnerTypeEnum(PermissionOwnerTypeEnum.USER);
                    permissionResBO.setOwnerId(t.toString());
                    permissionResBO.setResList(permissionResList);
                    return permissionResBO;
                }).collect(Collectors.toList());
                permissionMngManager.savePermissionRes(permissionList, new BelongType.Belonger(saveVO.getBelongType(), saveVO.getBelongerId()));
                break;
            default:
                throw new IllegalStateException("暂不支持的授权类型");
        }
    }

    private SysUserSaveBO convert2UserSaveBO(UserSaveVO userSaveVO, SysTenantDTO tenantDTO) {
        Long userId = userSaveVO.getId();

        // 转为BO
        SysUserSaveBO userSaveBO = null;
        if (userId == null) {
            userSaveBO = UserConvert.INSTANCE.vo2SaveBo(userSaveVO);
        } else {
            // 先查询出原有信息
            var userDO = userRepoProc.get(userId);
            Assert.notNull(userDO, "账号不存在");
            userSaveBO = UserConvert.INSTANCE.do2SaveBo(userDO);
            UserConvert.INSTANCE.copy2SaveBo(userSaveVO, userSaveBO);
        }

        // 行政区域
        AreaBO areaBO = AreaConvert.INSTANCE.vo2Bo(userSaveVO.getAreaVO());
        userSaveBO.setAreaBO(areaBO);

        // 默认值处理
        if (!StringUtils.hasText(userSaveBO.getSourceType())) {
            userSaveBO.setSourceType(UserSourceType.CREATE.getValue());
        }
        if (CollectionUtils.isEmpty(userSaveBO.getTerminals())) {
            userSaveBO.setTerminalsAdd(Set.of(Terminal.BACKEND.name()));
        }

        // 账号类型
        if (userId != null) {
            var userTypeList = userTypeRepoProc.getUserTypeBO(userId, tenantDTO.getId());
            userSaveBO.setUserTypes(new HashSet<>(userTypeList));
        }

        return userSaveBO;
    }

    private SysSubUserDO checkForSave(SubUserSaveVO saveVO) {
        SysSubUserDO subUserDO = saveVO.getId() == null ? new SysSubUserDO() : repoProc.get(saveVO.getId());
        Assert.notNull(subUserDO, "修改的数据不存在");

        var userInfo = saveVO.getUserInfo();
        Assert.notNull(saveVO.getUserInfo(), "账号信息为空");
        if (subUserDO.getId() != null) {
            Assert.isTrue(subUserDO.getSubUserId().equals(userInfo.getId()), "选择的账号不可变更");
        } else {
            if (userInfo.getId() != null) {
                Assert.isTrue(!repoProc.existsSubUser(userInfo.getId()), "该账号已是子账号");
            }
        }

        this.checkBaseSubUserSaveVO(saveVO);

        subUserDO.setPermissionType(saveVO.getGrantType().name());

        subUserDO.setType(saveVO.getBelongType().getValue());
        subUserDO.setTypeId(saveVO.getBelongerId());

        return subUserDO;
    }

    private void checkBaseSubUserSaveVO(SubUserBaseSaveVO saveVO) {
        if (saveVO.getGrantType() == null) {
            saveVO.setGrantType(SubUserPermissionTypeEnum.EXTENDS_ALL);
        }

        var belonger = this.normalize(saveVO.getBelongType(), saveVO.getBelongerId());
        saveVO.setBelongType(belonger.getBelongType());
        saveVO.setBelongerId(belonger.getBelongId());
    }

    private BelongType.Belonger normalize(BelongType belongType, String belongerId) {
        if (belongType == null) {
            return BelongType.getBelongerPersonal();
        }

        return new BelongType.Belonger(belongType, BelongType.getBelongerId(belongType, belongerId));
    }
}
